// ##################################################
// ###              TOBII_FRAME_HPP               ###
// ##################################################
#ifndef TOBII_FRAME_HPP
#define TOBII_FRAME_HPP

/**
 * \file Frame.hpp
 * \brief Describes a video frame sent by the tobii glasses or read in a video file.
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STADARD LIBRARIES
#include <memory>

// INTERNAL MODULES
#include "PrimitifTypes.hpp"
#include "Record.hpp"

// EXTERNAL LIBRARIES
#include <opencv2/core/core.hpp>

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###                 CLASS FRAME                ###
	// ##################################################
	
	/**
	 * \class Frame
	 * \brief cv::Mat video Record.
	 *
	 * This class describes a video frame sent by the tobii glasses or read in a video file.
	 *
	 **/
	class Frame : public cv::Mat, public Record
	{
	public:
		/**
		 * \brief Constructor (Default)
		 * \param timeStamp (Int64) time stamp of the frame
		 * \param status (UInt8) status of the frame, 0 = OK, else not ok.
		 * \param frame (cv::Mat) the decoded image data in Blue-Green-Red OpenCV color space
		 *
		 * Initialize a new Frame instance with the given parameters
		 *
		 **/
		Frame ( Int64 timeStamp = 0, UInt8 status = 0, const cv::Mat& frame = cv::Mat() );

		/**
		 * \brief Constructor (Copy)
		 * \param frame (const Frame&) frame to copy.
		 *
		 * Copy the given frame in a newly alocated Frame.
		 * Does not copy image data but reference them, see the OpenCV documentation.
		 *
		 **/
		Frame ( const Frame& frame );

		/**
		 * \brief Cast Operator (bool)
		 * \returns (bool) true is the frame contains data, false if not.
		 * \bug does not work
		 *
		 * Use this to check the validity of this Frame.
		 *
		 **/
		operator bool () const;

		/**
		 * \brief Assignement Operator (const Frame&)
		 * \param frame (const Frame&) The Frame to Copy
		 * \returns This Frame once set.
		 *
		 * Copy the given Frame into this Frame.
		 * Does not copy image data but reference them, see the OpenCV documentation.
		 *
		 **/
		Frame& operator= ( const Frame& frame );

		/**
		 * \brief Assignement Operator (const cv::Mat&)
		 * \param frame (const cv::Mat&) The Frame to Copy
		 * \returns This Frame once set.
		 *
		 * Copy the given Frame into this Frame.
		 * Does not copy image data but reference them, and does not set the time stamp and status, see the OpenCV documentation.
		 *
		 **/
		Frame& operator= ( const cv::Mat& frame );
	
	}; // END CLASS FRAME
	// ##################################################

	// ##################################################
	// ###      SHARED POINTER TYPE DEFINITION        ###
	// ##################################################
	
	/**
	 * \brief Shared pointer to a Frame record.
	 *
	 * Defines the type of a shared pointer to a Frame record.
	 *
	 **/
	using FramePtr = std::shared_ptr< Frame >;

	// END SHARED POINTER TYPE DEFINITION
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_FRAME_HPP
// ##################################################