// ##################################################
// ###          TOBII_ACCELERATION_HPP            ###
// ##################################################
#ifndef TOBII_ACCELERATION_HPP
#define TOBII_ACCELERATION_HPP

/**
 * \file Acceleration.hpp
 * \brief Describes Acceleration object data type, modeling Tobii Glasses 2 acceleration measurements.
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// INTERNAL MODULES
#include "Record.hpp"
#include "Vector3.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
namespace Tobii {

	// ##################################################
	// ###              CLASS ACCELERATION            ###
	// ##################################################
	
	/**
	 * \class Acceleration
	 * \brief Describes Tobii Glasses 2 accelerometer measurements.
	 *
	 * This object represents measurements comming from the Tobii Glasses 2 accelerometer.
	 * The acceleration is modeled using a 3D vector which values are in meter per second squared ( m/s^2 ),
	 * as specified in the Tobii SDK developer guide 1.0 in section D6.1.7.
	 * 
	 **/
	class Acceleration : public Record, public Vector3
	{
	public:
		/**
		 * \brief Constructor (Default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at whiwh the recording was performed by the accelerometer.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param acc (const Vector3&) The 3D acceleration vector in meter per second squared.
		 *
		 * Creates and initialize an object of type Acceleration.
		 *
		 **/
		Acceleration ( Int64 timeStamp = 0, UInt8 status = 0, const Vector3& acc = Vector3::zero() );

		/**
		 * \brief Constructor (Copy)
		 * \param acc (const Acceleration&) The Acceleration record to copy.
		 *
		 * Creates and initialize a copy of an object of type Acceleration.
		 *
		 **/
		Acceleration ( const Acceleration& acc );

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of 'this' Acceleration
		 *
		 * Creates and return a std::string representation of 'this' Acceleration record.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"ac":[dx,dy,dz]}
		 *
		 **/
		operator std::string () const;

		/**
		 * \brief Assignment operator (const Vector3&)
		 * \param acc (const Vector3&) the acceleration value.
		 * \return (Acceleration&) Reference to 'this' Acceleration once the acceleration value as been set.
		 *
		 * Sets 'this' Acceleration record value using a given Vector3.
		 *
		 **/
		Acceleration& operator= ( const Vector3& acc );

		/**
		 * \brief Assignment operator (const Acceleration&)
		 * \param acc (const Acceleration&) the acceleration record to copy.
		 * \return (Acceleration&) Reference to 'this' Acceleration once set.
		 *
		 * Sets 'this' Acceleration record time stamp, status and value using a given Acceleration record.
		 *
		 **/
		Acceleration& operator= ( const Acceleration& acc );
	
	}; // END CLASS ACCELERATION
	// ##################################################

	// ##################################################
	// ###      SHARED POINTER TYPE DEFINITION        ###
	// ##################################################

	/**
	 * \brief Shared pointer to an Acceleration record.
	 *
	 * Defines the type of a shared pointer to an Acceleration record.
	 *
	 **/
	using AccelerationPtr = std::shared_ptr< Acceleration >;

	// END SHARED POINTER TYPE DEFINITION
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	/**
	 * \brief Right-hand side output stream concatenation operator.
	 * \param stream (std::ostream&) stream where to concatenate the std::string representation of the Acceleration record.
	 * \param acc (const Acceleration&) the Acceleration record to represent
	 * \return the output stream once the Acceleration record representation has been concatenated to it.
	 *
	 * Concatenate the std::string representation of the Acceleration record in the output stream.
	 *
	 **/
	std::ostream& operator<< ( std::ostream& stream, const Acceleration& acc );

	/**
	 * \brief Left-hand side output stream concatenation operator.
	 * \param acc (const Acceleration&) the Acceleration record to represent.
	 * \param stream (std::ostream&) stream where to concatenate the std::string representation of the Acceleration record.
	 * \return the output stream once the Acceleration record representation has been concatenated to it.
	 *
	 * Concatenate the std::string representation of the Acceleration record in the output stream.
	 *
	 **/
	std::ostream& operator<< ( const Acceleration& acc, std::ostream& stream );

	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_ACCELERATION_HPP
// ##################################################