// ##################################################
// ###            TOBII_APPLICATION_CPP           ###
// ##################################################
#ifndef TOBII_APPLICATION_CPP
#define TOBII_APPLICATION_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "Application.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###             CLASS APPLICATION              ###
	// ##################################################
	
	Application::Application ( const int argc, const char* argv[] )
	: _name( argv[0] ), _display( false ), _record( false ), _dump( false )
	{}

	Application::~Application () {}

	const QString&			Application::name		() const	{ return this->_name;		}
	QString&				Application::name		()			{ return this->_name;		}
	const QString&			Application::recordFile	() const	{ return this->_recordFile;	}
	QString&				Application::recordFile	()			{ return this->_recordFile;	}
	const QString&			Application::dumpFolder	() const	{ return this->_dumpFolder;	}
	QString&				Application::dumpFolder	()			{ return this->_dumpFolder;	}
	bool					Application::display	() const	{ return this->_display;	}
	bool&					Application::display	()			{ return this->_display;	}
	bool					Application::record		() const	{ return this->_record;		}
	bool&					Application::record		()			{ return this->_record;		}
	bool					Application::dump		() const	{ return this->_dump;		}
	bool&					Application::dump		()			{ return this->_dump;		}
	const cv::VideoWriter&	Application::recorder	() const	{ return this->_recorder;	}
	cv::VideoWriter&		Application::recorder	()			{ return this->_recorder;	}

	void Application::parseOptions ( int argc, const char* argv[], int minNbArgs )
	{
		for ( int index(0) ; index < argc ; ++index )
		{
			QString arg( argv[index] );

			if ( arg == "-h" || arg == "-help" )
			{
				std::cout << this->help().toStdString() << std::endl;
				exit( EXIT_SUCCESS );
			}

			if ( arg == "-d" || arg == "-display" )
			{
				this->display() = true;
			}

			if ( arg == "-r" || arg == "-record" )
			{
				this->record() = true;
				this->recordFile() = argv[++index];
			}

			if ( arg == "-D" || arg == "-dump" )
			{
				this->dump() = true;
				this->dumpFolder() = argv[++index];
				
				QDir dir( this->dumpFolder() );
				if ( !dir.exists() )
					dir.mkpath( "." );
			}

			if ( arg == "-R" || arg == "-resolution" )
			{
				this->resolution().width  = atoi( argv[++index] );
				this->resolution().height = atoi( argv[++index] );
			}

			if ( arg == "-gbs" || arg == "-gaze-buffer-size" )
			{
				QString strval( argv[++index] );
				if ( strval == "max" || strval == "inf" )
					this->setGazeBufferMaxSize( std::numeric_limits<Int64>::max() );
				else
					this->setGazeBufferMaxSize( strval.toInt() );
			}	

			if ( arg == "-fbs" || arg == "-frame-buffer-size" )
			{
				QString strval( argv[++index] );
				if ( strval == "max" || strval == "inf" )
					this->setFrameBufferMaxSize( std::numeric_limits<Int64>::max() );
				else
					this->setFrameBufferMaxSize( strval.toInt() );
			}

			if ( arg == "-abs" || arg == "-acceleration-buffer-size" )
			{
				QString strval( argv[++index] );
				if ( strval == "max" || strval == "inf" )
					this->setAccelerationBufferMaxSize( std::numeric_limits<Int64>::max() );
				else
					this->setAccelerationBufferMaxSize( strval.toInt() );
			}

			if ( arg == "-obs" || arg == "-orientation-buffer-size" )
			{
				QString strval( argv[++index] );
				if ( strval == "max" || strval == "inf" )
					this->setOrientationBufferMaxSize( std::numeric_limits<Int64>::max() );
				else
					this->setOrientationBufferMaxSize( strval.toInt() );
			}

			if ( arg == "-k" || arg == "-keep-all-data" )
			{
				this->setKeepAllData( true );
			}

			if ( arg == "-fi" || arg == "-force-insterpolation" )
			{
				this->setForceInterpolation( true );
			}

			if ( arg == "-fd" || arg == "-force-distance-calculus" )
			{
				this->setForceComputingDistance( true );
			}
		}

		// Check the number of mandatory arguments
		if ( argc <= minNbArgs )
		{
			std::cerr << this->usage().toStdString() << std::endl;
			exit( EXIT_FAILURE );
		}

		// If record, create the ouput video
		if ( this->record() )
			this->recorder().open( this->recordFile().toStdString(), CV_FOURCC('M','P','E','G'), 25, this->resolution(), true );
	}

	QString Application::usage () const
	{
		return QString("USAGE : ") + this->name() + " <optionals...>, use -help for more informations.";
	}
	
	QString Application::help () const
	{
		std::stringstream helpstr;
		helpstr << this->usage().toStdString() << std::endl
		        << "Possible optional arguments are:" << std::endl
		        << "\tuse -h   or -help                                          to display this usefull message."                                    << std::endl
		        << "\tuse -d   or -display                                       to display the process with cv::imshow."                             << std::endl
		        << "\tuse -r   or -record                 + <output-file-name>   to record the process in a video using a cv::VideoWriter."           << std::endl
		        << "\tuse -D   or -dump                   + <output-folder-name> to save each frame of the porcess with imwrite."                     << std::endl
		        << "\tuse -R   or -resolution             + <width> <height>     to specifiy the working resolution."                                 << std::endl
		        << "\tuse -gbs or -gazebuffersize         + <size>               to specifiy the maximum size of the gaze recordings buffer."         << std::endl
		        << "\tuse -fbs or -framebuffersize        + <size>               to specifiy the maximum size of the frame recordings buffer."        << std::endl
		        << "\tuse -abs or -accelerationbuffersize + <size>               to specifiy the maximum size of the acceleration recordings buffer." << std::endl
		        << "\tuse -obs or -orientationbuffersize  + <size>               to specifiy the maximum size of the orientation recordings buffer."  << std::endl
		        << "\tuse -k   or -keep-all-data                                 to keep all gaze data, even the ones with error codes."              << std::endl
		        << "\tuse -fi  or -force-insterpolation                          to force interpolation of gaze, even if a record exists."            << std::endl
		        << "\tuse -fd  or -force-distance-calculus                       to recompute the distance of the fxation point."                     << std::endl;
		return QString::fromStdString( helpstr.str() );
	}

	void Application::displayFrame ( const cv::Mat& frame )
	{
		if ( this->display() )
		{
			cv::imshow( this->name().toStdString(), frame );
			cv::waitKey(1);
		}
	}

	void Application::dumpFrame ( const Frame& frame, const QString& frameName )
	{
		this->dumpFrame( frame, frame.timeStamp(), frameName );
	}

	void Application::dumpFrame ( const cv::Mat& frame, Int64 timeStamp, const QString& frameName )
	{
		if ( this->dump() )
		{
			QString filename( "%1/%2_%3ms.png" );
			filename = filename.arg( this->dumpFolder() ).arg( frameName ).arg( this->ptrFrame()->timeStamp() );

			cv::imwrite(
				filename.toStdString(),
				frame,
				std::vector< int >( {CV_IMWRITE_PNG_COMPRESSION, 0} )
			);
		}
	}

	void Application::recordFrame ( const cv::Mat& frame )
	{
		if ( this->record() )
			this->recorder() << frame;
	}

	// END CLASS APPLICATION
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_APPLICATION_CPP
// ##################################################