// ##################################################
// ###             TOBII_PROJECTPAGE_CPP          ###
// ##################################################
#ifndef TOBII_PROJECTPAGE_CPP
#define TOBII_PROJECTPAGE_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "ProjectPage.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

namespace Tobii {

	// ##################################################
	// ###            CLASS PROJECT-PAGE              ###
	// ##################################################

	ProjectPage::ProjectPage ( const QString& name, const QString& categoryFile, QWidget* ptrParent )
	: QDockWidget      ( name, ptrParent )
	, _ptrContentWidget( new QWidget      (this)                      )
	, _ptrLayout       ( new QGridLayout  (this->ptrContentWidget() ) )
	, _ptrControlePanel( new ControlePanel(this->ptrContentWidget() ) )
	, _ptrVideoDisplay ( new VideoDisplay (this->ptrContentWidget() ) )
	, _scalingFactor( 1.0 )
	{
		this->setAllowedAreas( Qt::TopDockWidgetArea );
		this->ptrContentWidget()->setLayout( this->ptrLayout() );
		this->setWidget( this->ptrContentWidget() );

		// Controle Panel:
		this->ptrControlePanel()->setCategories( categoryFile );
		connect( this->ptrControlePanel(), SIGNAL(     videoNextFrame()               ), this, SLOT(         nextFrame()               ) );
		connect( this->ptrControlePanel(), SIGNAL( videoPreviousFrame()               ), this, SLOT(     previousFrame()               ) );
		connect( this->ptrControlePanel(), SIGNAL(       videoSetView(const QString&) ), this, SLOT(   setSelectedView(const QString&) ) );
		connect( this->ptrControlePanel(), SIGNAL( videoSetZoomFactor(double)         ), this, SLOT(     setZoomFactor(double)         ) );
		connect( this->ptrControlePanel(), SIGNAL(    frameFromPicked()               ), this, SLOT(     pickFrameFrom()               ) );
		connect( this->ptrControlePanel(), SIGNAL(      frameToPicked()               ), this, SLOT(       pickFrameTo()               ) );
		connect( this->ptrControlePanel(), SIGNAL(    threshodChanged(double)         ), this, SLOT(      setThreshold(double)         ) );
		connect( this->ptrControlePanel(), SIGNAL(  computeAnnotation()               ), this, SLOT( computeAnnotation()               ) );
		connect( this->ptrControlePanel(), SIGNAL(    categoryChanged(const QString&) ), this, SLOT(    changeCategory(const QString&) ) );
	#ifdef  TRACK_STATS
		connect( this->ptrControlePanel(), SIGNAL(    resetStatistics()               ), this, SLOT(   resetStatistics()               ) );
		connect( this->ptrControlePanel(), SIGNAL(     saveStatistics(int)            ), this, SLOT(    saveStatistics(int)            ) );
		this->nbInstances() = 0;
	#endif

		this->ptrLayout()->addWidget( this->ptrControlePanel(), 0, 0, 1, 1, static_cast<Qt::Alignment>(Qt::AlignTop|Qt::AlignLeft)     );
		this->ptrLayout()->addWidget( this->ptrVideoDisplay(),  0, 1, 1, 1, static_cast<Qt::Alignment>(Qt::AlignVCenter|Qt::AlignLeft) );
		this->ptrLayout()->setContentsMargins(2,2,2,2);
		this->ptrLayout()->setSizeConstraint(QLayout::SetFixedSize);

		if ( ! name.isEmpty() )
			this->open(name);
	}

	// --------------------------------------------------
	// GETTERS - SETTERS

	const QWidget*			ProjectPage::ptrContentWidget	() const	{ return this->_ptrContentWidget;	}
	QWidget*				ProjectPage::ptrContentWidget	()			{ return this->_ptrContentWidget;	}
	const QGridLayout*		ProjectPage::ptrLayout			() const	{ return this->_ptrLayout;			}
	QGridLayout*			ProjectPage::ptrLayout			()			{ return this->_ptrLayout;			}
	const ControlePanel*	ProjectPage::ptrControlePanel	() const	{ return this->_ptrControlePanel;	}
	ControlePanel*&			ProjectPage::ptrControlePanel	()			{ return this->_ptrControlePanel;	}
	const VideoDisplay*		ProjectPage::ptrVideoDisplay	() const	{ return this->_ptrVideoDisplay;	}
	VideoDisplay*&			ProjectPage::ptrVideoDisplay	()			{ return this->_ptrVideoDisplay;	}

	const ProjectPtr&		ProjectPage::ptrProject			() const	{ return this->_ptrProject;			}
	ProjectPtr&				ProjectPage::ptrProject			()			{ return this->_ptrProject;			}
	const QString&			ProjectPage::selectedView		() const	{ return this->_selectedView;		}
	QString&				ProjectPage::selectedView		()			{ return this->_selectedView;		}
	double					ProjectPage::scalingFactor		() const	{ return this->_scalingFactor;		}
	double&					ProjectPage::scalingFactor		()			{ return this->_scalingFactor;		}
	Int64					ProjectPage::timeStampFrom		() const	{ return this->_timeStampFrom;		}
	Int64&					ProjectPage::timeStampFrom		()			{ return this->_timeStampFrom;		}
	Int64					ProjectPage::timeStampTo		() const	{ return this->_timeStampTo;		}
	Int64&					ProjectPage::timeStampTo		()			{ return this->_timeStampTo;		}
	double					ProjectPage::threshold			() const	{ return this->_threshold;			}
	double&					ProjectPage::threshold			()			{ return this->_threshold;			}

	// --------------------------------------------------
	// PUBLIC SLOTS

	void ProjectPage::setProject ( ProjectPtr ptrProject )
	{
		// Open the project
		this->ptrProject() = ptrProject;
		
		// set this widget displayed datas
		this->setWindowTitle( this->ptrProject()->name() );
		this->timeStampFrom() = this->ptrProject()->firstTimeStamp();
		this->timeStampTo()   = this->ptrProject()->lastTimeStamp();

		// set the controle pannel widgets displayed datas
		this->ptrControlePanel()->setProjectData( this->ptrProject() );

		// Update the rest (frame included)
		this->update();
	}

	void ProjectPage::setFrame ( const Frame& frame )
	{
		this->ptrVideoDisplay()->setFrame( frame, this->scalingFactor() );
		this->ptrVideoDisplay()->adjustSize();

		this->ptrControlePanel()->setFrameData(
			this->ptrProject()->currentIndex(),
			this->ptrProject()->frameCount(),
			this->ptrProject()->currentTimeStamp()
		);
		this->ptrControlePanel()->adjustSize();

		this->adjustSize();
		emit this->sizeChanged( this->size() );
	}

	void ProjectPage::closeEvent ( QCloseEvent* event )
	{
		this->close();
	}

	void ProjectPage::open ( const QString& dir )
	{
		// Open the project
		this->setProject( std::make_shared< Project >(dir) );
	}

	void ProjectPage::close ()
	{
		if ( this->ptrProject() )
			this->ptrProject()->close();
		this->QDockWidget::close();
	}

	void ProjectPage::update ()
	{
		Frame videoFrame      = this->ptrProject()->getFrameOf( Project::Video      );
		Frame saliencyFrame   = this->ptrProject()->getFrameOf( Project::Saliency   );
		Frame annotationFrame = this->ptrProject()->getFrameOf( Project::Annotation );

		if ( this->selectedView() == Project::Video )
			this->setFrame( videoFrame );

		else if ( this->selectedView() == Project::Saliency )
			this->setFrame( saliencyFrame );

		else if ( this->selectedView() == Project::SaliencyHeatMap )
			this->setFrame( Saliency::computeHeatMap(videoFrame, saliencyFrame) );

		else if ( this->selectedView() == Project::SaliencyWeightedMap )
			this->setFrame( Saliency::computeWeightedMap(videoFrame, saliencyFrame) );

		else if ( this->selectedView() == Project::Annotation )
			this->setFrame( annotationFrame );

		else if ( this->selectedView() == Project::AnnotationHeatMap )
			this->setFrame( Saliency::computeHeatMap(videoFrame, annotationFrame) );

		else if ( this->selectedView() == Project::AnnotationWeightedMap )
			this->setFrame( Saliency::computeWeightedMap(videoFrame, annotationFrame) );

		this->ptrVideoDisplay()->adjustSize();
		this->adjustSize();
		emit this->sizeChanged( this->size() );
	}

	// --------------------------------------------------
	// PRIVATE SLOTS

	void ProjectPage::nextFrame ()
	{
		this->ptrProject()->next();
		this->update();
	}

	void ProjectPage::previousFrame ()
	{
		this->ptrProject()->previous();
		this->update();
	}

	void ProjectPage::setSelectedView ( QString view )
	{
		this->selectedView() = view;
		this->update();
	}

	void ProjectPage::setZoomFactor ( double factor )
	{
		this->scalingFactor() = factor;
		this->update();
	}
	
	void ProjectPage::pickFrameFrom ()
	{
		this->timeStampFrom() = this->ptrProject()->currentTimeStamp();
		this->ptrControlePanel()->setFrameFrom( this->timeStampFrom() );
	}

	void ProjectPage::pickFrameTo ()
	{
		this->timeStampTo() = this->ptrProject()->currentTimeStamp();
		this->ptrControlePanel()->setFrameTo( this->timeStampTo() );
	}

	void ProjectPage::setThreshold ( double value )
	{
		this->threshold() = floor(value * 255);
		
		// Read the saliency map
		Frame saliencyFrame = this->ptrProject()->getFrameOf( Project::Saliency );

		// Threshold the saliency map
		saliencyFrame = Saliency::threshold( saliencyFrame, this->threshold() );

		// Write the annotztion map
		this->ptrProject()->setFrameOf( Project::Annotation, saliencyFrame );

		this->update();
	}

	void ProjectPage::computeAnnotation ()
	{
		Frame allzeros( 0, 0, cv::Mat(this->ptrProject()->resolution(), CV_8UC3, cv::Scalar(0,0,0)) );

	#ifdef TRACK_STATS
		this->nbInstances() = 0;
	#endif

		for ( Int64 timeStamp = this->ptrProject()->firstTimeStamp() ; timeStamp <= this->ptrProject()->lastTimeStamp() ; timeStamp += 40 )
		{
			if ( this->timeStampFrom() <= timeStamp && timeStamp <= this->timeStampTo() )
			{
				// Read the saliency map
				Frame saliencyFrame = this->ptrProject()->getFrameOf( Project::Saliency, timeStamp );
				
				// Threshold the saliency map
				saliencyFrame = Saliency::threshold( saliencyFrame, this->threshold() );
				
				// Write the annotztion map
				this->ptrProject()->setFrameOf( Project::Annotation, saliencyFrame, timeStamp );

				#ifdef TRACK_STATS
					++this->nbInstances();
				#endif
			}
			else
				this->ptrProject()->setFrameOf( Project::Annotation, allzeros, timeStamp );
		}

		this->update();
	}

	void ProjectPage::changeCategory ( const QString& text )
	{
		QSettings settings;
		this->ptrProject()->move( settings.value("workspace").toString()+"/"+text );
	}

#ifdef TRACK_STATS
	Int64	ProjectPage::nbInstances	() const { return this->_nbInstances; }
	Int64&	ProjectPage::nbInstances	()		 { return this->_nbInstances; }

	void ProjectPage::resetStatistics ()
	{
		this->nbInstances() = 0;
	}

	void ProjectPage::saveStatistics ( int timeElapsed )
	{
		if ( this->ptrProject() && this->ptrProject()->isOpen() )
		{
			QSettings	settings;
			QFile		statFile( settings.value("workspace").toString()+"/Statistics.txt" );
						statFile.open( QIODevice::Append|QIODevice::Text );
			QTextStream	txtStream( &statFile );
			txtStream
				<< this->ptrProject()->name()			<< "\t"
				<< this->ptrProject()->duration()		<< "\t"
				<< this->ptrProject()->frameCount()		<< "\t"
				<< this->ptrControlePanel()->gategory()	<< "\t"
				<< this->nbInstances()					<< "\t"
				<< this->nbInstances()*40				<< "\t"
				<< timeElapsed							<< "\n";
		}
	}
#endif

	// END CLASS PROJECT-PAGE
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_PROJECTPAGE_CPP
// ##################################################