/******************************************************
 * This is a port in C++11 of the Tobii Glasses 2 SDK *
 * This will ensure an easy access to the Tobii Glass *
 * Author: Pierre-Marie Plans                         *
 * E-mail: pierre.plans@gmail.com                     *
 ******************************************************/

#ifndef TOBIISDK_H_
#define TOBIISDK_H_

/**
 * \file TobiiGlassesSDK.h
 * \brief This class handles the Tobii Glasses 2 communications.
 * \author Pierre-Marie Plans
 * \date 12/08/2015
 **/

#define TOBII_LIVE_ON_WINDOW 0x01
#define TOBII_LIVE_ON_OPENCV 0x02

#include "Network.h"
#include "Streamer.h"
#include <map>
#include <functional>
#include <jaula.h> // uses libfl-dev

/**
 * \namespace Tobii
 * \brief Contains all the tools relative to Glasses 2 SDK
 **/

namespace Tobii {
	static const short DATAROW_EYEL_PC_FILLED = 0x0001; /*!< Flag for check if Pupil Center left is filled. */
	static const short DATAROW_EYEL_PD_FILLED = 0x0002; /*!< Flag for check if Pupil Diameter left is filled. */
	static const short DATAROW_EYEL_GD_FILLED = 0x0004; /*!< Flag for check if Gaze Direction right is filled. */
	static const short DATAROW_EYER_PC_FILLED = 0x0008; /*!< Flag for check if Pupil Center right is filled. */
	static const short DATAROW_EYER_PD_FILLED = 0x0010; /*!< Flag for check if Pupil Diameter right is filled. */
	static const short DATAROW_EYER_GD_FILLED = 0x0020; /*!< Flag for check if Gaze Direction right is filled. */
	static const short DATAROW_GP2_FILLED = 0x0040; /*!< Flag for check if Gaze Position 2D right is filled. */
	static const short DATAROW_GP3_FILLED = 0x0080; /*!< Flag for check if Gaze Position 3D right is filled. */
	static const short DATAROW_GY_FILLED = 0x0100; /*!< Flag for check if gyroscope is filled */
	static const short DATAROW_AC_FILLED = 0x0200; /*!< Flag for check if accelerometer is filled */
	static const short DATAROW_FIXATION_FILLED = 0x00FF; /*!< Flag for check if everything is filled */
	static const short DATAROW_FILLED = 0x03FF; /*!< Flag for check if everything is filled */
	
	static const char DATA_IS_TRACKING = 0x01; /*!< Flag for check if the data is tracking data */
	static const char DATA_IS_FRAME = 0x02; /*!< Flag for check if the data is video frame data */
	static const char DATA_IS_SYNC = 0x04; /*!< Flag for check if the data is sync data */
	/**
	 * \struct Eye TobiiGlassesSDK.h "Struct representing eye relative data."
	 **/
	struct Eye {
		float pupilCenter[3]; /*!< in mm can be used to compare the eye position with the tracksphere. */
		float pupilDiameter; /*!< in mm. */
		float gazeDirection[3]; /*!< unity vector with origo in the comea position. */
	};
	/**
	 * \struct DataRow TobiiGlassesSDK.h "Struct that will contain a row buffer of Glasses Data."
	 **/
	struct DataRow {
		int64_t ts; /*!< timestamp. */
		short check_flags; /*!< each binary in the check_flags will check if all the values are filled */
		Eye left; /*!< left eye. */
		Eye right; /*!< right eye. */
		float gazePosition2D[2]; /*!< in pixels position on the scene camera uimage. */
		float gazePosition3D[3]; /*!< in mm relative to scene camera. */
		float gyroscope[3]; /*!< glasses rotation in deg/s. */
		float accelerometer[3]; /*!< glasses acceleration in m/s^2. */
		DataRow() {
			check_flags = 0x0000; // we will set the flags to 0 here.
		}
	};
	/**
	 * \struct DataRow TobiiGlassesSDK.h "Struct that will contain a row buffer of Glasses Video Frames."
	 * The frame takes the reference given by TobiiGlassesSDK but it will not copy it.
	 * Maybe this will be interesting later, we have to investigate further.
	 **/
	struct DataRowFrame {
		int64_t pts; /*!< timestamp given for this frame. */
		Frame * frame; /*!< the video frame in OpenCV format. */
		DataRowFrame(int64_t _pts, Frame * _frame) {
			pts = _pts;
			*frame = Tobii::Frame(*_frame);
		}
	};
	
	/**
	 * \struct Data TobiiGlassesSDK.h "Struct that will contain a buffer of Glasses Data."
	 **/
	struct Data {
		std::map<int64_t, DataRowFrame> bufferFrames; /*!< the frames buffer. */
		std::map<int64_t, DataRow> bufferData; /*!< the data buffer. */
		int64_t lastSyncTS;
		int64_t lastSyncPTS;
	};

	/**
	 * \class GlassesSDK
	 * \brief GlassesSDK handles Tobii Glasses 2 communications.
	 * 
	 * It is working with Network API and Streamer API from the same package.
	 * This class provide some methods to easily retrieve information from the Tobii
	 * Glasses 2 using video callback and data callback functions for the livestream or
	 * simply http requests giving json responses from the Tobii.
	 **/
	class GlassesSDK {
		// attributes, flags
		private:
			std::string m_IPAddress;	/*!< Glasses IP address: to set parameter value = 192.168.71.50. */
			int m_port;	/*!< Not used for the moment. */
			Network m_udpNw;	/*!< An instance to the Network API. */
			Streamer m_videoStreamer;	/*!< An instance to the Streamer API. */
			bool m_isRunning;	/*!< false if GlassesSDK is not running. */
			bool m_isStreaming;	/*!< false if Streamer is not Streaming. */
			std::function< void(std::string &) > m_dataCallback;	/*!< Callback that is called each time a data package is received. */
			std::function< void(Data & data, unsigned int sizeBuffer, const char flag) > m_dataFilledCallback;	/*!< Callback that is called each time a data is filled. */
			Data m_data;	/*!< Data buffer. */
			unsigned int m_data_size;	/*!< Data buffer size. To set paramter value = 10 */
		public:
			static const char HTTP_HEADER_JSON = 0x01;	/*!< flag to set the application/json HTTP Header. */

		// methods, constructors and destructors
		private:

			/**
			 *  \brief post_request
			 *
			 * This method provide an easy way to do an http post request.
			 *
			 * \param url : the url where we want to do the request.
			 * \param data : post data that will be attached at the request.
			 * \param [out] response : response obtained after the request.
			 * \param [optional] flags : flag for the headers.
			 **/
			int post_request(const std::string & url,const std::string & data, std::string & response, char flags = 0x00);
			/**
			 *  \brief send_keepalive_msg
			 *
			 * Internal method used as a routine to send udp keep_alives messages to the Tobii Glasses 2. This is necessary to have livestream packages sent from the Tobii Glasses 2.
			 *
			 * \param socket: The socket file descriptor used to send the network packets.
			 * \param msg: the network message sent to the Tobii Glasses 2.
			 * \param glassesIPAddress: glasses IP address in which we will send the data. This parameter should be removed in later versions.
			 * \param port: the Tobii Glasses 2 port used for the exchange.
			 **/
			void send_keepalive_msg(int socket, const std::string msg, const char * glassesIPAddress, int port);
			/**
			 *  \brief setIsRunning
			 *
			 * Will stop the Tobii Glasses 2 SDK livestream. But will not stop it properly.
			 *
			 * \param false for stopping the livestream.
			 **/
			void setIsRunning(bool);
			/**
			 *  \brief setIsStreaming
			 *
			 * Will stop the Tobii Glasses 2 SDK livestream.
			 *
			 * \param false for stopping the livestream.
			 **/
			void setIsStreaming(bool);
			/**
			 *  \brief onDataArrived
			 *
			 * Callback that is called each time a data package is received.
			 *
			 * \param the data received.
			 **/
			void onDataArrived(const std::string &);
			/**
			 *  \brief onFrameArrived
			 *
			 * Callback that is called each time a video frame is received.
			 *
			 * \param pts : the timestamp received as a int64_t.
			 * \param frame : the frame received as a cv::Mat Object.
			 **/
			void onFrameArrived(const int64_t pts, Frame * frame);

		public:
			/**
			 * \brief Constructor will setup glasses
			 * \param const char * glasses's IP address.
			 * \param unsigned int the SDK buffer size.
			 **/
			GlassesSDK(const char* glassesIPAddress = "192.168.71.50", unsigned int bufferSize = 10);
			/**
			 * \brief Destructor will clean and delete every instance.
			 **/
			~GlassesSDK();

			/**
			 *  \brief clearFramesBuffer
			 *
			 * Will clear the Frames buffer and let only buffer_size elements inside.
			 *
			 * \param buffer_size : the number of elements to keep in the buffer after cleaning.
			 **/
			void clearFramesBuffer(unsigned int buffer_size);

			/**
			 *  \brief clearDataBuffer
			 *
			 * Will clear the Data buffer and let only buffer_size elements inside.
			 *
			 * \param buffer_size : the number of elements to keep in the buffer after cleaning.
			 **/
			void clearDataBuffer(unsigned int buffer_size);
			/**
			 *  \brief clearBuffers
			 *
			 * Will clear the buffers and let only buffer_size elements inside.
			 *
			 * \param buffer_size : the number of elements to keep in the buffers after cleaning.
			 **/
			void clearBuffers(unsigned int buffer_size);

			/**
			 *  \brief setVideoCallback
			 *
			 * Will set the video callback by calling the setter of Streamer API instance.
			 *
			 * \param func : the function address.
			 **/
			void setVideoCallback(std::function< void(Frame *, const int64_t) > func);
			/**
			 *  \brief setDataCallback
			 *
			 * Will set the data callback.
			 *
			 * \param func : the function address.
			 **/
			void setDataCallback(std::function< void(std::string &) > );
			/**
			 *  \brief setDataCallback
			 *
			 * Will set the data callback.
			 * This callback is called each time the buffer is
			 * filled with a complete data.
			 *
			 * \param func : the function address.
			 **/
			void setDataFilledCallback(std::function<void(Data & data, unsigned int sizeBuffer, char flag)>);
			/**
			 *  \brief discovery
			 *
			 * Will launch a discovery over the network and will try to find a single Tobii Glasses.
			 *
			 * \param setSyncDate : if true, will give the actual computer timestamp to the Tobii for synchronization.
			 * \return true if a Tobii Glasses 2 is discovered.
			 **/
			bool discovery(bool setSyncDate);
			/**
			 *  \brief livestream
			 *
			 * Will livestream the glasses, this method will retrieve information from Tobii Glasses on livestream until the end.
			 *
			 * \param flag : TOBII_LIVE_ON_WINDOW will call Streamer::run function, TOBII_LIVE_ON_OPENCV will call Streamer::runCV which is necessary to use the videoCallback for video frame retrieval.
			 **/
			void livestream(int flag);

			/**
			 *  \brief getTimeStamp
			 *
			 * Give the timestamp since System Epoch.
			 *
			 * \return timestamp in microseconds
			 **/
			static uint64_t getTimeStamp();
			/**
			 *  \brief timestamp2date
			 *
			 * Convert a timestamp to string date format.
			 *
			 * \param timestamp : a timestamp in microseconds.
			 * \param [out] date : string date formatted.
			 **/
			static void timestamp2date(uint64_t t, std::string &);
			/**
			 *  \brief date2timestamp
			 *
			 * Convert a string date to timestamp.
			 *
			 * \param date : string date formatted.
			 * \param [out] timestamp : a timestamp in microseconds.
			 **/
			static void date2timestamp(const std::string & datestr, uint64_t * t);
			/**
			 *  \brief showServices
			 *
			 * This method will print on the standard output the html document that will give the list of Tobii services.
			 **/
			void showServices();
			/**
			 *  \brief listProjects
			 *
			 * Will list the projects on the Tobii Glasses 2
			 *
			 * \return JSON string with the projects list.
			 **/
			std::string listProjects();
			/**
			 *  \brief createParticipant
			 *
			 * Create a participant on the Tobii Glasses 2
			 *
			 * \param project_id The participant will be created within the project with this id.
			 * \return JSON string with the participant information.
			 **/
			std::string createParticipant(const char * project_id);
			/**
			 *  \brief createCalibration
			 *
			 * Create a calibration for a participant and a project id.
			 *
			 * \param project_id The calibration will be created within the project with this id.
			 * \param participant_id The calibration will be linked with the participant with this id.
			 * \return JSON string with the calibration information.
			 **/
			std::string createCalibration(const char * project_id, const char * participant_id);
			/**
			 *  \brief startCalibration
			 *
			 * Start the calibration
			 *
			 * \param calibration_id The calibration with this id will be started.
			 * \return JSON string with the calibration information.
			 **/
			std::string startCalibration(const char * calibration_id);
			/**
			 *  \brief stopCalibration
			 *
			 * Stop the calibration
			 *
			 * \param calibration_id The calibration with this id will be stopped.
			 * \return JSON string with the calibration information.
			 **/
			std::string stopCalibration(const char * calibration_id);
			/**
			 *  \brief wait_for_status
			 *
			 * Will launch requests until the values of field key in the response matches the refvals.
			 *
			 * \param api_action the url for the post request (the service uri).
			 * \param key the key element to test in the JSON answer.
			 * \param refvals the reference values to match.
			 * \return the value matched.
			 **/
			std::string wait_for_status(const std::string & api_action,const std::string & key, const std::vector<std::string> & refvals);
			/**
			 *  \brief calibrate
			 *
			 * Start a calibration and return if the calibration succeeded or not.
			 *
			 * \param calibration_id the calibration we want to start.
			 * \return true if calibration succeed, false if not.
			 **/
			bool calibrate(const std::string & calibration_id);
			/**
			 *  \brief listRecordings
			 *
			 * Will list the recordings on the Tobii Glasses 2
			 *
			 * \return JSON string with the recordings list.
			 **/
			std::string listRecordings();
			/**
			 *  \brief showRecording
			 *
			 * Will give the recording information.
			 *
			 * \param recording_id the recording id.
			 * \return JSON string with the recording information.
			 **/
			std::string showRecording(const char * recording_id);
			/**
			 *  \brief createRecording
			 *
			 * Create a recording on the Tobii Glasses 2
			 *
			 * \param participant_id the recording will be created for this participant.
			 * \return JSON string with the recording information.
			 **/
			std::string createRecording(const char * participant_id);
			/**
			 *  \brief startRecording
			 *
			 * Start a recording on the Tobii Glasses 2
			 *
			 * \param participant_id the recording that will be started.
			 * \return JSON string with the recording start information.
			 **/
			std::string startRecording(const char * recording_id);
			/**
			 *  \brief stopRecording
			 *
			 * Stop a recording on the Tobii Glasses 2
			 *
			 * \param participant_id the recording that will be stopped.
			 * \return JSON string with the recording stop information.
			 **/
			std::string stopRecording(const char * recording_id);
			/**
			 *  \brief stopLiveStream
			 *
			 * Will stop the Tobii Glasses 2 SDK livestream. This one is highly recommended.
			 **/
			void stopLiveStream();
			/**
			 *  \brief JAULA_traverse
			 *
			 * Will print to the stdout the content of the JAULA::Value which is json parsed with the JAULA parser.
			 **/
			static void JAULA_traverse(JAULA::Value * val, int level = 0);
	};
}
#endif
