/************************************
 * Author: Pierre-Marie Plans       *
 * E-mail: pierre.plans@gmail.com   *
 ************************************/

#ifndef NETWORK_H_
#define NETWORK_H_

#ifdef WINDOWS
#include <winsock.h>
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

#include <string>

#ifdef WINDOWS
	WSADATA wsaData;
#endif

/**
 * This class is a UDP Network wrapper for low level C functions.
 **/
class Network {
	static int portEncode(int);
	public:
	/**
	 * This constructor will setup the Network API for windows
	 * @author Pierre-Marie Plans
	 **/
	Network();
	/**
	 * Make a DGRAM socket and returns the socket file descriptor
	 * @author Pierre-Marie Plans
	 * @return socket file descriptor (int)
	 **/
	int makeSocket();
	/**
	 * Bind a destination address to a socket
	 * @author Pierre-Marie Plans
	 * @param in (int) socket file descriptor
	 * @param inout (struct sockaddr_in) address
	 * @return 1 if success, 0 else
	 **/
	int bindSocket(struct sockaddr_in & dest, int sock);
	/**
	 * Init a sockaddress with the values provided.
	 * @author Pierre-Marie Plans
	 * @param inout (struct sockaddr_in) address
	 * @param in (const char *) address to bind, NULL is considered as Broadcast address
	 * @param in (int) port
	 * @return 1 if success, 0 else
	 **/
	void configureSockAddr(struct sockaddr_in & sockaddr, const char * inetaddr, int port);
	/**
	 * Send a UDP packet to a specific address.
	 * @author Pierre-Marie Plans
	 * @param in (struct sockaddr_in) address
	 * @param in (int) socket file descriptor
	 * @param in (const char *) packet to send, finishing by '\0' char
	 * @return 1 if success, 0 else
	 **/
	int sendUDPTo(struct sockaddr_in & sockaddr, int sockfd, const char * msg);
	/**
	 * Receive an UDP packet from a given address. For the moment it reads only 1024 bytes.
	 * @author Pierre-Marie Plans
	 * @param in (struct sockaddr_in) address
	 * @param in (int) socket file descriptor
	 * @param out (std::string) packet received
	 * @return number of bytes read, 0 or less means an error occured.
	 **/
	int recvUDPFrom(struct sockaddr_in * sockaddr, int sockfd, std::string & msgout, bool dontwait=false);
	/**
	 * Closes the socket.
	 * @author Pierre-Marie Plans
	 * @param in (int) socket file descriptor
	 **/
	void clearSocket(int sock);
	/**
	 * Will print network API errors on STDERR standard output.
	 * @author Pierre-Marie Plans
	 **/
	void printErrors();
};

#endif
