#include "TobiiGlassesSDK.h"

#include <iostream>
#include <vector>
#include <sstream>
#include <algorithm>
#include <thread>
#include <string.h>
#include <time.h>
#include <curl/curl.h>
#include <exception>
#include <signal.h> // for signal interruption, this is for test app
#include <unistd.h>
#include <fcntl.h>
#include <opencv2/opencv.hpp>

size_t HTTPPostCallback(void *ptr, size_t size, size_t nmemb, void *userdata) {
	*(std::string*)userdata += std::string((char *)ptr);
	return size*nmemb;
}

Tobii::GlassesSDK::GlassesSDK( const char* glassesIPAddress )
:m_IPAddress(glassesIPAddress),
m_isRunning(true),
m_isStreaming(false),
m_dataCallback(NULL),
m_dataFilledCallback(NULL) {
	/* In windows, this will init the winsock stuff */ 
	curl_global_init(CURL_GLOBAL_ALL);

	m_videoStreamer.init(0, NULL);
}

Tobii::GlassesSDK::~GlassesSDK() {
	std::cout << "Tobii::GlassesSDK shutdown." << std::endl;
}

void Tobii::GlassesSDK::setVideoCallback(std::function< void(cv::Mat &, const int64_t) > func) {
	m_videoStreamer.setVideoCallback(func);
}

void Tobii::GlassesSDK::setDataCallback(std::function< void(std::string &) > func) {
	m_dataCallback = func;
}

void Tobii::GlassesSDK::setDataFilledCallback(std::function<void(Data &)> func) {
	m_dataFilledCallback = func;
}

int Tobii::GlassesSDK::post_request(const std::string & url,const std::string & data, std::string & response, char flags) {
	CURL *curl;
	CURLcode res;

	/* get a curl handle */ 
	curl = curl_easy_init();
	if(curl) {
		/* First set the URL that is about to receive our POST. This URL can
		 just as well be a https:// URL if that is what should receive the
		 data. */ 
		//curl_easy_setopt(curl, CURLOPT_PROTOCOLS, CURLPROTO_HTTP); 
		std::cout << "post_request " << url << std::endl;
		curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
		if(flags & Tobii::GlassesSDK::HTTP_HEADER_JSON) {
			curl_easy_setopt(curl, CURLOPT_HTTPHEADER, "Content-Type: application/json");
		}
		/* Now specify the POST data */ 
		if(data.compare("") != 0) {
			curl_easy_setopt(curl, CURLOPT_POSTFIELDS, data.c_str());
		}
		curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, HTTPPostCallback);
		curl_easy_setopt(curl, CURLOPT_WRITEDATA, &response);
	 
		/* Perform the request, res will get the return code */ 
		res = curl_easy_perform(curl);
		/* Check for errors */ 
		if(res != CURLE_OK) {
			fprintf(stderr, "curl_easy_perform() failed: %s\n", curl_easy_strerror(res));
			return 0;
		}
	 
		/* always cleanup */ 
		curl_easy_cleanup(curl);
	}	
	curl_global_cleanup();
	return 1;
}

bool Tobii::GlassesSDK::discovery(bool setSyncDate) {
	char sendLine[128];
	std::string msg;
	struct sockaddr_in saddr, saddr_listen, saddr_recv;
	struct timespec timer;
	int counter=10;
	uint64_t ts = 0;
	std::string date;
	int s = m_udpNw.makeSocket();
	memset(sendLine, 0, 64);
	if(setSyncDate)
	{
		ts = getTimeStamp();
		timestamp2date(ts, date);
		strcpy(sendLine, "{\"type\":\"discover\", \"date\":\"");
		strcpy(sendLine+28, date.c_str());
		strcpy(sendLine+48, "\"}");
	}else{
		strcpy(sendLine, "{\"type\":\"discover\"}");
	}
	timer.tv_sec=0;
	timer.tv_nsec=100000000;
	m_udpNw.configureSockAddr(saddr, m_IPAddress.c_str(), 13006);
	//m_udpNw.configureSockAddr(saddr, NULL, 13006);

	//int srecv = network.makeSocket(srecvaddr);
	m_udpNw.configureSockAddr(saddr_listen, NULL, 13006);
	m_udpNw.bindSocket(saddr_listen, s);

	m_udpNw.sendUDPTo(saddr, s, sendLine);
	std::cout << "Sent message \"" << sendLine << "\"." << std::endl;
	std::cout << "Listening:" << std::endl;
	while(--counter>1){
		int n = m_udpNw.recvUDPFrom(&saddr_recv, s, msg, true);
//		if(saddr_recv.
		nanosleep(&timer, NULL);
		if(n>0) {
			try {
				try {
					std::cout << msg << std::endl;
					std::istringstream is(msg);
					JAULA::Value * data = JAULA::Parser::parseStream(is);
					JAULA::Value::ValueType type = data->getType();

					if(type==JAULA::Value::TYPE_OBJECT)
					{
						std::map<std::string, JAULA::Value*> values = ((JAULA::Value_Object*)data)->getData();
						if(values["type"]!=NULL && ((JAULA::Value_String*)values["type"])->getData().compare("discover")==0) {
							std::cout << "discovery packet received" << std::endl;
							JAULA_traverse((JAULA::Value*)data, 0);
							continue;
						}else if(values["type"]!=NULL && ((JAULA::Value_String*)values["type"])->getData().compare("identity")==0) {
							std::cout << "identity packet received" << std::endl;
							return true;
						}
						JAULA_traverse(data);
					}
				}catch(JAULA::Exception & e) {
					std::cout << e.getOrigin() << " : " << e.getDetail() << std::endl;
				}
		
			}catch(const std::exception & e) {
				std::cout << "exception:" << e.what() << std::endl;
			}
		}
		else
			std::cout << "\tReceiving nothing..." << std::endl;
	}
	return false;
}

void Tobii::GlassesSDK::send_keepalive_msg(int socket, const std::string msg, const char * glassesIPAddress, int port) {
	struct timespec timer;
	struct sockaddr_in saddr;
	Network udpNw;
	timer.tv_sec=0;
	timer.tv_nsec=50000000; // 50000000 10-9 sec
	udpNw.configureSockAddr(saddr, glassesIPAddress, port);
	std::cout << "Start keepalive loop for " << std::string(glassesIPAddress) << ":" << port << std::endl;
	while(m_isRunning) {
		udpNw.sendUDPTo(saddr, socket, msg.c_str());
		nanosleep(&timer, NULL);
	}
	std::cout << "Stop keepalive loop for " << std::string(glassesIPAddress) << ":" << port << std::endl;
}

void Tobii::GlassesSDK::onDataArrived(const std::string & dataRecv) {
	try {
		std::istringstream is(dataRecv);
		try {
			JAULA::Value_Complex * data = JAULA::Parser::parseStream(is);
			//JAULA_traverse(data, 0);
			JAULA::Value::ValueType type = data->getType();
			// check if the data is an object, else, this is not tracking data
			if(type==JAULA::Value::TYPE_OBJECT) {
				std::map<int64_t, DataRow>::iterator it;
				std::map<std::string, JAULA::Value *> values = ((JAULA::Value_Object*)data)->getData();
				if(values["ts"] && !values["pts"] && !values["vts"]) {
					int64_t ts=((JAULA::Value_Number_Int*)values["ts"])->getData();
					it=m_data.bufferData.find(ts);
					// check if there is already data within the buffer with that value, if not we will create it.
					if(it==m_data.bufferData.end()) {
						it=m_data.bufferData.insert(std::pair<int64_t, DataRow>(ts, DataRow())).first;
						it->second.check_flags = 0x0000;
					}
					// eye related values
					if(values["eye"]) {
						// checking the eye to modify
						Eye * e = NULL;
						short flag = 0x0001;
						if(((JAULA::Value_String*)values["eye"])->getData().compare("right")) {
							e = &(it->second.right);
							flag = flag << 2;
						}else{
							e = &(it->second.left);
						}

						// eye checks
						if(values["pc"] && e!=NULL) {
							std::list<JAULA::Value*>::const_iterator itpc = ((JAULA::Value_Array*)values["pc"])->getData().begin();
							e->pupilCenter[0] = ((JAULA::Value_Number*)*itpc)->getData();
							itpc = std::next(((JAULA::Value_Array*)values["pc"])->getData().begin(), 1);
							e->pupilCenter[1] = ((JAULA::Value_Number*)*itpc)->getData();
							itpc = std::next(((JAULA::Value_Array*)values["pc"])->getData().begin(), 2);
							e->pupilCenter[2] = ((JAULA::Value_Number*)*itpc)->getData();
							it->second.check_flags |= flag;
						}else if(values["pd"] && e!=NULL) {
							e->pupilDiameter = ((JAULA::Value_Number*)values["pd"])->getData();
							it->second.check_flags |= (flag<<1);
						}
					}
					// gaze and inertial related values
					if(values["gd"]) {
						std::list<JAULA::Value*>::const_iterator itpd = ((JAULA::Value_Array*)values["gd"])->getData().begin();
						it->second.gazeDirection[0] = ((JAULA::Value_Number*)*itpd)->getData();
						itpd = std::next(((JAULA::Value_Array*)values["gd"])->getData().begin(), 1);
						it->second.gazeDirection[1] = ((JAULA::Value_Number*)*itpd)->getData();
						itpd = std::next(((JAULA::Value_Array*)values["gd"])->getData().begin(), 2);
						it->second.gazeDirection[2] = ((JAULA::Value_Number*)*itpd)->getData();
						it->second.check_flags |= DATAROW_GD_FILLED;
					}else if(values["gp"]) {
						std::list<JAULA::Value*>::const_iterator itgp = ((JAULA::Value_Array*)values["gp"])->getData().begin();
						it->second.gazePosition2D[0] = ((JAULA::Value_Number*)*itgp)->getData();
						itgp = std::next(((JAULA::Value_Array*)values["gp"])->getData().begin(), 1);
						it->second.gazePosition2D[1] = ((JAULA::Value_Number*)*itgp)->getData();
						it->second.check_flags |= DATAROW_GP2_FILLED;
					}else if(values["gp3"]) {
						std::list<JAULA::Value*>::const_iterator itgp3 = ((JAULA::Value_Array*)values["gp3"])->getData().begin();
						it->second.gazePosition3D[0] = ((JAULA::Value_Number*)*itgp3)->getData();
						itgp3 = std::next(((JAULA::Value_Array*)values["gp3"])->getData().begin(), 1);
						it->second.gazePosition3D[1] = ((JAULA::Value_Number*)*itgp3)->getData();
						itgp3 = std::next(((JAULA::Value_Array*)values["gp3"])->getData().begin(), 2);
						it->second.gazePosition3D[2] = ((JAULA::Value_Number*)*itgp3)->getData();
						it->second.check_flags |= DATAROW_GP3_FILLED;
					}else if(values["gy"]) {
						std::list<JAULA::Value*>::const_iterator itgy = ((JAULA::Value_Array*)values["gy"])->getData().begin();
						it->second.gyroscope[0] = ((JAULA::Value_Number*)*itgy)->getData();
						itgy = std::next(((JAULA::Value_Array*)values["gy"])->getData().begin(), 1);
						it->second.gyroscope[1] = ((JAULA::Value_Number*)*itgy)->getData();
						itgy = std::next(((JAULA::Value_Array*)values["gy"])->getData().begin(), 2);
						it->second.gyroscope[2] = ((JAULA::Value_Number*)*itgy)->getData();
						it->second.check_flags |= DATAROW_GY_FILLED;
					}else if(values["ac"]) {
						std::list<JAULA::Value*>::const_iterator itac = ((JAULA::Value_Array*)values["ac"])->getData().begin();
						it->second.accelerometer[0] = ((JAULA::Value_Number*)*itac)->getData();
						itac = std::next(((JAULA::Value_Array*)values["ac"])->getData().begin(), 1);
						it->second.accelerometer[1] = ((JAULA::Value_Number*)*itac)->getData();
						itac = std::next(((JAULA::Value_Array*)values["ac"])->getData().begin(), 2);
						it->second.accelerometer[2] = ((JAULA::Value_Number*)*itac)->getData();
						it->second.check_flags |= DATAROW_AC_FILLED;
					}
					if(it->second.check_flags == DATAROW_FIXATION_FILLED && m_dataFilledCallback!=NULL) {
						m_dataFilledCallback(m_data);
					}
				}
			}
		}catch(JAULA::Exception & e) {
			std::cout << e.getOrigin() << " : " << e.getDetail() << std::endl;
		}
		
	}catch(const std::exception & e) {
		std::cout << e.what() << std::endl;
	}
}

void Tobii::GlassesSDK::livestream(int flag) {
	std::string msg;
	struct sockaddr_in /*saddr_listen,*/ saddr_recv;
	int port = 49152;
	//int recvPort = 1024;

	std::string KA_DATA_MSG = "{\"type\": \"live.data.unicast\", \"key\": \"some_GUID\", \"op\": \"start\"}";
	std::string KA_VIDEO_MSG = "{\"type\": \"live.video.unicast\", \"key\": \"some_other_GUID\", \"op\": \"start\"}";
	std::string PIPELINE_DEF = "";
	if(flag == TOBII_LIVE_ON_WINDOW) {
		/*LAGGI ONE std::string PIPELINE_DEF = "udpsrc do-timestamp=true name=src blocksize=1316 closefd=false buffer-size=5600 !" \
			"mpegtsdemux !" \
			"queue !" \
			"ffdec_h264 max-threads=0 !" \
			"ffmpegcolorspace !" \
			"xvimagesink name=video double-buffer=true pixel-aspect-ratio=2/2";*/
		/*BEST ONE 07/08/2015 <1s lag, noise std::string PIPELINE_DEF = "udpsrc do-timestamp=true name=src blocksize=1024 closefd=false buffer-size=4096 !" \
			"mpegtsdemux !" \
			"queue !" \
			"ffdec_h264 max-threads=2 !" \
			"ffmpegcolorspace !" \
			"xvimagesink name=video double-buffer=true pixel-aspect-ratio=1/1";*/
		//PIPELINE_DEF = "udpsrc do-timestamp=true name=src blocksize=1024 closefd=false buffer-size=8192 !" \
			"mpegtsdemux !" \
			"queue !" \
			"ffdec_h264 max-threads=4 !" \
			"ffmpegcolorspace !" \
			"xvimagesink name=video double-buffer=true pixel-aspect-ratio=1/1";
		/*PIPELINE_DEF = "udpsrc do-timestamp=true name=src blocksize=1024 buffer-size=8192 !" \
			"mpegtsdemux !" \
			"queue !" \
			"ffdec_h264 max-threads=4 !" \
			"ffmpegcolorspace !" \
			"xvimagesink name=video double-buffer=true pixel-aspect-ratio=1/1";*/
		PIPELINE_DEF = "udpsrc do-timestamp=true name=src blocksize=1024 buffer-size=8192 !" \
			"tsdemux !" \
			"h264parse !" \
			"queue !" \
			"avdec_h264 max-threads=4 !" \
			"videoconvert !" \
			"timeoverlay font-desc=\"Verdana bold 18px\" valignment=1 !" \
			"xvimagesink name=video double-buffer=true pixel-aspect-ratio=1/1";
	}else if(flag == TOBII_LIVE_ON_OPENCV) {
		// GSTREAMER 0.10
		//PIPELINE_DEF = "udpsrc do-timestamp=true name=src blocksize=1024 closefd=false buffer-size=8192 !" \
			"mpegtsdemux !" \
			"queue !" \
			"ffdec_h264 max-threads=4 !" \
			"ffmpegcolorspace !" \
			"appsink name=video sync=true emit-signals=true";
		// GSTREAMER 1.0
		PIPELINE_DEF = "udpsrc do-timestamp=true name=src blocksize=1024 buffer-size=8192 !" \
			"tsdemux !" \
			"h264parse !" \
			"queue !" \
			"avdec_h264 max-threads=4 !" \
			"videoconvert !" \
			"video/x-raw, format=BGR !" \
			/*"jpegenc !" \*/
			"appsink name=video sync=true emit-signals=true";
	}
	int s_data = m_udpNw.makeSocket();
	std::thread KA_data_thread(&Tobii::GlassesSDK::send_keepalive_msg, this, s_data, KA_DATA_MSG, m_IPAddress.c_str(), port);
	int s_video = m_udpNw.makeSocket();
	std::thread KA_video_thread(&Tobii::GlassesSDK::send_keepalive_msg, this, s_video, KA_VIDEO_MSG, m_IPAddress.c_str(), port);

	m_isStreaming = true;

	// setup streaming
	m_videoStreamer.setupPipeline(PIPELINE_DEF);
	m_videoStreamer.configureSrcSockFD(s_video);
	std::thread stream_thread;
	if(flag == TOBII_LIVE_ON_WINDOW) {
		stream_thread = std::thread(&Streamer::run, &m_videoStreamer );
	}else if(flag == TOBII_LIVE_ON_OPENCV) {
		stream_thread = std::thread(&Streamer::runCV,  &m_videoStreamer );
	}

	// running streaming
	std::cout << "Listening..." << std::endl;
	while(m_isRunning)
	{

		if(!m_isStreaming)
		{
			m_videoStreamer.stop();
			stream_thread.join();
			if(!m_videoStreamer.isRunning()) {
				m_isRunning = false;
				KA_data_thread.join();
				KA_video_thread.join();
				continue;
			}
		}

		int n = m_udpNw.recvUDPFrom(&saddr_recv, s_data, msg, true);
		if(n>0) {			
			onDataArrived(msg);
			if(m_dataCallback!=NULL)
			{
				m_dataCallback(msg);
			}
			//std::cout << "data: " << msg << std::endl;
			/*std::map<std::string, std::string> values;
			values = parseMessage(msg, values);
			std::map<std::string, std::string>::iterator it = values.find("type");
			if(it!=values.end() && it->second.compare("discover")==0) {
				std::cout << "discovery packet received" << std::endl;
				continue;
			}else if(it!=values.end() && it->second.compare("identity")==0) {
				//std::cout << msg << std::endl;
				for(it=values.begin();it!=values.end();it++) {
					//std::cout << it->first << " - " << it->second << std::endl;
				}
			}*/
		}
	}
	if(fcntl(s_data, F_GETFD) != -1 || errno != EBADF)
		close(s_data);
	if(fcntl(s_video, F_GETFD) != -1 || errno != EBADF)
		close(s_video);
}

uint64_t Tobii::GlassesSDK::getTimeStamp() {
    struct timeval tv;
    gettimeofday(&tv,NULL);
    return tv.tv_sec*(uint64_t)1000000+tv.tv_usec;
}

void Tobii::GlassesSDK::timestamp2date(uint64_t t, std::string & date) {
	// format  "date":"2014-09-24T12:13:14Z"
	char buf[32];
	uint64_t ms = (t-t%1000)/1000;
	uint64_t timestamp = (ms-ms%1000)/1000;
	memset(buf, 0, 32);
	strftime(buf, 32, "%Y-%m-%dT%H:%M:%SZ", localtime ((time_t*)&timestamp));
	date = std::string(buf);
	return;
}

void Tobii::GlassesSDK::date2timestamp(const std::string & datestr, uint64_t * t) {
	struct tm tm;
	if ( strptime(datestr.c_str(), "%Y-%m-%d %H:%M:%S", &tm) != NULL ) {
		*t = (uint64_t)mktime(&tm);
	}
}

void Tobii::GlassesSDK::showServices() {
	std::string resp;
	std::string requestURI="http://"+m_IPAddress+"/services";
	post_request(requestURI,"", resp);
}

std::string Tobii::GlassesSDK::listProjects() {
	std::string resp;
	std::string requestURI="http://"+m_IPAddress+"/api/projects";
	post_request(requestURI,"", resp);
	return resp;
}

std::string Tobii::GlassesSDK::createParticipant(const char * project_id) {
	std::string data("{'pa_project': "+std::string(project_id)+"}");
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/participants";
	post_request(requestURI, data, json_data);
	return json_data;
}

std::string Tobii::GlassesSDK::createCalibration(const char * project_id, const char * participant_id) {
	std::string data("{'ca_project': "+std::string(project_id)+", 'ca_type': 'default', 'ca_participant': "+std::string(participant_id)+"}");
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/calibrations";
	post_request(requestURI, data, json_data);
	return json_data;
}

std::string Tobii::GlassesSDK::startCalibration(const char * calibration_id) {
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/calibrations/"+std::string(calibration_id)+"/start";
	post_request(requestURI, "", json_data);
	return json_data;
}

std::string Tobii::GlassesSDK::stopCalibration(const char * calibration_id) {
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/calibrations/"+std::string(calibration_id)+"/stop";
	post_request(requestURI, "", json_data);
	return json_data;
}

std::string Tobii::GlassesSDK::wait_for_status(const std::string & api_action,const std::string & key, const std::vector<std::string> & refvals) {
	std::string url = "http://"+m_IPAddress+"/"+api_action;
	int running = 1;
	std::string response;
	struct timespec timer;
	timer.tv_sec=1;
	timer.tv_nsec=0;
	while(running) {
		post_request(url, "", response, HTTP_HEADER_JSON);
		try {
			std::istringstream is(response);
			try {
				JAULA::Value * data = JAULA::Parser::parseStream(is);
				JAULA::Value::ValueType type = data->getType();
				
				if(type==JAULA::Value::TYPE_OBJECT)
				{
					std::map<std::string, JAULA::Value*> values = ((JAULA::Value_Object*)data)->getData();
					for(unsigned int i=0;i<refvals.size();i++) {
						if(values[key]!=NULL && ((JAULA::Value_String*)values[key])->getData().compare(refvals[i])==0) {
							running = 0;
							response = ((JAULA::Value_String*)values[key])->getData();
						}
					}
				}
			}catch(JAULA::Exception & e) {
				std::cout << e.getOrigin() << " : " << e.getDetail() << std::endl;
			}
		}catch(const std::exception & e) {
			std::cout << e.what() << std::endl;
		}
		nanosleep(&timer, NULL);
	}
	return response;
}

bool Tobii::GlassesSDK::calibrate(const std::string & calibration_id) {
	std::vector<std::string> refvals;
	startCalibration(calibration_id.c_str());
	refvals.push_back("failed");
	refvals.push_back("calibrated");
	std::string status = wait_for_status("/api/calibrations/" + calibration_id + "/status", "ca_state", refvals);
	return status.compare("failed")==0;
}

std::string Tobii::GlassesSDK::listRecordings() {
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/recordings";
	post_request(requestURI, "", json_data);
	std::cout << json_data << std::endl;
	return json_data;
}

std::string Tobii::GlassesSDK::showRecording(const char * recording_id) {
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/recordings/"+std::string(recording_id);
	post_request(requestURI, "", json_data);
	return json_data;
}

std::string Tobii::GlassesSDK::createRecording(const char * participant_id) {
	std::string data("{'rec_participant': "+std::string(participant_id)+"}");
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/recordings";
	post_request(requestURI, data, json_data);
	return json_data;
}

std::string Tobii::GlassesSDK::startRecording(const char * recording_id) {
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/recordings/"+std::string(recording_id)+"/start";
	post_request(requestURI, "", json_data);
	return json_data;
}

std::string Tobii::GlassesSDK::stopRecording(const char * recording_id) {
	std::string json_data;
	std::string requestURI="http://"+m_IPAddress+"/api/recordings/"+std::string(recording_id)+"/stop";
	post_request(requestURI, "", json_data);
	return json_data;
}

void Tobii::GlassesSDK::setIsRunning(bool value) {
	m_isRunning = value;
}

void Tobii::GlassesSDK::setIsStreaming(bool value) {
	m_isStreaming = value;
}

void Tobii::GlassesSDK::stopLiveStream() {
	setIsStreaming(false);
}

void Tobii::GlassesSDK::JAULA_traverse(JAULA::Value * val, int level) {
	JAULA::Value::ValueType type = val->getType();
	std::string prefix="";
	for(int i=0;i<level;i++) {
		prefix+="\t";
	}
	if(type==JAULA::Value::TYPE_ARRAY) {
		std::list<JAULA::Value*> values = ((JAULA::Value_Array*)val)->getData();
		std::list<JAULA::Value*>::iterator it;
		for(it=values.begin();it!=values.end();it++) {
			JAULA_traverse(*it, level+1);
		}
	}else if(type==JAULA::Value::TYPE_OBJECT) {
		std::map<std::string, JAULA::Value *> values = ((JAULA::Value_Object*)val)->getData();
		std::map<std::string, JAULA::Value *>::iterator it;
		for(it=values.begin();it!=values.end();it++) {
			std::cout << prefix << it->first << ": ";
			JAULA_traverse(it->second, level+1);
		}
	}else if(type==JAULA::Value::TYPE_NULL) {
		std::cout << "NULL" << std::endl;
	}else if(type==JAULA::Value::TYPE_STRING){
		std::cout << ((JAULA::Value_String*)val)->getData() << std::endl;
	}else if(type==JAULA::Value::TYPE_NUMBER){
		std::cout << ((JAULA::Value_Number*)val)->getData() << std::endl;
	}else if(type==JAULA::Value::TYPE_NUMBER_INT){
		std::cout << ((JAULA::Value_Number_Int*)val)->getData() << std::endl;
	}else if(type==JAULA::Value::TYPE_BOOLEAN){
		std::cout << ((JAULA::Value_Boolean*)val)->getData() << std::endl;
	}
	return;
}
