/******************************************************
 * This is a wrapper for GStreamer-0.10               *
 * This will ensure an easy access to the GStreamer   *
 * API                                                *
 * Author: Pierre-Marie Plans                         *
 * E-mail: pierre.plans@gmail.com                     *
 ******************************************************/

#ifndef STREAMER_H_
#define STREAMER_H_

#include <string>
#include <functional>
#include <string.h>
#include <opencv2/opencv.hpp>
#include <gst/gst.h>
#include <gst/app/gstappsink.h>

/**
 * \file Streamer.h
 * \brief {This class handles the GStreamer stream capture 
 * in order to get the video frames from the Tobii}
 * \author Pierre-Marie Plans
 * \date 12/08/2015
 **/

// GSTREAMER
// http://blog.nicolargo.com/2009/03/gstreamer-la-theorie.html
// http://gstreamer.freedesktop.org/data/doc/gstreamer/head/manual/html/index.html
// http://benow.ca/docs/gstreamer-manual.pdf
// http://gstreamer.freedesktop.org/data/doc/gstreamer/head/manual/html/chapter-programs.html
// http://gstreamer-devel.966125.n4.nabble.com/udpsrc-performance-td4655463.html
/**
 * \struct CustomData
 * \brief CustomData is used for internal data sharing.
 *
 * We need this struct in order to communicate
 * with the GStreamer callbacks system.
 **/
typedef struct _CustomData {
	gboolean is_live;	/*!< false if it is not running. */
	GstElement *pipeline;	/*!< GStreamer pipeline reference. */
	GstElement *appsink;	/*!< GStreamer pipeline appsink reference. */
	GMainLoop *loop;	/*!< GStreamer main loop reference. */
	GstMapInfo *info;	/*!< Frame buffer Map info reference. */
	IplImage *frame;	/*!< Frame buffer. */
	std::function< void(cv::Mat &, const int64_t) > 
	videoCallback;
	/*!< Callback that is called each time a frame is filled. */
} CustomData;

/**
 * \class Streamer
 * \brief Streamer handles the video streaming input.
 * 
 * It is working with GStreamer and OpenCV. GStreamer will take data information from the pipeline.
 * And call the video callback with the information filled. The pipeline must finish with appsink sink.
 * @TODO improve class data, can we remove the struct and use the class instead ?
 **/
class Streamer {
	GstElement * m_pipeline;	/*!< GStreamer pipeline reference (the same one on the struct). */
	CustomData m_data;	/*!< The instance of CustomData. */
	bool m_isRunning;	/*!< false if it is not running. */
	public:
		/**
		 *  \brief Streamer constructor
		 **/
		Streamer();
		/**
		 *  \brief Streamer destructor
		 **/
		~Streamer();
		/**
		 *  \brief Streamer init
		 *
		 * init will setup gstreamer and the class instance. argc and argv are used by
		 * GStreamer for initialisation.
		 *
		 * \param argc : number of program arguments
		 * \param argv : pointer to the program arguments
		 **/
		void init(int argc, char ** argv);
		/**
		 *  \brief setupPipeline
		 *
		 * Will set the pipeline passed as argument and launch the GStreamer parsing.
		 *
		 * \param pipeline : GStreamer pipeline
		 * \return 1 if success, 0 if not
		 **/
		int setupPipeline(const std::string & pipeline);
		/**
		 *  \brief configureSrcSockFD
		 *
		 * Will set the udp socket file descriptor to the udpsrc from the GStreamer pipeline.
		 *
		 * \param pipeline : GStreamer pipeline
		 * \return 1 if success, 0 if not
		 **/
		void configureSrcSockFD(int sockfd);
		/**
		 *  \brief setVideoCallback
		 *
		 * Will set the video callback.
		 *
		 * \param pipeline : GStreamer pipeline
		 * \return 1 if success, 0 if not
		 **/
		void setVideoCallback(std::function< void(cv::Mat &, const int64_t) >);
		/**
		 *  \brief run
		 *
		 * run is the method that will handle the GStreamer main loop.
		 * Call it if you want to see the streaming in a screen window.
		 **/
		void run();
		/**
		 *  \brief runCV
		 *
		 * runCV is the method that will handle the GStreamer main loop.
		 * Call it if you want to use the frames in the buffer.
		 * Only this method will invoke the video callback.
		 **/
		void runCV();
		/**
		 *  \brief stop
		 *
		 * Will stop the GStreamer main_loop.
		 **/
		void stop();
		/**
		 *  \brief isRunning
		 *
		 * check if Streamer is running.
		 *
		 * \return true if Streamer is running, false if not
		 **/
		bool isRunning();
};
/* The appsink has received a buffer */
/**
 * \fn static GstFlowReturn cb_new_sample(GstElement *sink, CustomData *data)
 * \brief Callback that will retrieve the video frame from the GStreamer pipeline.
 *
 * This callback is called when a new GStreamer sample is available from the appsink.
 * This is used internally and should be removed from the header lately.
 *
 * \param GstElement * sink : GstAppSink* sink, the pipeline output reference.
 * \param CustomData * data : shared data used to retrieve the videoframe.
 * \return GST_FLOW_OK if everything goes fine, GST_FLOW_ERROR if not.
 */
static GstFlowReturn cb_new_sample(GstElement *sink, CustomData *data);

/**
 * \fn void cb_message(GstBus *bus, GstMessage *msg, CustomData *data)
 * \brief Callback that will retrieve pipeline messages.
 *
 * This callback is called when a new GStreamer pipeline message is received.
 * This callback is used internally and should be removed from the header lately.
 *
 * \param GstBus * bus : GstBus pipeline reference.
 * \param GstMessage * msg : message received.
 * \param CustomData * data : shared data.
 * \return GST_FLOW_OK if everything goes fine, GST_FLOW_ERROR if not.
 */
void cb_message(GstBus *bus, GstMessage *msg, CustomData *data);

#endif
