#include "Streamer.h"
#include <iostream>
#include <gio/gio.h>
#include <vector>

Streamer::Streamer() {

}

Streamer::~Streamer() {
	std::cout << "Streamer end." << std::endl;
	if(m_pipeline!=NULL) {
		gst_element_set_state (m_pipeline, GST_STATE_NULL);
		gst_object_unref (GST_OBJECT(m_pipeline));
	}
	if(m_data.info!=NULL)
		delete m_data.info;
}

void Streamer::init(int argc, char ** argv) {
	/* Initialize GStreamer */
	gst_init (&argc, &argv);

	/* Initialize our data structure */
	memset (&m_data, 0, sizeof (m_data));
}

int Streamer::setupPipeline(const std::string & conf) {
	GError * gerror = NULL;
	m_pipeline = gst_parse_launch (conf.c_str(), &gerror);
	if(gerror!=NULL) {
		g_print("Unable to parse the pipeline: %s\n", gerror->message);
		return 0;
	}
	return 1;
}

void Streamer::configureSrcSockFD(int sockfd) {
	GstElement *source;
	// trying to get the socket attribute and set it to our socket fd
	source = gst_bin_get_by_name (GST_BIN (m_pipeline), "src");
	//gst_udpsrc_set_property(GST_OBJECT(source), PROP_SOCKFD, &sockfd, NULL);
	#ifdef GSTREAMER_010
	g_object_set (GST_OBJECT(source), "sockfd", sockfd, NULL);
	#else
		//g_object_set(GST_OBJECT(source), "sockfd", sockfd, NULL);
		GError * gerror = NULL;
		GSocket * gSock = g_socket_new_from_fd(sockfd, &gerror);
		if(gerror!=NULL) {
			g_print("Unable to create the socket: %s\n", gerror->message);
			return;
		}
		g_object_set(GST_OBJECT(source), "socket", gSock, (char*)NULL); // sentinel NULL
		if(gerror!=NULL) {
			g_print("Unable to set the socket attribute: %s\n", gerror->message);
			return;
		}
	#endif
	g_object_unref (source);
}

void Streamer::setVideoCallback(std::function< void(cv::Mat &, const int64_t) > func) {
	m_data.videoCallback = func;
}

void Streamer::run() {
	GstBus *bus;
	GstStateChangeReturn ret;
	GMainLoop *main_loop;

	m_isRunning = true;
	bus = gst_element_get_bus (m_pipeline);

	/* Start playing */
	ret = gst_element_set_state (m_pipeline, GST_STATE_PLAYING);
	if (ret == GST_STATE_CHANGE_FAILURE) {
		g_printerr ("Unable to set the pipeline to the playing state.\n");
		gst_object_unref (m_pipeline);
		return;
	} else if (ret == GST_STATE_CHANGE_NO_PREROLL) {
		m_data.is_live = TRUE;
	}

	main_loop = g_main_loop_new (NULL, FALSE);
	m_data.loop = main_loop;
	m_data.pipeline = m_pipeline;

	gst_bus_add_signal_watch (bus);
	g_signal_connect (bus, "message", G_CALLBACK (cb_message), &m_data);

	g_main_loop_run (main_loop);

	/* Free resources */
	g_main_loop_unref (main_loop);
	gst_object_unref (bus);
	m_isRunning = false;
}

void Streamer::runCV() {
	GstBus *bus;
	GstStateChangeReturn ret;
	GMainLoop *main_loop;

	m_isRunning = true;
	bus = gst_element_get_bus (m_pipeline);

	/* Start playing */
	ret = gst_element_set_state (m_pipeline, GST_STATE_PLAYING);
	if (ret == GST_STATE_CHANGE_FAILURE) {
		g_printerr ("Unable to set the pipeline to the playing state.\n");
		gst_object_unref (m_pipeline);
		return;
	} else if (ret == GST_STATE_CHANGE_NO_PREROLL) {
		m_data.is_live = TRUE;
	}

	main_loop = g_main_loop_new (NULL, FALSE);
	m_data.loop = main_loop;
	m_data.pipeline = m_pipeline;
	m_data.appsink = gst_bin_get_by_name (GST_BIN (m_pipeline), "video");
	m_data.frame = NULL;
	m_data.info = new GstMapInfo();

	gst_bus_add_signal_watch (bus);
	g_signal_connect (bus, "message", G_CALLBACK (cb_message), &m_data);
	g_signal_connect (m_data.appsink, "new-sample", G_CALLBACK (cb_new_sample), &m_data);

	g_main_loop_run (main_loop);

	/* Free resources */
	g_main_loop_unref (main_loop);
	gst_object_unref (bus);
	m_isRunning = false;
}

/*bool CaptureGstBuffer(GstAppSink *sink, CustomData *data)
{            
    GstBuffer *buffer = gst_app_sink_pull_buffer(sink);
    if (buffer)
    {        
        IplImage *frame = cvCreateImage(cvSize(720, 576), IPL_DEPTH_16U, 3);
        if (frame == NULL)
        {
            g_printerr("IplImageFrame is null.\n");
        }
        else
        {
            //buffer = gst_app_sink_pull_buffer(sink);
            frame->imageData = (char*)GST_BUFFER_DATA(buffer);        
            if (frame->imageData == NULL)
            {
                g_printerr("IplImage data is null.\n");        
            }
        }           
    }            
    return true;
}*/

void Streamer::stop() {
	g_main_loop_quit(m_data.loop);
}

bool Streamer::isRunning() {
	return m_isRunning;
}

void cb_message (GstBus *bus, GstMessage *msg, CustomData *data) {
	switch (GST_MESSAGE_TYPE (msg)) {
	case GST_MESSAGE_ANY:
		g_print ("Streamer::Received a message\n");
	case GST_MESSAGE_ERROR: {
		GError *err;
		gchar *debug;
		 
		gst_message_parse_error (msg, &err, &debug);
		g_print ("Streamer::Error: %s\n", err->message);
		g_error_free (err);
		g_free (debug);
		 
		gst_element_set_state (data->pipeline, GST_STATE_READY);
		g_main_loop_quit (data->loop);
		break;
	}
	case GST_MESSAGE_EOS:
		/* end-of-stream */
		gst_element_set_state (data->pipeline, GST_STATE_READY);
		g_main_loop_quit (data->loop);
		break;
	case GST_MESSAGE_BUFFERING: {
		gint percent = 0;

		/* If the stream is live, we do not care about buffering. */
		if (data->is_live) break;

		gst_message_parse_buffering (msg, &percent);
		g_print ("Streamer::Buffering (%3d%%)\r", percent);
		/* Wait until buffering is complete before start/resume playing */
		if (percent < 100)
			gst_element_set_state (data->pipeline, GST_STATE_PAUSED);
		else
			gst_element_set_state (data->pipeline, GST_STATE_PLAYING);
		break;
	}
	case GST_MESSAGE_CLOCK_LOST:
		/* Get a new clock */
		gst_element_set_state (data->pipeline, GST_STATE_PAUSED);
		gst_element_set_state (data->pipeline, GST_STATE_PLAYING);
	break;
	case GST_MESSAGE_WARNING:
		GError *err;
		gchar *debug;
		gst_message_parse_warning(msg, &err, &debug);
		g_print ("Streamer::Warning: %s\n", err->message);
		g_error_free (err);
		g_free (debug);
	break;
	default:
		/* Unhandled message */
	break;
	}
}

/* The appsink has received a buffer */
static GstFlowReturn cb_new_sample(GstElement *sink, CustomData *data) {
	static int i=0;
	GstBuffer *buffer = NULL;
	GstSample *sample = NULL;
	GstCaps *caps = NULL;
	const GstStructure * structure = NULL;
	int nFrameWidth = -1, nFrameHeight = -1;
	int depth = 0;

	/* Retrieve the buffer */
	g_signal_emit_by_name (sink, "pull-sample", &sample);
	//sample = gst_app_sink_pull_sample((GstAppSink*)sink);

	if (sample != NULL) {
		// setup image file name
		std::stringstream s;
		std::string fname = "test_";
		s << fname << i++ << ".jpg";

		buffer = gst_sample_get_buffer (sample);
		//structure = gst_sample_get_info (sample);
		caps = gst_sample_get_caps(sample);
		structure = gst_caps_get_structure(caps, 0);
		if(buffer!=NULL) {
			// get structure information
			if(structure!=NULL) {
		       		gst_structure_get_int(structure, "width", &nFrameWidth );
				gst_structure_get_int(structure, "height", &nFrameHeight ); 
			}
			if(data->frame == NULL) {

				/*!
				 * \file cap_gstreamer.cpp
				 * \author Nils Hasler <hasler@mpi-inf.mpg.de>
				 *         Max-Planck-Institut Informatik
				 * \author Dirk Van Haerenborgh <vhdirk@gmail.com>
				 * \author Pierre-Marie Plans <pierre.plans@gmail.com> (few import fixes)
				 *
				 * \brief Use GStreamer to read/write video
				 */
				#if GST_VERSION_MAJOR > 0
				depth = 0;
				const gchar* name = gst_structure_get_name(structure);
				const gchar* format = gst_structure_get_string(structure, "format");

				if (!name || !format) {
					std::cout << "Streamer::error : name and format NULL" << std::endl;
				    return GST_FLOW_ERROR;
				}

				// we support 3 types of data:
				//     video/x-raw, format=BGR   -> 8bit, 3 channels
				//     video/x-raw, format=GRAY8 -> 8bit, 1 channel
				//     video/x-bayer             -> 8bit, 1 channel
				// bayer data is never decoded, the user is responsible for that
				// everything is 8 bit, so we just test the caps for bit depth

				if (strcasecmp(name, "video/x-raw") == 0)
				{
				    if (strcasecmp(format, "BGR") == 0) {
					depth = 3;
				    }
				    else if(strcasecmp(format, "GRAY8") == 0){
					depth = 1;
				    }
				}
				else if (strcasecmp(name, "video/x-bayer") == 0)
				{
				    depth = 1;
				}
				g_print("%s %s\n", name, format);
				#endif
				// end of code inspired
				if (depth > 0) {
					data->frame = cvCreateImageHeader(cvSize(nFrameWidth, nFrameHeight), IPL_DEPTH_8U, depth);
				} else {
					gst_caps_unref(caps);
					std::cout << "Streamer::error : depth<=0" << std::endl;
					return GST_FLOW_ERROR;
				}

				gst_caps_unref(caps);
			}

			gboolean success = gst_buffer_map(buffer,data->info, (GstMapFlags)GST_MAP_READ);
			if (!success){
				//something weird went wrong here. abort. abort.
				//fprintf(stderr,"GStreamer: unable to map buffer");
				std::cout << "Streamer::error : unable to get buffer info" << std::endl;
				return GST_FLOW_ERROR;
			}
			data->frame->imageData = (char*)data->info->data;
			cv::Mat frameMat(data->frame);
			gst_buffer_unmap(buffer,data->info);
			if(data->videoCallback!=NULL) {
				data->videoCallback(frameMat, GST_BUFFER_PTS(buffer));
			}
			/*try {
				cv::imwrite(s.str().c_str(), frameMat);
			}catch (const std::exception & ex) {
				fprintf(stderr, "Exception converting image to PNG format: %s\n", ex.what());
				return GST_FLOW_ERROR;
			}*/

			/* The only thing we do in this example is print a * to indicate a received buffer */
			//g_print ("%dx%d\n", nFrameWidth, nFrameHeight);
		}
	}else{
		return GST_FLOW_ERROR;
	}
	return GST_FLOW_OK;
}

