#include <map>
#include <functional>
#include <jaula.h> // uses libfl-dev
#include <fstream>

std::function< void(Data & data, unsigned int sizeBuffer, const char flag) > m_dataFilledCallback;	/*!< Callback that is called each time a data is filled. */

const short DATAROW_EYEL_PC_FILLED = 0x0001; /*!< Flag for check if Pupil Center left is filled. */
const short DATAROW_EYEL_PD_FILLED = 0x0002; /*!< Flag for check if Pupil Diameter left is filled. */
const short DATAROW_EYEL_GD_FILLED = 0x0004; /*!< Flag for check if Gaze Direction right is filled. */
const short DATAROW_EYER_PC_FILLED = 0x0008; /*!< Flag for check if Pupil Center right is filled. */
const short DATAROW_EYER_PD_FILLED = 0x0010; /*!< Flag for check if Pupil Diameter right is filled. */
const short DATAROW_EYER_GD_FILLED = 0x0020; /*!< Flag for check if Gaze Direction right is filled. */
const short DATAROW_GP2_FILLED = 0x0040; /*!< Flag for check if Gaze Position 2D right is filled. */
const short DATAROW_GP3_FILLED = 0x0080; /*!< Flag for check if Gaze Position 3D right is filled. */
const short DATAROW_GY_FILLED = 0x0100; /*!< Flag for check if gyroscope is filled */
const short DATAROW_AC_FILLED = 0x0200; /*!< Flag for check if accelerometer is filled */
const short DATAROW_FIXATION_FILLED = 0x00FF; /*!< Flag for check if everything is filled */
const short DATAROW_FILLED = 0x03FF; /*!< Flag for check if everything is filled */
	
const char DATA_IS_TRACKING = 0x01; /*!< Flag for check if the data is tracking data */
const char DATA_IS_FRAME = 0x02; /*!< Flag for check if the data is video frame data */
const char DATA_IS_SYNC = 0x04; /*!< Flag for check if the data is sync data */
/**
 * \struct Eye TobiiGlassesSDK.h "Struct representing eye relative data."
 **/
struct Eye {
	float pupilCenter[3]; /*!< in mm can be used to compare the eye position with the tracksphere. */
	float pupilDiameter; /*!< in mm. */
	float gazeDirection[3]; /*!< unity vector with origo in the comea position. */
};
/**
 * \struct DataRow TobiiGlassesSDK.h "Struct that will contain a row buffer of Glasses Data."
 **/
struct DataRow {
	int64_t ts; /*!< timestamp. */
	short check_flags; /*!< each binary in the check_flags will check if all the values are filled */
	Eye left; /*!< left eye. */
	Eye right; /*!< right eye. */
	float gazePosition2D[2]; /*!< in pixels position on the scene camera uimage. */
	float gazePosition3D[3]; /*!< in mm relative to scene camera. */
	float gyroscope[3]; /*!< glasses rotation in deg/s. */
	float accelerometer[3]; /*!< glasses acceleration in m/s^2. */
	DataRow() {
		check_flags = 0x0000; // we will set the flags to 0 here.
	}
};
/**
 * \struct DataRow TobiiGlassesSDK.h "Struct that will contain a row buffer of Glasses Video Frames."
 * The frame takes the reference given by TobiiGlassesSDK but it will not copy it.
 * Maybe this will be interesting later, we have to investigate further.
 **/
struct DataRowFrame {
	int64_t pts; /*!< timestamp given for this frame. */
	cv::Mat frame; /*!< the video frame in OpenCV format. */
	DataRowFrame(int64_t _pts, cv::Mat _frame) {
		pts = _pts;
		frame = _frame;
	}
};

/**
 * \struct Data TobiiGlassesSDK.h "Struct that will contain a buffer of Glasses Data."
 **/
struct Data {
	std::map<int64_t, DataRowFrame> bufferFrames; /*!< the frames buffer. */
	std::map<int64_t, DataRow> bufferData; /*!< the data buffer. */
	int64_t lastSyncTS;
	int64_t lastSyncPTS;
};

Data m_data;

void parse(const std::string & dataRecv) {
	try {
		std::istringstream is(dataRecv);
		try {
			JAULA::Value_Complex * data = JAULA::Parser::parseStream(is);
			//JAULA_traverse(data, 0);
			JAULA::Value::ValueType type = data->getType();
			// check if the data is an object, else, this is not tracking data
			if(type==JAULA::Value::TYPE_OBJECT) {
				std::map<int64_t, DataRow>::iterator it;
				std::map<std::string, JAULA::Value *> values = ((JAULA::Value_Object*)data)->getData();
				if(values["ts"] && !values["vts"]) {
					if(values["pts"]) {
						m_data.lastSyncTS = ((JAULA::Value_Number_Int*)values["ts"])->getData();
						m_data.lastSyncPTS = ((JAULA::Value_Number_Int*)values["pts"])->getData();
						m_dataFilledCallback(m_data,  m_data_size, DATA_IS_SYNC);
					}else{
						int64_t ts=((JAULA::Value_Number_Int*)values["ts"])->getData();
						it=m_data.bufferData.find(ts);
						// check if there is already data within the buffer with that value, if not we will create it.
						if(it==m_data.bufferData.end()) {
							it=m_data.bufferData.insert(std::pair<int64_t, DataRow>(ts, DataRow())).first;
							it->second.check_flags = 0x0000;
						}
						// eye related values
						if(values["eye"]) {
							// checking the eye to modify
							Eye * e = NULL;
							short flag = 0x0001;
							if(((JAULA::Value_String*)values["eye"])->getData().compare("right")) {
								e = &(it->second.right);
								flag = flag << 3;
							}else{
								e = &(it->second.left);
							}

							// eye checks
							if(values["pc"] && e!=NULL) {
								std::list<JAULA::Value*>::const_iterator itpc = ((JAULA::Value_Array*)values["pc"])->getData().begin();
								e->pupilCenter[0] = ((JAULA::Value_Number*)*itpc)->getData();
								itpc = std::next(((JAULA::Value_Array*)values["pc"])->getData().begin(), 1);
								e->pupilCenter[1] = ((JAULA::Value_Number*)*itpc)->getData();
								itpc = std::next(((JAULA::Value_Array*)values["pc"])->getData().begin(), 2);
								e->pupilCenter[2] = ((JAULA::Value_Number*)*itpc)->getData();
								it->second.check_flags |= flag;
							}else if(values["pd"] && e!=NULL) {
								e->pupilDiameter = ((JAULA::Value_Number*)values["pd"])->getData();
								it->second.check_flags |= (flag<<1);
							}else if(values["gd"]) {
								std::list<JAULA::Value*>::const_iterator itpd = ((JAULA::Value_Array*)values["gd"])->getData().begin();
								e->gazeDirection[0] = ((JAULA::Value_Number*)*itpd)->getData();
								itpd = std::next(((JAULA::Value_Array*)values["gd"])->getData().begin(), 1);
								e->gazeDirection[1] = ((JAULA::Value_Number*)*itpd)->getData();
								itpd = std::next(((JAULA::Value_Array*)values["gd"])->getData().begin(), 2);
								e->gazeDirection[2] = ((JAULA::Value_Number*)*itpd)->getData();
								it->second.check_flags |= (flag<<2);
							}
						}
						// gaze and inertial related values
						if(values["gp"]) {
							std::list<JAULA::Value*>::const_iterator itgp = ((JAULA::Value_Array*)values["gp"])->getData().begin();
							it->second.gazePosition2D[0] = ((JAULA::Value_Number*)*itgp)->getData();
							itgp = std::next(((JAULA::Value_Array*)values["gp"])->getData().begin(), 1);
							it->second.gazePosition2D[1] = ((JAULA::Value_Number*)*itgp)->getData();
							it->second.check_flags |= DATAROW_GP2_FILLED;
						}else if(values["gp3"]) {
							std::list<JAULA::Value*>::const_iterator itgp3 = ((JAULA::Value_Array*)values["gp3"])->getData().begin();
							it->second.gazePosition3D[0] = ((JAULA::Value_Number*)*itgp3)->getData();
							itgp3 = std::next(((JAULA::Value_Array*)values["gp3"])->getData().begin(), 1);
							it->second.gazePosition3D[1] = ((JAULA::Value_Number*)*itgp3)->getData();
							itgp3 = std::next(((JAULA::Value_Array*)values["gp3"])->getData().begin(), 2);
							it->second.gazePosition3D[2] = ((JAULA::Value_Number*)*itgp3)->getData();
							it->second.check_flags |= DATAROW_GP3_FILLED;
						}else if(values["gy"]) {
							std::list<JAULA::Value*>::const_iterator itgy = ((JAULA::Value_Array*)values["gy"])->getData().begin();
							it->second.gyroscope[0] = ((JAULA::Value_Number*)*itgy)->getData();
							itgy = std::next(((JAULA::Value_Array*)values["gy"])->getData().begin(), 1);
							it->second.gyroscope[1] = ((JAULA::Value_Number*)*itgy)->getData();
							itgy = std::next(((JAULA::Value_Array*)values["gy"])->getData().begin(), 2);
							it->second.gyroscope[2] = ((JAULA::Value_Number*)*itgy)->getData();
							it->second.check_flags |= DATAROW_GY_FILLED;
						}else if(values["ac"]) {
							std::list<JAULA::Value*>::const_iterator itac = ((JAULA::Value_Array*)values["ac"])->getData().begin();
							it->second.accelerometer[0] = ((JAULA::Value_Number*)*itac)->getData();
							itac = std::next(((JAULA::Value_Array*)values["ac"])->getData().begin(), 1);
							it->second.accelerometer[1] = ((JAULA::Value_Number*)*itac)->getData();
							itac = std::next(((JAULA::Value_Array*)values["ac"])->getData().begin(), 2);
							it->second.accelerometer[2] = ((JAULA::Value_Number*)*itac)->getData();
							it->second.check_flags |= DATAROW_AC_FILLED;
						}
						if(it->second.check_flags == DATAROW_FIXATION_FILLED && m_dataFilledCallback!=NULL) {
							m_dataFilledCallback(m_data,  m_data_size, DATA_IS_TRACKING);
						}
					}
				}
			}
		}catch(JAULA::Exception & e) {
			std::cout << e.getOrigin() << " : " << e.getDetail() << std::endl;
		}
		
	}catch(const std::exception & e) {
		std::cout << e.what() << std::endl;
	}
}

void cb(Data & data, unsigned int sizeBuffer, const char flag) {

}

int main(int argc, char ** argv) {
	m_dataFilledCallback = &cb;
	std::ifstream infile(argv[1]);
	std::string line;
	while (std::getline(infile, line))
	{
		parse(line);
	}

}
