/******************************************************
 * This is a port in C++11 of the Tobii Glasses 2 SDK *
 * This will ensure an easy access to the Tobii Glass *
 * Author: Pierre-Marie Plans                         *
 * E-mail: pierre.plans@gmail.com                     *
 ******************************************************/

#ifndef TOBIISDK_H_
#define TOBIISDK_H_

/**
 * \file TobiiGlassesSDK.h
 * \brief This class handles the Tobii Glasses 2 communications.
 * \author Pierre-Marie Plans
 * \date 12/08/2015
 **/

#define TOBII_LIVE_ON_WINDOW 0x01
#define TOBII_LIVE_ON_OPENCV 0x02
#define TOBII_LIVE_ON_FILE 0x03

#include <map>
#include <functional>
#include <jaula.h> // uses libfl-dev
#include <vector>

/**
 * \namespace Tobii
 * \brief Contains all the tools relative to Glasses 2 SDK
 **/

namespace Tobii {
	static const short DATAROW_EYEL_PC_FILLED = 0x0001; /*!< Flag for check if Pupil Center left is filled. */
	static const short DATAROW_EYEL_PD_FILLED = 0x0002; /*!< Flag for check if Pupil Diameter left is filled. */
	static const short DATAROW_EYEL_GD_FILLED = 0x0004; /*!< Flag for check if Gaze Direction right is filled. */
	static const short DATAROW_EYER_PC_FILLED = 0x0008; /*!< Flag for check if Pupil Center right is filled. */
	static const short DATAROW_EYER_PD_FILLED = 0x0010; /*!< Flag for check if Pupil Diameter right is filled. */
	static const short DATAROW_EYER_GD_FILLED = 0x0020; /*!< Flag for check if Gaze Direction right is filled. */
	static const short DATAROW_GP2_FILLED = 0x0040; /*!< Flag for check if Gaze Position 2D right is filled. */
	static const short DATAROW_GP3_FILLED = 0x0080; /*!< Flag for check if Gaze Position 3D right is filled. */
	static const short DATAROW_GY_FILLED = 0x0100; /*!< Flag for check if gyroscope is filled */
	static const short DATAROW_AC_FILLED = 0x0200; /*!< Flag for check if accelerometer is filled */
	static const short DATAROW_FIXATION_FILLED = 0x00FF; /*!< Flag for check if everything is filled */
	static const short DATAROW_FILLED = 0x03FF; /*!< Flag for check if everything is filled */
	
	static const char DATA_IS_TRACKING = 0x01; /*!< Flag for check if the data is tracking data */
	static const char DATA_IS_FRAME = 0x02; /*!< Flag for check if the data is video frame data */
	static const char DATA_IS_SYNC = 0x04; /*!< Flag for check if the data is sync data */
	/**
	 * \struct Eye TobiiGlassesSDK.h "Struct representing eye relative data."
	 **/
	struct Eye {
		float pupilCenter[3]; /*!< in mm can be used to compare the eye position with the tracksphere. */
		float pupilDiameter; /*!< in mm. */
		float gazeDirection[3]; /*!< unity vector with origo in the comea position. */
	};
	/**
	 * \struct DataRow TobiiGlassesSDK.h "Struct that will contain a row buffer of Glasses Data."
	 **/
	struct DataRow {
		int64_t ts; /*!< timestamp. */
		short check_flags; /*!< each binary in the check_flags will check if all the values are filled */
		Eye left; /*!< left eye. */
		Eye right; /*!< right eye. */
		float gazePosition2D[2]; /*!< in pixels position on the scene camera uimage. */
		float gazePosition3D[3]; /*!< in mm relative to scene camera. */
		float gyroscope[3]; /*!< glasses rotation in deg/s. */
		float accelerometer[3]; /*!< glasses acceleration in m/s^2. */
		DataRow() {
			check_flags = 0x0000; // we will set the flags to 0 here.
		}
	};
	/**
	 * \struct DataRow TobiiGlassesSDK.h "Struct that will contain a row buffer of Glasses Video Frames."
	 * The frame takes the reference given by TobiiGlassesSDK but it will not copy it.
	 * Maybe this will be interesting later, we have to investigate further.
	 **/
	struct DataRowFrame {
		int64_t pts; /*!< timestamp given for this frame. */
	};
	
	/**
	 * \struct Data TobiiGlassesSDK.h "Struct that will contain a buffer of Glasses Data."
	 **/
	struct Data {
		std::map<int64_t, DataRowFrame> bufferFrames; /*!< the frames buffer. */
		std::map<int64_t, DataRow> bufferData; /*!< the data buffer. */
		int64_t lastSyncTS;
		int64_t lastSyncPTS;
	};

	/**
	 * \class GlassesSDK
	 * \brief GlassesSDK handles Tobii Glasses 2 communications.
	 * 
	 * It is working with Network API and Streamer API from the same package.
	 * This class provide some methods to easily retrieve information from the Tobii
	 * Glasses 2 using video callback and data callback functions for the livestream or
	 * simply http requests giving json responses from the Tobii.
	 **/
	class GlassesSDKFile {
		// attributes, flags
		private:
			std::string m_fileName;
			std::string m_result_wooding_folder; /*******************Mariem*******************/
			bool m_TR;
			bool m_isRunning;
			std::function< void(std::string &) > m_dataCallback;	/*!< Callback that is called each time a data package is received. */
			std::function< void(Data & data, unsigned int sizeBuffer, const char flag) > m_dataFilledCallback;	/*!< Callback that is called each time a data is filled. */
			Data m_data;	/*!< Data buffer. */
			unsigned int m_data_size;	/*!< Data buffer size. To set paramter value = 10 */

		// methods, constructors and destructors
		private:
			/**
			 *  \brief onDataArrived
			 *
			 * Callback that is called each time a data package is received.
			 *
			 * \param the data received.
			 **/
			void onDataArrived(const std::string &);

		public:
			/**
			 * \brief Constructor will setup glasses
			 * \param const char * glasses's IP address.
			 * \param unsigned int the SDK buffer size.
			 **/
			GlassesSDKFile(const char * fileName = "", unsigned int bufferSize = 10);
			GlassesSDKFile(const char * fileName, unsigned int bufferSize, const char * result_wooding_folder, bool TR);

			/**
			 * \bool Destructor will clean and delete every instance.
			 **/
			~GlassesSDKFile();
			/**
			 *  \brief setDataCallback
			 *
			 * Will set the data callback.
			 *
			 * \param func : the function address.
			 **/
			void setDataCallback(std::function< void(std::string &) > );
			/**
			 *  \brief setDataCallback
			 *
			 * Will set the data callback.
			 * This callback is called each time the buffer is
			 * filled with a complete data.
			 *
			 * \param func : the function address.
			 **/
			void setDataFilledCallback(std::function<void(Data & data, unsigned int sizeBuffer, char flag)>);
			/**
			 *  \brief discovery
			 *
			 * Will launch a discovery over the network and will try to find a single Tobii Glasses.
			 *
			 * \param setSyncDate : if true, will give the actual computer timestamp to the Tobii for synchronization.
			 * \return true if a Tobii Glasses 2 is discovered.
			 **/
			bool discovery(bool setSyncDate);
			/**
			 *  \brief livestream
			 *
			 * Will livestream the glasses, this method will retrieve information from Tobii Glasses on livestream until the end.
			 *
			 * \param flag : TOBII_LIVE_ON_WINDOW will call Streamer::run function, TOBII_LIVE_ON_OPENCV will call Streamer::runCV which is necessary to use the videoCallback for video frame retrieval.
			 **/
			void livestream(int flag);

			/**
			 *  \brief getTimeStamp
			 *
			 * Give the timestamp since System Epoch.
			 *
			 * \return timestamp in microseconds
			 **/
			static uint64_t getTimeStamp();
			/**
			 *  \brief timestamp2date
			 *
			 * Convert a timestamp to string date format.
			 *
			 * \param timestamp : a timestamp in microseconds.
			 * \param [out] date : string date formatted.
			 **/
			static void timestamp2date(uint64_t t, std::string &);
			/**
			 *  \brief date2timestamp
			 *
			 * Convert a string date to timestamp.
			 *
			 * \param date : string date formatted.
			 * \param [out] timestamp : a timestamp in microseconds.
			 **/
			static void date2timestamp(const std::string & datestr, uint64_t * t);
			/**
			 *  \brief calibrate
			 *
			 * Start a calibration and return if the calibration succeeded or not.
			 *
			 * \param calibration_id the calibration we want to start.
			 * \return true if calibration succeed, false if not.
			 **/
			bool calibrate(const std::string & calibration_id);
			/**
			 *  \brief stopLiveStream
			 *
			 * Will stop the Tobii Glasses 2 SDK livestream. This one is highly recommended.
			 **/
			void stopLiveStream();



			/************Mariem*****************/
			double claculDistanceEspace(double x1,double x2, double y1, double y2,double z1 ,double z2);
			double calculSigma (double Distance_cible);
			std::vector<double> interpolationSigmaTR (std::vector<double> sigma,std::vector<double> time_stamp);
			//void interpolationSigmaTR (std::vector<double> sigma,std::vector<double> time_stamp);
			std::vector< std::pair<double,double> > interpolationFrameTR (std::vector<double> time_stamp, std::vector< std::pair<double,double> > coordinates);
			//void interpolationFrameTR (std::vector<double> time_stamp, std::vector< std::pair<double,double> > coordinates);
			
	};
}
#endif



