// ##################################################
// ###          TOBII_VIDEOSTREAM_HPP             ###
// ##################################################
#ifndef TOBII_VIDEOSTREAM_HPP
#define TOBII_VIDEOSTREAM_HPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STANDARD LIBRARIES: 
#include <iostream>
#include <vector>

// INTERNAL MODULES:
#include "Stream.hpp"
#include "VideoHeader.hpp"
#include "TobiiSDK/Data/Frame.hpp"

// EXTERNAL LIBRARIES: 
#include <opencv2/highgui/highgui.hpp>

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
namespace Tobii {

	// ##################################################
	// ###            CLASS VIDEO-STREAM              ###
	// ##################################################

	class VideoStream : public Stream, public VideoHeader
	{
	public:
		VideoStream ( const QString& filename = "" );
		VideoStream ( const VideoStream& stream );
		virtual ~VideoStream();

		virtual bool open ( const QString& filename ) = 0;
		virtual bool isOpen () const = 0;
		virtual bool close () = 0;

		VideoStream& operator=( const VideoStream& stream );

	}; // END CLASS VIDEO-STREAM
	// ##################################################

	// ##################################################
	// ###           CLASS I-VIDEO-STREAM             ###
	// ##################################################

	class IVideoStream : public virtual VideoStream
	{
	public:
		IVideoStream ( const QString& filename = "" );
		IVideoStream ( const VideoStream& stream );
		virtual ~IVideoStream();

		virtual Frame getNextFrame		() = 0;
		virtual Frame getPreviousFrame	() = 0; 
		virtual Frame getFrameAt		( Int64 timestamp ) = 0;
		virtual Frame getCurrentFrame	() = 0;

		virtual IVideoStream& operator>>( Frame& frame ) = 0;
	
	}; // END CLASS I-VIDEO-STREAM
	// ##################################################

	// ##################################################
	// ###           CLASS O-VIDEO-STREAM             ###
	// ##################################################

	class OVideoStream : public virtual VideoStream
	{
	public:
		OVideoStream ( const QString& filename = "" );
		OVideoStream ( const VideoStream& stream );
		virtual ~OVideoStream();

		virtual OVideoStream& operator<<( const Frame& frame ) = 0;
	
	}; // END CLASS O-VIDEO-STREAM
	// ##################################################

	// ##################################################
	// ###           CLASS IO-VIDEO-STREAM            ###
	// ##################################################

	class IOVideoStream : public IVideoStream, public OVideoStream
	{
	public:
		IOVideoStream ( const QString& filename = "" );
		IOVideoStream ( const VideoStream& stream );
		virtual ~IOVideoStream();

		virtual bool move ( const QString& foldername ) = 0;

		virtual Frame getNextFrame		() = 0;
		virtual Frame getPreviousFrame	() = 0; 
		virtual Frame getFrameAt		( Int64 timestamp ) = 0;
		virtual Frame getCurrentFrame	() = 0;

		virtual void setFrameAt		 ( Int64 timestamp, const Frame& frame ) = 0;
		virtual void setCurrentFrame ( const Frame& frame ) = 0;
		virtual void addFrame		 ( const Frame& frame ) = 0;

		virtual IVideoStream& operator>>( Frame& frame ) = 0;
		virtual OVideoStream& operator<<( const Frame& frame ) = 0;
	};

	// END CLASS IO-VIDEO-STREAM
	// ##################################################

	using VideoStreamPtr   = std::shared_ptr< VideoStream   >;
	using IVideoStreamPtr  = std::shared_ptr< IVideoStream  >;
	using OVideoStreamPtr  = std::shared_ptr< OVideoStream  >;
	using IOVideoStreamPtr = std::shared_ptr< IOVideoStream >;

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_VIDEOSTREAM_HPP
// ##################################################