// ##################################################
// ###       TOBII_VIDEOFRAMESSTREAM_CPP          ###
// ##################################################
#ifndef TOBII_VIDEOFRAMESSTREAM_CPP
#define TOBII_VIDEOFRAMESSTREAM_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "VideoFramesStream.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

namespace Tobii {

	// ##################################################
	// ###         CLASS VIDEO-FRAMES-STREAM          ###
	// ##################################################

	// --------------------------------------------------
	// CONSTRUCTOR - DESTRUCTOR

	VideoFramesStream::VideoFramesStream ( const QString& foldername, bool openEmpty, const QString& extension )
	: VideoStream( foldername ), _extension( extension ), _isOpen( false )
	{
		if ( ! foldername.isEmpty() )
			this->open( foldername, openEmpty );
	}

	VideoFramesStream::VideoFramesStream ( const VideoFramesStream& stream )
	: VideoStream( stream ), _extension( stream.extension() ), _isOpen( stream.isOpen() )
	{}

	VideoFramesStream& VideoFramesStream::operator=( const VideoFramesStream& stream )
	{
		this->VideoStream::operator=( stream );
		this->isOpen() = stream.isOpen();
		return (*this);
	}
	// --------------------------------------------------
	// GETTERS - SETTERS

	const QString&	VideoFramesStream::fullpath		() const	{ return this->_fullpath;	}
	QString&		VideoFramesStream::fullpath		()			{ return this->_fullpath;	}
	const QString&	VideoFramesStream::extension	() const	{ return this->_extension;	}
	QString&		VideoFramesStream::extension	()			{ return this->_extension;	}

	QString VideoFramesStream::getFileName ( Int64 timeStamp ) const
	{
		return QString( "%1/Frame_%2.%3" ).arg( this->fullpath() ).arg( timeStamp ).arg( this->extension() );
	}

	// --------------------------------------------------
	// OPEN - CLOSE

	bool VideoFramesStream::move ( const QString& foldername )
	{
		QDir dir( foldername+"/"+this->name() );

		if ( ! dir.exists() )
			dir.mkpath( dir.absolutePath() );

		QDir directory( this->fullpath() );
		for ( const QString& filename : directory.entryList( QStringList(QString("*.%1").arg(this->extension())), QDir::Files ) )
		{
			QFile file;
			if ( ! file.rename( this->fullpath()+"/"+filename, foldername+"/"+this->name()+"/"+filename ) )
			{
				qDebug() << "Failed to move file" << this->fullpath()+"/"+filename;
				return false;
			}
		}

		dir.rmdir( this->fullpath() );

		this->fullpath() = foldername+"/"+this->name();
		return true;
	}

	bool VideoFramesStream::open ( const QString& foldername )
	{
		return this->open( foldername, false );
	}

	bool VideoFramesStream::open ( const QString& foldername, bool openEmpty, const QString& extention )
	{
		// Init header data:
		this->fps()        = 25;
		this->frameCount() = 0;
		this->format()     = CV_8UC3;

		// open the directory
		QDir directory( foldername );

		// If the directory doesn't exist, the stream is being created
		if ( ! directory.exists() )
			directory.mkpath( directory.absolutePath() );

		// Init header data:
		this->fullpath()         = directory.absolutePath();
		this->name()             = directory.dirName();
		this->currentTimeStamp() = -40;
		this->currentIndex()     = -1;

		if ( openEmpty == false )
		{
			QStringList filters; filters << "Frame_*.png" << "Frame_*.jpg";
			QFileInfoList listOfFrames = directory.entryInfoList( filters, QDir::Files );

			for ( const QFileInfo& fileinfo : listOfFrames )
			{
				// Prep. to receive a frame:
				this->next();

				// Parse the name:
				this->indexes().emplace( this->currentTimeStamp(), this->currentIndex() );
				++this->frameCount();

				// if the resolution is not set, read it on the first image
				if ( this->resolution().width == 0 || this->resolution().height == 0 )
				{
					cv::Mat image = cv::imread( fileinfo.absoluteFilePath().toStdString(), CV_LOAD_IMAGE_COLOR );
					this->resolution().width  = image.cols;
					this->resolution().height = image.rows;
				}
			}

			// Init currenttimestamp:
			if ( this->frameCount() != 0 )
				this->setCurrentTimeStamp( 0 );
		}

		// Set to 'open':
		this->isOpen() = true;

		return this->isOpen();
	}

	bool	VideoFramesStream::isOpen	() const { return this->_isOpen; }
	bool&	VideoFramesStream::isOpen	()		 { return this->_isOpen; }

	bool VideoFramesStream::close ()
	{
		this->isOpen() = false;
		return this->isOpen();
	}

	// --------------------------------------------------
	// READ

	Frame VideoFramesStream::getFrameAt ( Int64 timeStamp )
	{
		QString filepath = this->getFileName(timeStamp);

		if ( QFile(filepath).exists() )
			return Frame ( timeStamp, 0, cv::imread( filepath.toStdString(), CV_LOAD_IMAGE_COLOR ) );

		return Frame( timeStamp, -1 );
	}

	Frame VideoFramesStream::getCurrentFrame ()
	{
		return this->getFrameAt( this->currentTimeStamp() );
	}

	Frame VideoFramesStream::getNextFrame ()
	{
		this->next();
		return this->getCurrentFrame();
	}

	Frame VideoFramesStream::getPreviousFrame ()
	{
		this->previous();
		return this->getCurrentFrame();
	}

	VideoFramesStream& VideoFramesStream::operator>> ( Frame& frame )
	{
		frame = this->getNextFrame();
		return (*this);
	}

	// --------------------------------------------------
	// WRITE

	void VideoFramesStream::setFrameAt ( Int64 timestamp, const Frame& frame )
	{
		cv::imwrite( this->getFileName(timestamp).toStdString(), frame, {CV_IMWRITE_PNG_COMPRESSION, 0, CV_IMWRITE_JPEG_QUALITY, 100} );
	}

	void VideoFramesStream::setCurrentFrame ( const Frame& frame )
	{
		this->setFrameAt( this->currentTimeStamp(), frame );
	}

	void VideoFramesStream::addFrame  ( const Frame& frame )
	{
		this->currentTimeStamp() += 40;
		this->currentIndex()     += 1;
		this->frameCount()       += 1;
		this->indexes().emplace( this->currentTimeStamp(), this->currentIndex() );
		this->setFrameAt( this->currentTimeStamp(), frame );

		// If this stream resolution isn't set, deduice it from this frame
		if ( this->resolution().width == 0 || this->resolution().height == 0 )
		{
			this->resolution().width  = frame.cols;
			this->resolution().height = frame.rows;
		}
	}

	VideoFramesStream& VideoFramesStream::operator<< ( const Frame& frame )
	{
		this->addFrame( frame );
		return (*this);
	}

	// END CLASS VIDEO-FRAMES-STREAM
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_VIDEOFRAMESSTREAM_CPP
// ##################################################