// ##################################################
// ###         TOBII_INTERPOLATION_CPP            ###
// ##################################################
#ifndef TOBII_INTERPOLATION_CPP
#define TOBII_INTERPOLATION_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "Interpolation.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###           NAMESPACE INTERPOLATE            ###
	// ##################################################
	namespace Interpolate {

		GazeFixation live ( const GazeRecordings& recordings, Int64 timeStamp, Int64 windowSize, bool forceInterpolation, bool forceComputingDistance )
		{
			// If the interpolation is forced, or the gaze at this time stamp doesn't exist
			// We perform interpolation:
			if ( forceInterpolation || recordings.find( timeStamp ) == recordings.end() )
			{
				// vector of data to interpolate, respectivly the time stamp of each 2D record,
				// the 2D gaze positions on x, the 2D gaze position on y, distance
				std::vector< double > gazeTimeStamps, gazePositions2DX, gazePositions2DY, gazeDistances;

				// Interpolation window: [timeStamp-windowSize, timeStamp+windowSize]
				// Considere all gaze fixations in it (add them to the above vectors)
				for ( Int64 currentTimeStamp( -windowSize+timeStamp ) ; currentTimeStamp <= (windowSize+timeStamp) ; ++currentTimeStamp )
				{
					// Look for a gaze fixation at this shifted time-stamp
					GazeRecordings::ConstIterator gazeIterator = recordings.find( currentTimeStamp );

					// If it exist
					if ( gazeIterator != recordings.end() )
					{
						GazeFixation fixation( gazeIterator->second );

						if ( fixation.isFieldSet( GazeFixation::ValidityFlag::POSITION_2D | GazeFixation::ValidityFlag::POSITION_3D ) )
						{
							// Considere it's fields 2D
							gazeTimeStamps.push_back( fixation.timeStamp() );
							gazePositions2DX.push_back( fixation.position2D().x() );
							gazePositions2DY.push_back( fixation.position2D().y() );

							// force recomputing the distance ?
							if ( forceComputingDistance )
								fixation.computeDistance();

							gazeDistances.push_back( fixation.distance() );
						}
					}
				}

				// If the window does not contain at least 2 salient point,
				// it is impossible to interpolate
				if ( gazeTimeStamps.size() >= 2 )
				{
					// Look at the offset of the first, and last gaze fixation found
					Int64 minOffset( gazeTimeStamps.front() - timeStamp );
					Int64 maxOffset( gazeTimeStamps.back()  - timeStamp );
					
					// We need one to be 'past' to the query time-stamp (sign negative),
					// and one to be 'neer-future' to the query time-stamp (sign positive).
					// otherwise the interpolation could be biased.
					if ( sign(minOffset) != sign(maxOffset) )
					{
						// Result gaze fixation
						Tobii::Point2 position2D;
						float distance;

						// Spline variables (on X, and Y):
						tk::spline spline2DX, spline2DY, splineDis;

						// Create spline functions
						spline2DX.set_points( gazeTimeStamps, gazePositions2DX );
						spline2DY.set_points( gazeTimeStamps, gazePositions2DY );
						splineDis.set_points( gazeTimeStamps, gazeDistances );

						// Query an interpolated value
						position2D.x() = spline2DX.at(timeStamp);
						position2D.y() = spline2DY.at(timeStamp);
						distance       = splineDis.at(timeStamp);

						// Check the interpolated value:
						UInt8 status( 0 );
						GazeFixation gazeFixation( timeStamp, 0 );
						if ( position2D.x() < 0 || position2D.x() > 1 || position2D.y() < 0 || position2D.y() > 1 )
						{
							return gazeFixation;
						}

						// Create gaze
						gazeFixation.setPosition2D( position2D );
						gazeFixation.setDistance( distance );
						return gazeFixation;
					}
				}
				// Not anough data to interpolate live
				return GazeFixation();
			}
			// The gaze already exists, and interpolation isn't forced,
			// So just return it
			return recordings.at( timeStamp );
		}

	// #ifdef OFFLINE_INTERPOLATION
	// 	std::vector< GazeFixation > vector ( const GazeRecordings& recordings, const std::vector<double>& timeStamps, bool forceInterpolation )
	// 	{
	// 		// vector of data to interpolate, respectivly:
	// 		// the gaze positions on x, the gaze position on y
	// 		std::vector< double > gazeTimeStamps, gazePositionsX, gazePositionsY;

	// 		for ( auto gazeIterator : recordings )
	// 		{
	// 			// Considere it's fields
	// 			gazeTimeStamps.push_back( gazeIterator.first				   );
	// 			gazePositionsX.push_back( gazeIterator.second.position2D().x() );
	// 			gazePositionsY.push_back( gazeIterator.second.position2D().y() );
	// 		}

	// 	#ifdef USE_TK_SPLINE
	// 		// Interpoation is separated on X and Y w.r.t. time
	// 		tk::spline splineX, splineY;

	// 		// Create spline functions
	// 		splineX.set_points( gazeTimeStamps, gazePositionsX );
	// 		splineY.set_points( gazeTimeStamps, gazePositionsY );
	// 	#endif
	// 	#ifdef USE_ALGIB
	// 		// Data (function value x,y, of time)
	// 		alglib::real_1d_array array_t;
	// 		alglib::real_1d_array array_x;
	// 		alglib::real_1d_array array_y;

	// 		array_t.setcontent( gazeTimeStamps.size(), &(gazeTimeStamps[0]) );
	// 		array_t.setcontent( gazePositionsX.size(), &(gazePositionsX[0]) );
	// 		array_t.setcontent( gazePositionsY.size(), &(gazePositionsY[0]) );

	// 		// Polynome stuff
	// 		alglib::ae_int_t				nbBasisFunction = 2;
	// 		alglib::ae_int_t				info;
	// 		alglib::ae_int_t				nbSamples = gazeTimeStamps.size();
	// 		alglib::barycentricinterpolant	polynomeX, polynomeY;
	// 		alglib::polynomialfitreport		report;

	// 		// Fit polynome
	// 		alglib::polynomialfit( array_t, array_x, nbBasisFunction, nbSamples, polynomeX, report );
	// 		alglib::polynomialfit( array_t, array_y, nbBasisFunction, nbSamples, polynomeY, report );
	// 	#endif


	// 		// Query interval, outside of this, interpolation is not define
	// 		Int64 firstTimeStamp = gazeTimeStamps.front();
	// 		Int64  lastTimeStamp = gazeTimeStamps.back();

	// 		// Query interpolated values
	// 		std::vector< GazeFixation > gazeFixations( timeStamps.size() );

	// 		// For each query time stamp:
	// 		for ( int index(0) ; index < timeStamps.size() ; ++index )
	// 		{
	// 			// Get the time stamp
	// 			Int64 timeStamp = timeStamps.at(index);

	// 			// If this time stamp is poossible to interpolate
	// 			if ( firstTimeStamp <= timeStamp && timeStamp <= lastTimeStamp )
	// 			{
	// 				// If the interpolation is forced or the gaze doesn't exist,
	// 				// We interpolate:
	// 				if ( forceInterpolation || recordings.find( timeStamp ) == recordings.end() )
	// 				{
	// 					// Result gaze fixation
	// 					float gaze_x, gaze_y;

	// 					// Query an interpolated value
	// 				#ifdef USE_TK_SPLINE
	// 					gaze_x = splineX.at(timeStamp);
	// 					gaze_y = splineY.at(timeStamp);
	// 				#endif
	// 				#ifdef USE_ALGIB
	// 					gaze_x = barycentriccalc(polynomeX, timeStamp);
	// 					gaze_y = barycentriccalc(polynomeY, timeStamp);
	// 				#endif

	// 					// Check the interpolated value:
	// 					UInt8 status( 0 );
	// 					if ( gaze_x < 0 || gaze_x > 1 || gaze_y < 0 || gaze_y > 1 )
	// 					{
	// 						gaze_x = 0;
	// 						gaze_y = 0;
	// 						status = 1;
	// 					}

	// 					// Re-use other existing gaze fields
	// 					// if ( recordings.at( timeStamp ) != recordings.end() )
	// 					// 	gazeFixations.at( index ) = recordings.at( timeStamp )

	// 					// Set interpolated values
	// 					gazeFixations.at( index ).timeStamp() = timeStamp;
	// 					gazeFixations.at( index ).status()    = status;
	// 					gazeFixations.at( index ).position2D().x() = gaze_x;
	// 					gazeFixations.at( index ).position2D().y() = gaze_y;
	// 				}
	// 				// The gaze exists and we do not force interpolation,
	// 				// Just use the existing
	// 				else
	// 					gazeFixations.at( index ) = recordings.at( timeStamp );
	// 			}
	// 		}
	// 		return gazeFixations;
	// 	}
	// #endif // OFFLINE_INTERPOLATION 

	} // END NAMESPACE INTERPOLATE
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_INTERPOLATION_CPP
// ##################################################