// ##################################################
// ###              TOBII_SALIENCY_CPP            ###
// ##################################################
#ifndef TOBII_SALIENCY_CPP
#define TOBII_SALIENCY_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "Saliency.hpp"
#include <iostream>

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###             NAMESPACE SALIENCY             ###
	// ##################################################	
	namespace Saliency {

		float sigmaAtResolution ( const cv::Size& resolution )
		{
			return std::max( resolution.width/2.0f*std::tan(alpha * PI / 180.0f)/std::tan(beta * PI / 180.0f), 0.0f );
		}

		float computeSigma ( const GazeFixation& fixation, cv::Size resolution )
		{
			return std::max( sigmaAtResolution(resolution)*A/(fixation.distance() + epsilon), 0.0f );
		}
			
		Frame computeSaliencyMap ( const GazeFixation& fixation, cv::Size resolution )
		{
			// Create zero saliency map
			Frame saliencyMap( fixation.timeStamp(), 0, cv::Mat::zeros(resolution, CV_8UC1) );

			// Check if the fixation is complete
			if ( fixation.isFieldSet( Tobii::GazeFixation::ValidityFlag::POSITION_2D | Tobii::GazeFixation::ValidityFlag::DISTANCE ) )
			{
				// Read frame dimension
				int width  = resolution.width;
				int height = resolution.height;
				int width_2  = std::floor( width/2.0);
				int height_2 = std::floor(height/2.0);

				// Read fixation
				Tobii::Point2 center = fixation.position2D();
				int center_x = std::floor(center.x() * width  );
				int center_y = std::floor(center.y() * height );

				// Compute sigma
				double sigma = computeSigma( fixation, resolution );

				// Compute the saliency
				cv::Mat gaussian1DX, gaussian1DY;

				gaussian1DX = cv::getGaussianKernel( height*3, sigma, CV_32FC1 );
				gaussian1DY = cv::getGaussianKernel( width*3,  sigma, CV_32FC1 );

				gaussian1DX = gaussian1DX.rowRange( height + height_2 - center_y, 2*height + height_2 - center_y );
				gaussian1DY = gaussian1DY.rowRange(  width + width_2  - center_x, 2*width  + width_2  - center_x );

				saliencyMap = gaussian1DX * gaussian1DY.t();

				// Normalize it, scale its dynamic range, and convert to unsigedn 64-bits floats
				double min(0), max(0);
				cv::minMaxLoc( saliencyMap, &min, &max );
				saliencyMap.convertTo( saliencyMap, CV_8UC1, 255/max );
			}

			return saliencyMap;
		}

		Frame computeSaliencyMapGPU ( const GazeFixation& fixation, cv::Size resolution )
		{
			// Create zero saliency map
			Frame saliencyMap( fixation.timeStamp(), 0, cv::Mat::zeros(resolution, CV_8UC1) );

			// Check if the fixation is complete
			if ( fixation.isFieldSet( Tobii::GazeFixation::ValidityFlag::POSITION_2D | Tobii::GazeFixation::ValidityFlag::DISTANCE ) )
			{
				// Read frame dimension
				int width  = resolution.width;
				int height = resolution.height;
				int width_2  = std::floor( width/2.0);
				int height_2 = std::floor(height/2.0);

				// Read fixation
				Tobii::Point2 center = fixation.position2D();
				int center_x = std::floor(center.x() * width  );
				int center_y = std::floor(center.y() * height );

				// Compute sigma
				double sigma = computeSigma( fixation, resolution );

				// Compute the saliency
				cv::Mat gaussian1DX, gaussian1DY;

				gaussian1DX = cv::getGaussianKernel( height*3, sigma, CV_32FC1 );
				gaussian1DY = cv::getGaussianKernel( width*3,  sigma, CV_32FC1 );

				gaussian1DX = gaussian1DX.rowRange( height + height_2 - center_y, 2*height + height_2 - center_y );
				gaussian1DY = gaussian1DY.rowRange(  width + width_2  - center_x, 2*width  + width_2  - center_x );

				// ---------------------------------------------------------------
				// GPU 
				cv::gpu::GpuMat gpuGaussian1DX, gpuGaussian1DY, gpuSaliency;

				// Upload gaussian X, and Y
				gpuGaussian1DX.upload( gaussian1DX );
				gpuGaussian1DY.upload( gaussian1DY );

				// this is equivalent to: gpuSaliency = 1.0 * gpuGaussian1DY * ( gpuGaussian1DY.t() ) + 0.0*cv::gpu::GpuMat()
				// see http://docs.opencv.org/2.4/modules/gpu/doc/operations_on_matrices.html
				cv::gpu::gemm( gpuGaussian1DX, gpuGaussian1DY, 1.0, cv::gpu::GpuMat(), 0.0, gpuSaliency, cv::GEMM_2_T );

				// Normalize it, scale its dynamic range, and convert to unsigedn 64-bits floats
				double min(0), max(0);
				cv::gpu::minMax( gpuSaliency, &min, &max );
				gpuSaliency.convertTo( gpuSaliency, CV_8UC1, 255/max );

				// Dowload back
				gpuSaliency.download( saliencyMap );
			}

			return saliencyMap;
		}

		Frame computeHeatMap ( const Frame& frame, const Frame& saliencyMap, double alpha )
		{
			// Compute the heat-map
			Frame heatMap( frame.timeStamp(), 0 );
			cv::applyColorMap( saliencyMap, heatMap, cv::COLORMAP_JET );

			// Mix with the frame
			heatMap = alpha * frame + (1-alpha) * heatMap;
			heatMap.convertTo( heatMap, CV_8UC3 );

			return heatMap;
		}

		Frame computeWeightedMap ( const Frame& frame, const Frame& saliencyMap )
		{
			cv::Mat imageHLS;
			cv::cvtColor( frame, imageHLS, CV_BGR2HLS );

			std::vector< cv::Mat > channelsHLS;
			cv::split( imageHLS, channelsHLS );

			cv::Mat saliency01 = saliencyMap;
			if ( saliency01.type() != CV_8UC1 )
				cv::cvtColor( saliency01, saliency01, CV_BGR2GRAY );

			saliency01.convertTo( saliency01, CV_32FC1, 1.0f/255.0f );
			channelsHLS.at(1).convertTo( channelsHLS.at(1), CV_32FC1 );

			cv::multiply( channelsHLS.at(1), saliency01, channelsHLS.at(1) );

			channelsHLS.at(1).convertTo( channelsHLS.at(1), CV_8UC1 );

			Frame output( frame.timeStamp(), 0 );
			cv::merge( channelsHLS, output );
			cv::cvtColor( output, output, CV_HLS2BGR );

			return output;
		}

		Frame threshold ( const Frame& saliency, double threshold )
		{
			Frame thresholded = saliency;
			if ( thresholded.type() != CV_8UC1 )
			{
				cv::cvtColor( thresholded, thresholded, CV_BGR2GRAY );
				cv::threshold( thresholded, thresholded, threshold, 255, cv::THRESH_BINARY );
				cv::cvtColor( thresholded, thresholded, CV_GRAY2BGR );
			}
			else
				cv::threshold( thresholded, thresholded, threshold, 255, cv::THRESH_BINARY );
			return thresholded;
		}


	} // END CLASS NAMESPACE SALIENCY
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_SALIENCY_CPP
// ##################################################