// ##################################################
// ###           TOBII_PRIMITIFTYPES_HPP          ###
// ##################################################
#ifndef TOBII_PRIMITIFTYPES_HPP
#define TOBII_PRIMITIFTYPES_HPP

/**
 * \file PrimitifTypes.hpp
 * \brief Contains typedefs for some primitif types, as well as basic mathematic function for them.
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STANDARD LIBRARIES
#include <cstdint>
#include <istream>
#include <fstream>
#include <ostream>
#include <sstream>
#include <string>

// ##################################################
// ###               NAMESPACE DATA               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
 namespace Tobii {

	// ##################################################
	// ###         PRIMITIF TYPE DEFINITIONS          ###
	// ##################################################

	/**
	 * \typedef UInt8
	 * \brief Unsigned 8-bit integer type.
	 *
	 * Defines the unsigned 8-bit integer type.
	 *
	 **/
	using UInt8 = unsigned short;

	/**
	 * \typedef Int64
	 * \brief 64-bit integer type.
	 *
	 * Defines the 64-bit integer type.
	 *
	 **/
	using Int64 = int64_t;

	// END PRIMITIF TYPE DEFINITIONS
	// ##################################################
	
	// ##################################################
	// ###            UTIL MATH FUNCTIONS             ###
	// ##################################################

	/**
	 * \brief Computes and returns the sign of the given argument.
	 * \tparam Type (typename) type of the object we want to check the sign.
	 * \param val (Type) value we want to compute the sign.
	 * \return (int) -1 if the argument is negative, 0 if it is 0, and +1 if it it positive 
	 *
	 * Computes and returns the sign of the given argument:
	 * - -1 if argument < 0
	 * -  0 if argument = 0
	 * - +1 if argument > 0
	 * Type(0) must be a valide constructor, and both less-than and greater-than operators must be valid operators as well.
	 *
	 **/
	template< typename Type >
	int sign( Type val )
	{
    	return (Type(0) < val) - (val < Type(0));
	}

	/**
	 * \brief Borns the given value between two given values.
	 * \tparam Type (typename) type of the object we want to born.
	 * \param val (Type) value we want to born.
	 * \param left (Type) lower bound.
	 * \param right (Type) upper bound.
	 * \return (Type) The borned value
	 *
	 * Computes and returns the sign of the given argument:
	 * - left  if argument < left
	 * - rigth if argument > right
	 * - val else
	 * Operators "operator<=", "operator<", "operator>=", "operator>" must be valid operators for the given template parameter.
	 *
	 **/
	template< typename Type >
	Type born( Type val, Type left, Type right )
	{
		if ( left <= val && val <= right )
			return val;
		if ( val < left )
			return left;
		if ( val > right )
			return right;
	}

	/**
	 * \brief Computes the square of the given argument.
	 * \tparam Type (typename) type of the object we want to compute the square.
	 * \param val (Type) Value we want to compute the square of.
	 * \return (Type) The squared value.
	 *
	 * Computes and returns the square of the given argument, by taking the product of it by itself:
	 * Operator "operator*" must be a valid operator for the given template parameter.
	 *
	 **/
	template< typename Type >
	Type square( Type val )
	{
		return val * val;
	}

	// END UTIL MATH FUNCTIONS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_PRIMITIFTYPES_HPP
// ##################################################