// ##################################################
// ###              DATA_POINT2_HPP               ###
// ##################################################
#ifndef DATA_POINT2_HPP
#define DATA_POINT2_HPP

/**
 * \file Point2.hpp
 * \brief Describes 2 dimentional points with floating-point type.
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STANDARD DEPENDENCES
#include <istream>
#include <ostream>
#include <sstream>
#include <string>

// INTERNAL MODULES
#include "PrimitifTypes.hpp"

// ##################################################
// ###               NAMESPACE TOBII              ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
 namespace Tobii {

	// ##################################################
	// ###                CLASS POINT2                ###
	// ##################################################
	
	/**
	 * \class Point2
     * \brief Describes 2 dimentional points with floating-point type.
     *
     * Point2 describes 2 dimentional points with floating-point type.
     * 
     **/
	class Point2
	{
	private:
		float _x; /*!< Coordinate on the X (1.0f, 0.0f) axis. */
		float _y; /*!< Coordinate on the Y (0.0f, 1.0f) axis. */

	public:
		/**
		 * \brief Constructor (Default)
		 * \param x (float) Position of the point along the X axis.
		 * \param y (float) Position of the point along the Y axis.
		 *
		 * Creates and initialize a 2 dimentional point with floating point coordinates.
		 *
		 **/
		Point2 ( float x = 0, float y = 0 );

		/**
		 * \brief Constructor (Copy)
		 * \param point (const Point2&) The Point2 to copy.
		 *
		 * Creates and initialize a copy of an object of type Point2.
		 *
		 **/
		Point2 ( const Point2& point );
		
		/**
		 * \brief Creates a 2 dimentional point with coordinates (0.0f, 0.0f).
		 * \return (Point2) The Point2 created.
		 *
		 * Creates Creates a 2 dimentional point with coordinates (0.0f, 0.0f).
		 *
		 **/
		static Point2 zero ();
		
		/**
		 * \brief Getter (float)
		 * \return (float) The X coordinate of this point.
		 *
		 * Return the X coordinate of this point.
		 *
		 **/
		float x () const;

		/**
		 * \brief Getter / Setter (float&)
		 * \return (float&) The reference of the X coordinates of this point.
		 *
		 * Return the The reference of this point X coordinate, which can be used to modify its value.
		 *
		 **/
		float& x ();

		/**
		 * \brief Getter (float)
		 * \return (float) The Y coordinate of this point.
		 *
		 * Return the Y coordinate of this point.
		 *
		 **/
		float y () const;

		/**
		 * \brief Getter / Setter (float&)
		 * \return (float&) The reference of the Y coordinates of this point.
		 *
		 * Return the The reference of this point Y coordinate, which can be used to modify its value.
		 *
		 **/
		float& y ();

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of 'this' Point2
		 *
		 * Creates and return a std::string representation of 'this' Point2.
		 * The representation is in JSON format:
		 * [x,y]
		 *
		 **/
		operator std::string () const;
		
		/**
		 * \brief Assignment operator (const Point2&)
		 * \param point (const Point2&) the Point2 to copy.
		 * \return reference to 'this' Point2 once set.
		 *
		 * Sets 'this' Point2 coordinates from a given Point2.
		 *
		 **/
		Point2& operator= ( const Point2& point );
	
	}; // END CLASS POINT2
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	std::ostream& operator<< ( std::ostream& ostream, const Point2& point );

	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // DATA_POINT2_HPP
// ##################################################