// ##################################################
// ###           TOBII_ORIENTATION_HPP            ###
// ##################################################
#ifndef TOBII_ORIENTATION_HPP
#define TOBII_ORIENTATION_HPP

/**
 * \file Orientation.hpp
 * \brief Describes Orientation object data type, modeling Tobii Glasses 2 orientation measurements.
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// INTERNAL MODULES
#include "Record.hpp"
#include "Vector3.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
 namespace Tobii {

	// ##################################################
	// ###              CLASS ORIENTATION             ###
	// ##################################################
		
	/**
	 * \class Orientation
     * \brief Describes Tobii Glasses 2 orientation measurements.
     *
     * This object represents measurements comming from the Tobii Glasses 2 guroscope.
     * The orientation is modeled using a 3D vector which values are in degre per seconds ( °/s ).
     * as specified in the Tobii SDK developer guide 1.0 in section D6.1.
     * 
     **/
	class Orientation : public Record, public Vector3
	{
	public:
		/**
		 * \brief Constructor (Default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at whiwh the recording was performed by the gyroscope.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param orientation (const Vector3&) The 3D orientation vector in degree per second.
		 *
		 * Creates and initialize an object of type Orienation with the given parameters.
		 *
		 **/
		Orientation ( Int64 timeStamp = 0, UInt8 status = 0, const Vector3& orientation = Vector3::zero() );


		/**
		 * \brief Constructor (Copy)
		 * \param orientation (const Orientation&) The Orientation record to copy.
		 *
		 * Creates and initialize a copy of an object of type Orientation.
		 *
		 **/
		Orientation ( const Orientation& orientation );

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of 'this' Orientation
		 *
		 * Creates and return a std::string representation of 'this' Orientation record.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"gy":[dx,dy,dz]}
		 *
		 **/
		operator std::string () const;

		/**
		 * \brief Assignment operator (const Vector3&)
		 * \param orientation (const Vector3&) the orientation value.
		 * \return (Orientation&) reference to 'this' Orientation once the orientation value as been set.
		 *
		 * Sets 'this' Orientation record value using a given Vector3.
		 *
		 **/
		Orientation& operator= ( const Vector3& orientation );

		/**
		 * \brief Assignment operator (const Orientation&)
		 * \param orientation (const Orientation&) the orientation record to copy.
		 * \return (Orientation&) reference to 'this' Orientation once set.
		 *
		 * Sets 'this' Orientation record time stamp, status and value using a given Orientation record.
		 *
		 **/
		Orientation& operator= ( const Orientation& orientation );
	
	}; // END CLASS ORIENTATION
	// ##################################################

	// ##################################################
	// ###      SHARED POINTER TYPE DEFINITION        ###
	// ##################################################

	/**
	 * \brief Shared pointer to an Orientation record.
	 *
	 * Defines the type of a shared pointer to an Orientation record.
	 *
	 **/
	using OrientationPtr = std::shared_ptr< Orientation >;

	// END SHARED POINTER TYPE DEFINITION
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	/**
	 * \brief Right-hand side output stream concatenation operator.
	 * \param stream (std::ostream&) stream where to concatenate the std::string representation of the Orientation record.
	 * \param orientation (const Orientation&) the Orientation record to represent
	 * \return (std::ostream&) The output stream once the Orientation record representation has been concatenated to it.
	 *
	 * Concatenate the std::string representation of the Orientation record in the output stream.
	 *
	 **/
	std::ostream& operator<< ( std::ostream& stream, const Orientation& orientation );

	/**
	 * \brief Left-hand side output stream concatenation operator.
	 * \param orientation (const Orientation&) the Orientation record to represent.
	 * \param stream (std::ostream&) stream where to concatenate the std::string representation of the Orientation record.
	 * \return (std::ostream&) The output stream once the Orientation record representation has been concatenated to it.
	 *
	 * Concatenate the std::string representation of the Orientation record in the output stream.
	 *
	 **/
	std::ostream& operator<< ( const Orientation& orientation, std::ostream& stream );
	
	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_ORIENTATION_HPP
// ##################################################