// ##################################################
// ###           TOBII_GAZEFIXATION_HPP           ###
// ##################################################
#ifndef TOBII_GAZEFIXATION_HPP
#define TOBII_GAZEFIXATION_HPP

/**
 * \file Eye.hpp
 * \brief Describes the gaze data as modeled by the Tobii Glasses 2
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STADARD LIBRARIES
#include <memory>

// INTERNAL MODULES
#include "PrimitifTypes.hpp"
#include "Point2.hpp"
#include "Record.hpp"
#include "Eye.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
 namespace Tobii {

	// ##################################################
	// ###              CLASS GAZE-FIXATION           ###
	// ##################################################

	/**
	 * \class GazeFixation
	 * \brief Describes Tobii Glasses 2 GazeFixation data type.
	 *
	 * This class describes GazeFixation data as represent by the Tobii Glasses 2.
	 *
	 **/
	class GazeFixation : public Record
	{
	public:
		/**
		 * \enum ValidityFlag
		 * \brief Flag used to check the gaze fields are set or not.
		 *
		 * This flag is used to specify which fields of the GazeFixation data structure as been set (by a data packet for example).
		 *
		 **/
		enum ValidityFlag : short
		{
			NONE        = 0x00, /*!< None of the GazeFixation fields are set.	*/
			POSITION_2D = 0x01, /*!< The 2D position field is set.				*/
			POSITION_3D = 0x02, /*!< The 3D position field is set.				*/
			DISTANCE    = 0x04, /*!< The distance field is set.					*/
			ALL         = 0x07  /*!< All of the above fields are set. This doesn't mean that each eyes are set ! */
		};

	private:
		Eye				_leftEye;		/*!< Left eye data.		*/
		Eye				_rightEye;		/*!< Right eye data.	*/
		Point2			_position2D;	/*!< 2D position of the fixation on the camera plan.	*/
		Point3			_position3D;	/*!< 3D position of the fixation in the camera space.	*/
		float			_distance;		/*!< Distance of the 3D fixation point to the camera.	*/
		ValidityFlag	_validityFlag;	/*!< Validity flag to track validity of the above fields (except the eyes).	*/

	public:
		/**
		 * \brief Constructor (Default)
		 * \param timeStamp (Int64) time stamp of this GazeFixation Record
		 * \param status (UInt8) status of this GazeFixation Record
		 * 
		 * Initialize a GazeFixation with the given parameter and set its validity flag to GazeFixation::ValidityFlag::NONE.
		 *
		 **/
		GazeFixation ( Int64 timeStamp = 0, Int64 status = 0 );

		/**
		 * \brief Constructor (Full)
		 * \param timeStamp (Int64) time stamp at which the gaze fixation was recorded by the Tobii Glasses 2.
		 * \param status (UInt8) status of the record, 0 means ok.
		 * \param left (const Eye&) The left Eye data.
		 * \param right (const Eye&) The right Eye data.
		 * \param position2D (const Point2&) The gaze fixation position on the camera plan.
		 * \param position3D (const Point3&) The gaze fixation position in the camera space (used to initialize the distance).
		 * 
		 * Initialize anGazeFixation, setting all its fields to the given values, and set its validity flag to Eye::ValidityFlag::ALL.
		 *
		 **/
		GazeFixation ( Int64 timeStamp, Int64 status, const Eye& left, const Eye& right, const Point2& position2D, const Point3& position3D );

		/**
		 * \brief Constructor (Copy)
		 * \param gaze (const GazeFixation&) GazeFixation to copy.
		 * 
		 * Initialize an GazeFixation, copying all fields and validity flag from the given GazeFixation.
		 *
		 **/
		GazeFixation ( const GazeFixation& gaze );

		/**
		 * \brief Getter (const Eye&)
		 * \return (const Eye&) The constant reference to the left eye data.
		 *
		 * Return the constant reference to the left Eye data of this GazeFixation.
		 **/
		const Eye& leftEye () const;

		/**
		 * \brief Getter / Setter (const Eye&)
		 * \return (Eye&) The reference to the left eye data.
		 *
		 * Return the reference to the left Eye data of this GazeFixation, which can be later modified
		 **/
		Eye& leftEye ();

		/**
		 * \brief Getter (const Eye&)
		 * \return (const Eye&) The constant reference to the right eye data.
		 *
		 * Return the constant reference to the right Eye data of this GazeFixation.
		 **/
		const Eye& rightEye () const;

		/**
		 * \brief Getter / Setter (const Eye&)
		 * \return (Eye&) The reference to the right eye data.
		 *
		 * Return the reference to the right Eye data of this GazeFixation, which can be later modified
		 **/
		Eye& rightEye ();

		/**
		 * \brief Getter (const Point2&)
		 * \return (const Point2&) The constant reference to the 2D position of this GazeFixation on the camera plan.
		 *
		 * Return the constant reference to the 2D position of this GazeFixation on the camera plan,
		 * between (0,0) for top-left corner of the frame,
		 * and between (1,1) for bottom-right corner of the frame,
		 **/
		const Point2& position2D () const;
		
		/**
		 * \brief Getter / Setter (Point2&)
		 * \return (Point2&) The reference to the 2D position of this GazeFixation on the camera plan.
		 *
		 * Return the reference to the 2D position of this GazeFixation on the camera plan, whiwh can later be used to modify it.
		 * between (0,0) for top-left corner of the frame,
		 * and between (1,1) for bottom-right corner of the frame,
		 **/
		Point2& position2D ();

		/**
		 * \brief Getter (const Point3&)
		 * \return (const Point3&) The constant reference to the 3D position of this GazeFixation in the camera space.
		 *
		 * Return the constant reference to the 3D position of this GazeFixation in the camera space.
		 **/
		const Point3& position3D () const;

		/**
		 * \brief Getter (Point3&)
		 * \return (Point3&) The reference to the 3D position of this GazeFixation in the camera space.
		 *
		 * Return the reference to the 3D position of this GazeFixation in the camera space, whiwh can later be used to modify it.
		 **/
		Point3& position3D ();

		/**
		 * \brief Getter (float)
		 * \return (float) The distancen of this GazeFixation 3D position to the camera origine.
		 *
		 * Return The distancen of this GazeFixation 3D position to the camera origine.
		 **/
		float distance () const;

		/**
		 * \brief Getter (float)
		 * \return (float&) The reference of the distancen of this GazeFixation 3D position to the camera origine.
		 *
		 * Return The distancen of this GazeFixation 3D position to the camera origine.
		 **/
		float& distance ();
		
		/**
		 * \brief Getter (ValidityFlag)
		 * \return (ValidityFlag) The ValidityFlag of this GazeFixation.
		 *
		 * Return The ValidityFlag of this GazeFixation.
		 **/
		ValidityFlag validityFlag () const;

		/**
		 * \brief Getter / Setter (ValidityFlag&)
		 * \return (ValidityFlag&) The reference to the ValidityFlag of this GazeFixation.
		 *
		 * Return The reference to the ValidityFlag of this GazeFixation, which can later be used to modify it.
		 **/
		ValidityFlag& validityFlag ();

		/**
		 * \brief Getter (bool)
		 * \param field (ValidityFlag) ValidityFlag to check for.
		 * \return (bool) True if the given validity flag is set, False if not.
		 * 
		 * Returns True if the given validity flag is set, False if not.
		 * It is possible to ask for one or more flag, specifying the field parameter as: "FLAG_1 | FLAG_2 | FLAG_3 | ...".
		 *
		 **/
		bool isFieldSet ( ValidityFlag field ) const;

		/**
		 * \brief Getter (bool)
		 * \return (bool) True if all the fieds of the eye are set, including the ones of each eyes, false otherwise.
		 * \see isFieldSet
		 * 
		 * Returns True if the all the ValidityFlag of this GazeFixation are set and if all the Eye::ValidityFlag of both eyes are set, False if not.
		 * This is not the same as isFieldSet( GazeFixation::ValidityFlag::ALL );
		 *
		 **/
		bool isSet () const;
		
		/**
		 * \brief Getter (bool)
		 * \return (bool) True if this GazeFixation datas are sufficient to compute the GazeFixation distance, False if not.
		 * 
		 * Return bool True if this GazeFixation datas are sufficient to compute the GazeFixation distance, False if not:
		 * The flags Eye::ValidityFlag::PUPIL_CENTER and Eye::ValidityFlag::GAZE_DIRECTION must be set and the gaze directions must intersect.
		 *
		 **/
		bool isDistanceComputable () const;

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) String representation of this GazeFixation
		 * \see Eye::operator(str::string)
		 *
		 * Create and returns a string representation of this GazeFixation.
		 * The representation is in JSON format:
		 * {"ts":Int64,"s":UInt8,"left":[left-eye-string],"right":[right-eye-string],"gp":[x,y],"gp3":[x,y,z]}
		 *
		 **/
		operator std::string () const;

		/**
		 * \brief Getter / Setter (bool)
		 * \param threshold (float) Distance threshold at which gaze direction are considered to be crossing.
		 * \return (float) the distance of the 3D fication point.
		 * 
		 * Computes the distance of the 3D fixation point, update it in this object, and returns it.
		 *
		 **/
		float computeDistance ( float threshold = 0.3 );

		/**
		 * \brief Setter (ValidityFlag)
		 * \param field (ValidityFlag) The ValidityFlag to set
		 * 
		 * Set the given ValidityFlag.
		 *
		 **/
		void setFieldValidity ( ValidityFlag field );

		/**
		 * \brief Setter (const Point2&)
		 * \param position2D (const Point2&) The 2D gaze position on the camera plan
		 * 
		 * Set the 2D gaze position on the camera plan and sets the associated ValidityFlag.
		 *
		 **/
		void setPosition2D ( const Point2& position2D );

		/**
		 * \brief Setter (const Point3&)
		 * \param position3D (const Point3&) The 2D gaze position in the camera space
		 * 
		 * Set the 3D gaze position in the camera space and sets the associated ValidityFlag.
		 *
		 **/
		void setPosition3D ( const Point3& position3D );

		/**
		 * \brief Setter (float)
		 * \param distance (float) The distance of the 3D gaze fixation in the camera space
		 * 
		 * Set the distance to the 3D gaze position in the camera space and sets the associated ValidityFlag.
		 *
		 **/
		void setDistance ( float distance );

		/**
		 * \brief Assignment operator (const GazeFixation&)
		 * \param gaze (const GazeFixation&) The GazeFixation to copy.
		 * \return (GazeFixation&) Reference to this GazeFixation once set.
		 *
		 * Sets this GazeFixation fields and ValidityFlag.
		 *
		 **/
		GazeFixation& operator= ( const GazeFixation& gaze );
	
	}; // END CLASS GAZE-FIXATION
	// ##################################################

	// ##################################################
	// ###      SHARED POINTER TYPE DEFINITION        ###
	// ##################################################
	
	/**
	 * \brief Shared pointer to an GazeFixation record.
	 *
	 * Defines the type of a shared pointer to an GazeFixation record.
	 *
	 **/
	using GazeFixationPtr = std::shared_ptr< GazeFixation >;

	// END SHARED POINTER TYPE DEFINITION
	// ##################################################

	// ##################################################
	// ###            LOGICAL OPERATORS               ###
	// ##################################################
	
	/**
	 * \brief Logical-OR operator (GazeFixation::ValidityFlag)
	 * \param flag1 (GazeFixation::ValidityFlag) left had side flag of the logical OR.
	 * \param flag2 (GazeFixation::ValidityFlag) right had side flag of the logical OR.
	 * \return (GazeFixation::ValidityFlag) ValidityFlag that describes the validity of both given flags.
	 *
	 * Performs and returns logical-OR of the two given flags, preserving the type of the result.
	 * Use this operator to check for mulitple flag at once.
	 * 
	 **/
	GazeFixation::ValidityFlag operator| ( GazeFixation::ValidityFlag flag1, GazeFixation::ValidityFlag flag2 );

	// END LOGICAL OPERATORS
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	std::ostream& operator<< ( std::ostream& ostream, const GazeFixation::ValidityFlag flag );
	std::ostream& operator<< ( std::ostream& ostream, const GazeFixation& gaze );

	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_GAZEFIXATION_HPP
// ##################################################