// ##################################################
// ###           TOBII_GAZEFIXATION_CPP           ###
// ##################################################
#ifndef TOBII_GAZEFIXATION_CPP
#define TOBII_GAZEFIXATION_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "GazeFixation.hpp"
#include <iostream>

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################
namespace Tobii {

	// ##################################################
	// ###              CLASS GAZEFIXATION            ###
	// ##################################################

	GazeFixation::GazeFixation ( Int64 timeStamp, Int64 status )
	: Record( timeStamp, status ), _validityFlag( ValidityFlag::NONE )
	{}

	GazeFixation::GazeFixation ( Int64 timeStamp, Int64 status, const Eye& left, const Eye& right, const Point2& position2D, const Point3& position3D )
	: Record( timeStamp, status ), _leftEye( left ), _rightEye( right ), _position2D( position2D ), _position3D( position3D ), _distance( position3D.norm() ), _validityFlag( ValidityFlag::ALL )
	{}

	GazeFixation::GazeFixation ( const GazeFixation& gaze )
	: Record( gaze ), _leftEye( gaze.leftEye() ), _rightEye( gaze.rightEye() ), _position2D( gaze.position2D() ), _position3D( gaze.position3D() ), _distance( gaze.distance() ), _validityFlag( gaze.validityFlag() )
	{}

	const Eye&					GazeFixation::leftEye		() const	{ return this->_leftEye;		}
	Eye&						GazeFixation::leftEye		()			{ return this->_leftEye;		}
	const Eye&					GazeFixation::rightEye		() const	{ return this->_rightEye;		}
	Eye&						GazeFixation::rightEye		()			{ return this->_rightEye;		}
	const Point2&				GazeFixation::position2D	() const	{ return this->_position2D;		}
	Point2&						GazeFixation::position2D	()			{ return this->_position2D;		}
	const Point3&				GazeFixation::position3D	() const	{ return this->_position3D;		}
	Point3&						GazeFixation::position3D	()			{ return this->_position3D;		}
	float						GazeFixation::distance	 	() const	{ return this->_distance;		}
	float&						GazeFixation::distance	 	()			{ return this->_distance;		}
	GazeFixation::ValidityFlag	GazeFixation::validityFlag	() const 	{ return this->_validityFlag;	}
	GazeFixation::ValidityFlag&	GazeFixation::validityFlag	()		 	{ return this->_validityFlag;	}

	bool GazeFixation::isFieldSet ( ValidityFlag field ) const
	{
		return ( this->validityFlag() & field );
	}

	bool GazeFixation::isSet() const
	{
		return ( this->isFieldSet(ValidityFlag::ALL) && this->leftEye().isSet() && this->rightEye().isSet() );
	}

	bool GazeFixation::isDistanceComputable () const
	{
		if (  this->leftEye().isFieldSet( Eye::ValidityFlag::PUPIL_CENTER | Eye::ValidityFlag::GAZE_DIRECTION )
		&&   this->rightEye().isFieldSet( Eye::ValidityFlag::PUPIL_CENTER | Eye::ValidityFlag::GAZE_DIRECTION ) )
		{
			return ( ( this->leftEye().gazeDirection().dz() * this->rightEye().gazeDirection().dx() - this->leftEye().gazeDirection().dx() * this->rightEye().gazeDirection().dz() ) != 0 );
		}
		return false;
	}

	GazeFixation::operator std::string () const
	{
		std::stringstream formatedOutput;
		formatedOutput
			<< "{\"ts\":"		<< this->timeStamp()
			<< ",\"s\":"		<< this->status()
			<< ",\"left\":"		<< this->leftEye()
			<< ",\"right\":"	<< this->rightEye()
			<< ",\"gp\":"		<< this->position2D()
			<< ",\"gp3\":"		<< this->position3D()
			<< ",\"d\":"		<< this->distance()
			<< ",\"vf\":"		<< this->validityFlag()
			<< "}";

		return formatedOutput.str();
	}

	float GazeFixation::computeDistance ( float threshold )
	{
		// Read left eye
		Point3  lpc = this->leftEye().pupilCenter();
		Vector3 lgd = this->leftEye().gazeDirection();

		// Read right eye
		Point3  rpc = this->rightEye().pupilCenter();
		Vector3 rgd = this->rightEye().gazeDirection();

		// Determinant:
		float det( lgd.dz() * rgd.dx() - lgd.dx() * rgd.dz() );

		if ( det != 0.0f )
		{
			// Parametre des deux droites
			float tLeft  = ( -rgd.dz() * (rpc.x() - lpc.x()) + rgd.dx() * (rpc.z() - lpc.z()) ) / det;
			float tRight = ( -lgd.dz() * (rpc.x() - lpc.x()) + lgd.dx() * (rpc.z() - lpc.z()) ) / det;
		
			// Verification des parametres
			float lFixY = lpc.y() + tLeft  * lgd.dy();
			float rFixY = rpc.y() + tRight * rgd.dy();

			if ( std::abs(rFixY - lFixY) / std::max(rFixY, lFixY) >= -threshold
			&&   std::abs(rFixY - lFixY) / std::max(rFixY, lFixY) <=  threshold )
			{
				Point3 fixation(
					lpc.x() + tLeft * lgd.dx(),
					lpc.y() + tLeft * lgd.dy(),
					lpc.z() + tLeft * lgd.dy()
				);
				this->setDistance( fixation.norm() );
				return this->distance();
			}
		}
		return -1.0f;
	}

	void GazeFixation::setFieldValidity ( ValidityFlag field )
	{
		this->validityFlag() = this->validityFlag() | field;
	}

	void GazeFixation::setPosition2D ( const Point2& position2D )
	{
		this->position2D() = position2D;
		this->setFieldValidity( ValidityFlag::POSITION_2D );
	}

	void GazeFixation::setPosition3D ( const Point3& position3D )
	{
		this->position3D() = position3D;
		this->distance()   = position3D.norm();
		this->setFieldValidity( ValidityFlag::POSITION_3D | ValidityFlag::DISTANCE );
	}

	void GazeFixation::setDistance ( float distance )
	{
		this->distance() = distance;
		this->setFieldValidity( ValidityFlag::DISTANCE );
	}


	GazeFixation& GazeFixation::operator= ( const GazeFixation& gaze )
	{
		this->Record::operator=( gaze );
		this->leftEye()		 = gaze.leftEye();
		this->rightEye()	 = gaze.rightEye();
		this->position2D()	 = gaze.position2D();
		this->position3D()	 = gaze.position3D();
		this->distance()     = gaze.distance();
		this->validityFlag() = gaze.validityFlag();
		return (*this);
	}

	// END CLASS GAZEFIXATION
	// ##################################################

	// ##################################################
	// ###            LOGICAL OPERATORS               ###
	// ##################################################
	
	GazeFixation::ValidityFlag operator| ( GazeFixation::ValidityFlag flag1, GazeFixation::ValidityFlag flag2 )
	{
		return static_cast< GazeFixation::ValidityFlag >( static_cast< short >(flag1) | static_cast< short >(flag2) );
	}

	// END LOGICAL OPERATORS
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	std::ostream& operator<< ( std::ostream& ostream, const GazeFixation::ValidityFlag flag )
	{
		return ostream << static_cast< short >( flag );
	}
	
	std::ostream& operator<< ( std::ostream& ostream, const GazeFixation& gaze )
	{
		return ostream << static_cast< std::string >( gaze );
	}
	
	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_GAZEFIXATION_CPP
// ##################################################