// ##################################################
// ###               TOBII_EYE_HPP                ###
// ##################################################
#ifndef TOBII_EYE_HPP
#define TOBII_EYE_HPP

/**
 * \file Eye.hpp
 * \brief Describes the gaze data fields specific to one eye
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// INTERNAL MODULES
#include "Point3.hpp"
#include "Vector3.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
 namespace Tobii {

	// ##################################################
	// ###                  CLASS EYE                 ###
	// ##################################################
	
	/**
	 * \class Eye
	 * \brief Describes Tobii Glasses 2 Eye data type.
	 *
	 * This class regroups gaze fixation data fields that are specific to one eye.
	 *
	 **/
	class Eye
	{
	public:
		/**
		 * \enum ValidityFlag
		 * \brief Flag used to check the eye's fields are set or else, if the data has just been initialized to 0.
		 *
		 * This flag is used to specify which fields of the Eye data structure as been set (by a data packet for example).
		 *
		 **/
		enum ValidityFlag : short
		{
			NONE            = 0x00, /*!< None of the Eye's fields are set.	*/
			PUPIL_CENTER    = 0x01, /*!< The pupilCenter field is set.		*/
			PUPIL_DIAMETER  = 0x02, /*!< The pupilDiameter field is set.	*/
			GAZE_DIRECTION  = 0x04, /*!< The gazeDirection field is set.	*/
			ALL             = 0x07, /*!< All the fields are set.			*/
		};

	private:
		Point3		 _pupilCenter;		/*!< Pupil center of the eye.	*/
		float		 _pupilDiameter;	/*!< Pupil diameter of the eye.	*/	
		Vector3		 _gazeDirection;	/*!< Gaze direction of the eye.	*/
		ValidityFlag _validityFlag;		/*!< Validy flag of the eye.	*/

	public:
		/**
		 * \brief Constructor (Default)
		 * 
		 * Initialize an Eye with all fields to 0 and set its validity flag to Eye::ValidityFlag::NONE.
		 *
		 **/
		Eye ();

		/**
		 * \brief Constructor (Full)
		 * \param pupilCenter (const Point3&) pupil center of the eye.
		 * \param pupilDiameter (float) pupil diameter of the eye.
		 * \param gazeDirection (const Vector3&) gaze direction of the eye.
		 * 
		 * Initialize an Eye, setting all its fields to the given values, and set its validity flag to Eye::ValidityFlag::ALL.
		 *
		 **/
		Eye ( const Point3& pupilCenter, float pupilDiameter, const Vector3& gazeDirection );

		/**
		 * \brief Constructor (Copy)
		 * \param eye (const Eye&) Eye to copy.
		 * 
		 * Initialize an Eye, copying all fields and validity flag from the given eye.
		 *
		 **/
		Eye ( const Eye& eye );

		/**
		 * \brief Getter (const Point3&)
		 * \return (const Point3&) Constant reference of the eye's pupil center.
		 * 
		 * Returns a constant reference of the pupil center of this eye.
		 *
		 **/
		const Point3& pupilCenter () const;

		/**
		 * \brief Getter / Setter (Point3&)
		 * \return (Point3&) Reference of the eye's pupil center.
		 * 
		 * Returns a reference of the pupil center of this eye, which can be later used to modified the pupil center on x, y and y.
		 * To directly assign the whole pupil center see "setPupilCenter" which also sets the validity flag associated to this field.
		 *
		 **/
		Point3& pupilCenter ();

		/**
		 * \brief Getter (float)
		 * \return (float) Constant reference of the eye's pupil diameter.
		 * 
		 * Returns a constant reference of the pupil diameter of this eye.
		 *
		 **/
		float pupilDiameter	() const;

		/**
		 * \brief Getter / Setter (float&)
		 * \return (float&) Reference of the eye's pupil diameter.
		 * \see setPupilDiameter
		 *
		 * Returns a reference of the pupil diameter of this eye, which can be later used to modified the pupil diameter.
		 * To directly assign the whole pupil diameter see "Eye::setPupilDiameter" which also sets the validity flag associated to this field.
		 *
		 **/
		float& pupilDiameter ();

		/**
		 * \brief Getter (const Vector3&)
		 * \return (const Vector3&) Constant reference of the eye's gaze direction.
		 * 
		 * Returns a constant reference of the gaze direction of this eye.
		 *
		 **/
		const Vector3& gazeDirection () const;

		/**
		 * \brief Getter / Setter (Point3&)
		 * \return (Vector3&) Reference of the eye's gaze direction.
		 * \see setGazeDirection
		 *
		 * Returns a reference of the gaze direction of this eye, which can be later used to modified the gaze direction on x, y and y.
		 * To directly assign the whole gaze direction see "Eye::setGazeDirection" which also sets the validity flag associated to this field.
		 *
		 **/
		Vector3& gazeDirection ();

		/**
		 * \brief Getter (ValidityFlag)
		 * \return (ValidityFlag) Validy flag of this eye.
		 * \see isFieldSet
		 *
		 * Returns the eye validity flag of this eye.
		 * To check whether a fiel is set or not, see "bool Eye::isFieldSet() const"
		 *
		 **/
		ValidityFlag validityFlag () const;

		/**
		 * \brief Getter (ValidityFlag)
		 * \return (ValidityFlag) Validy flag of this eye.
		 * \see isFieldSet
		 *
		 * Returns the eye validity flag of this eye.
		 * To check whether a fiel is set or not, see "bool Eye::isFieldSet() const"
		 *
		 **/
		ValidityFlag& validityFlag ();

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) String representation of this Eye
		 *
		 * Create and returns a string representation of this Eye.
		 * The representation is in JSON format:
		 * {"pc":[x,y,z],pd":float,"gd":[dx,dy,dz]}
		 *
		 **/
		operator std::string () const;

		/**
		 * \brief Getter (bool)
		 * \param field (ValidityFlag) ValidityFlag to check for.
		 * \return (bool) True if the given ValidityFlag is set, False if not.
		 * 
		 * Returns True if the given ValidityFlag is set, False if not.
		 * It is possible to ask for one or more flag, specifying the field parameter as: "FLAG_1 | FLAG_2 | FLAG_3 | ...".
		 *
		 **/
		bool isFieldSet	( ValidityFlag field ) const;

		/**
		 * \brief Getter (bool)
		 * \return (bool) True if all the fieds of the eye are set, false otherwise.
		 * \see isFieldSet
		 * 
		 * Returns True if the all the validity flag of this eye are set, False if not.
		 * Same as isFieldSet( Eye::ValidityFlag::ALL );
		 *
		 **/
		bool isSet () const;

		/**
		 * \brief Setter (ValidityFlag)
		 * \param field (ValidityFlag) set validity flag(s) to set.
		 * 
		 * Sets one or more validity flags.
		 *
		 **/
		void setFieldValidity ( ValidityFlag field );

		/**
		 * \brief Setter (const Point3&)
		 * \param pupilCenter (const Point3&) The pupil center.
		 *
		 * Sets the value of this eye's pupil center and sets the associated validityFlag.
		 *
		 **/
		void setPupilCenter ( const Point3& pupilCenter );

		/**
		 * \brief Setter (float)
		 * \param diameter (float) The pupil diameter
		 *
		 * Sets the value of this eye's pupil diameter and sets the associated validityFlag.
		 *
		 **/
		void setPupilDiameter ( float diameter );

		/**
		 * \brief Setter (const Vector3&)
		 * \param gazeDirection (const Vector3&) The gaze direction.
		 *
		 * Sets the value of this eye's gaze direction and sets the associated validityFlag.
		 *
		 **/
		void setGazeDirection ( const Vector3& gazeDirection );

		/**
		 * \brief Assignment operator (const Eye&)
		 * \param eye (const Eye&) The Eye to copy.
		 * \return (Eye&) Reference to this Eye once set.
		 *
		 * Sets this Eye pupil center, pupil diameter, gaze direction, and validity field, copying the ones of the given Eye.
		 *
		 **/
		Eye& operator= ( const Eye& eye );

	}; // END CLASS EYE
	// ##################################################

	// ##################################################
	// ###            LOGICAL OPERATORS               ###
	// ##################################################

	/**
	 * \brief Logical-OR operator (Eye::ValidityFlag)
	 * \param flag1 (Eye::ValidityFlag) left had side flag of the logical OR.
	 * \param flag2 (Eye::ValidityFlag) right had side flag of the logical OR.
	 * \return (Eye::ValidityFlag) validity flag that describes the validity of both given flags.
	 *
	 * Performs and returns logical-OR of the two given flags, preserving the type of the result.
	 * Use this operator to check for mulitple flag at once.
	 * 
	 **/
	Eye::ValidityFlag operator| ( Eye::ValidityFlag flag1, const Eye::ValidityFlag flag2 );

	// END LOGICAL OPERATORS
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	std::ostream& operator<< ( std::ostream& ostream, Eye::ValidityFlag validityFlag );
	std::ostream& operator<< ( std::ostream& ostream, const Eye& eye );

	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_EYE_HPP
// ##################################################