// ##################################################
// ###           TOBII_DATAPACKETS_HPP            ###
// ##################################################
#ifndef TOBII_DATAPACKETS_HPP
#define TOBII_DATAPACKETS_HPP

/**
 * \file DataPackets.hpp
 * \brief Describes all the data packets objects type, modeling all Tobii Glasses 2 live data stream contents .
 * \author PÉREZ DE SAN ROMAN Philippe
 * \date 16/02/2016
 * \version 1.0
 **/

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

// STADARD LIBRARIES
#include <memory>

// INTERNAL MODULES
#include "PrimitifTypes.hpp"
#include "Point2.hpp"
#include "Point3.hpp"
#include "Vector3.hpp"
#include "Record.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

/**
 * \namespace Tobii
 * 
 * Regroups all Tobii Glasses 2 SDK related types and algorithmes.
 **/
namespace Tobii {

	// ##################################################
	// ###             CLASS DATA-PACKET              ###
	// ##################################################

	/**
	 * \class DataPacket
	 * \brief Describes Tobii Glasses 2 live data packets interface.
	 *
	 * This class is an interface to all data packets that can be received from Tobii Glasses 2 through the livedata stream.
	 * See Tobii Glasses 2 SDK Developer guide 1.0, in section D6 for an exaustive list.
	 *
	 **/
	class DataPacket : public Record
	{
	private:
		std::string _name; /*!< name of the data packet. This is usualy the string found in the JSON document before the value of the packet */

	public:
		/**
		 * \brief Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param name (std::string) The name of the data packet
		 *
		 * Creates and initialize an object of type DataPacket, setting its time stamp, status, and name.
		 *
		 **/
		DataPacket ( Int64 timeStamp = 0, UInt8 status = 0, const std::string& name = "" );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const DataPacket&) The DataPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type DataPacket.
		 *
		 **/
		DataPacket ( const DataPacket& gazePacket );

		/**
		 * \brief Destrutor (virtual)
		 *
		 * Does nothing but to allow for safe public inheritance to this class.
		 *
		 **/
		virtual ~DataPacket ();

		/**
		 * \brief Getter (const std::string&)
		 * \return (const std::string&) the const reference to the name of this DataPacket
		 *
		 * Returns a constant reference to the name of this DataPacket
		 *
		 */ 
		const std::string& name () const;

		/**
		 * \brief Getter / Setter (std::string&)
		 * \return (std::string&) the reference to the name of this DataPacket
		 *
		 * Returns a reference to the name of the DataPacket, which can later be used to modify this name.
		 *
		 */ 
		std::string& name ();

		/**
		 * \brief Cast operator (std::string) [pure vitual]
		 * \return (std::string) representation of this DataPacket
		 *
		 * Every child class implements this method, to creates a std::string representation of this DataPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"name":value}
		 *
		 **/
		virtual operator std::string () const = 0;
		
		/**
		 * \brief Assignment operator (const DataPacket&)
		 * \param gazePacket (const DataPacket&) the DataPacket to copy.
		 * \return (DataPacket&) reference to this DataPacket once set.
		 *
		 * Sets this DataPacket record time stamp, status and name using a given DataPacket fields.
		 *
		 **/
		DataPacket& operator= ( const DataPacket& gazePacket );
	
	}; // END CLASS GAZE-PACKET
	// ##################################################

	// ##################################################
	// ###            CLASS EYE-DATA-PACKET           ###
	// ##################################################

	/**
	 * \class EyeDataPacket
	 * \brief Describes Tobii Glasses 2 live data packets interface for packets specific to one eye.
	 *
	 * The eye data packets describe the data one eye pupil center with respect to the camera origine in millimeters.
	 *
	 **/
	class EyeDataPacket : public DataPacket
	{
	public:
		/**
		 * \enum Eye
		 * \brief Flag used to specified which eye the packets data comes from, if it applies to the type of data.
		 *
		 * Some DataPacket are specific to one eye, this flag is used to specify which eye is concerned by this DataPacket.
		 * This is usualy specified in the JSON document as: {"eye":"left"} or {"eye":"right"} 
		 *
		 **/
		enum Eye : short
		{
			NONE  = 0x00, /*!< The DataPacket does not concern any eyes	*/
			LEFT  = 0x01, /*!< The DataPacket concerns the left eye		*/
			RIGHT = 0x02, /*!< The DataPacket concerns the right eye	*/
			BOTH  = 0x04  /*!< The DataPacket concerns both eyes		*/
		};
		
	private:
		Eye _eye; /*!< Specifies which eye's pupil center this packet describes */

	public:
		/**
		 * \brief Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param name (std::string) The name of the data packet
		 * \param eye (Eye) The eye flag that this packets describes
		 *
		 * Creates and initialize an object of type EyeDataPacket, setting its time stamp, status, name, and eye.
		 *
		 **/
		EyeDataPacket ( Int64 timeStamp = 0, UInt8 status = 0, const std::string& name = "", Eye eye = Eye::NONE );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const EyeDataPacket&) The EyeDataPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type EyeDataPacket.
		 *
		 **/
		EyeDataPacket ( const EyeDataPacket& gazePacket );

		/**
		 * \brief Getter (Eye)
		 * \return (Eye) The eye that this packet describes.
		 *
		 * Returns which eye is associated to this data packet.
		 *
		 **/
		Eye eye () const;

		/**
		 * \brief Getter / Setter (Eye&)
		 * \return (Eye&) The reference off the eye that this packet describes.
		 *
		 * Returns the reference off the eye associated to this data packet, which can later be modified.
		 *
		 **/
		Eye& eye ();

		/**
		 * \brief Cast operator (std::string) [pure vitual]
		 * \return (std::string) representation of this EyeDataPacket
		 *
		 * Every child class implements this method, to creates a std::string representation of this EyeDataPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"eye":eye,name":value}
		 *
		 **/
		virtual operator std::string () const = 0;

		/**
		 * \brief Assignment operator (const EyeDataPacket&)
		 * \param gazePacket (const EyeDataPacket&) the DataPacket to copy.
		 * \return (EyeDataPacket&) reference to this EyeDataPacket once set.
		 *
		 * Sets this EyeDataPacket record time stamp, status, name and eye, using a given EyeDataPacket.
		 *
		 **/
		EyeDataPacket& operator= ( const EyeDataPacket& gazePacket );
	
	}; // END CLASS EYE-DATA-PACKET
	// ##################################################

	// ##################################################
	// ###          CLASS PUPIL-CENTER-PACKET         ###
	// ##################################################

	/**
	 * \class PupilCenterPacket
	 * \brief Describes Tobii Glasses 2 live pupil center data packets.
	 *
	 * The pupil center packets describe the 3D position of one eye pupil center with respect to the camera origine in millimeters.
	 *
	 **/
	class PupilCenterPacket : public EyeDataPacket, public Point3
	{
	public:
		/**
		 * \brief Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param pupilCenter (const Point3&) The pupil center position
		 * \param eye (Eye) The eye flag that this packets describes
		 *
		 * Creates and initialize an object of type PupilCenterPacket, setting its time stamp, status, pupil center position, and eye.
		 *
		 **/
		PupilCenterPacket ( Int64 timeStamp = 0, UInt8 status = 0, const Point3& pupilCenter = Point3::zero(), Eye eye = Eye::NONE );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const PupilCenterPacket&) The PupilCenterPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type PupilCenterPacket.
		 *
		 **/
		PupilCenterPacket ( const PupilCenterPacket& gazePacket );

		/**
		 * \brief Getter (const Point3&)
		 * \return (const Point3&) The pupil center coordinates of this packet.
		 *
		 * Returns The pupil center coordinates of this packet.
		 *
		 **/
		const Point3& pupilCenter () const;

		/**
		 * \brief Getter / Setter (Point3&)
		 * \return (Point3&) The pupil center coordinates of this packet.
		 *
		 * Returns The reference to the pupil center of this packet, whiwh can later be used to modifed its coordinates.
		 *
		 **/
		Point3& pupilCenter ();

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string )representation of this DataPacket
		 *
		 * Creates a std::string representation of this DataPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"pc":[x,y,z]}
		 *
		 **/
		virtual operator std::string () const;

		/**
		 * \brief Assignment operator (const Point3&)
		 * \param center (const Point3&) the pupil center position
		 * \return (PupilCenterPacket&) reference to this PupilCenterPacket once set.
		 *
		 * Sets this PupilCenterPacket position.
		 *
		 **/
		PupilCenterPacket& operator= ( const Point3& center );

		/**
		 * \brief Assignment operator (const PupilCenterPacket&)
		 * \param gazePacket (const PupilCenterPacket&) the DataPacket to copy.
		 * \return (PupilCenterPacket&) reference to this PupilCenterPacket once set.
		 *
		 * Sets this PupilCenterPacket record time stamp, status, name and pupil center, using a given PupilCenterPacket.
		 *
		 **/
		PupilCenterPacket& operator= ( const PupilCenterPacket& gazePacket );
	
	}; // END CLASS PUPIL-CENTER-PACKET
	// ##################################################

	// ##################################################
	// ###         CLASS PUPIL-DIAMETER-PACKET        ###
	// ##################################################

	/**
	 * \class PupilDiameterPacket
	 * \brief Describes Tobii Glasses 2 live pupil diameter data packets.
	 *
	 * The pupil diameter packets describe the diameter of one eye pupil in millimeters.
	 *
	 **/
	class PupilDiameterPacket : public EyeDataPacket
	{
	private:
		float _pupilDiameter;	/*!< diameter in millimeter of one eye */

	public:
		/**
		 * \brief Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param pupilDiameter (float) the pupil diameter in millimeters
		 * \param eye (Eye) The eye flag that this packets describes
		 *
		 * Creates and initialize an object of type PupilDiameterPacket, setting its time stamp, status, pupil diameter and eye
		 *
		 **/
		PupilDiameterPacket ( Int64 timeStamp = 0, UInt8 status = 0, float pupilDiameter = 0, Eye eye = Eye::NONE );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const PupilDiameterPacket&) The PupilDiameterPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type PupilDiameterPacket.
		 *
		 **/
		PupilDiameterPacket ( const PupilDiameterPacket& gazePacket );

		/**
		 * \brief Getter (float)
		 * \return (float) the pupil diameter value of this packet.
		 *
		 * Returns the pupil diameter value of this packet.
		 *
		 **/
		float pupilDiameter () const;

		/**
		 * \brief Getter / Setter (float&)
		 * \return (float&) The reference off the pupil diameter of this packet.
		 *
		 * Returns the reference off the pupil diameter of this packet, which can later be modified.
		 *
		 **/
		float& pupilDiameter ();

		/**
		 * \brief Cast operator (float)
		 * \return (float) the pupil diameter value of this packet.
		 *
		 * Returns the pupil diameter value of this packet.
		 *
		 **/
		virtual operator float () const;

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of this PupilDiameterPacket
		 *
		 * Creates a std::string representation of this PupilDiameterPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"pd":value}
		 *
		 **/
		virtual operator std::string () const;

		/**
		 * \brief Assignment operator (float)
		 * \param pupilDiameter (float) the pupil diameter
		 * \return (PupilDiameterPacket&) reference to this PupilDiameterPacket once set.
		 *
		 * Sets this PupilDiameterPacket diameter.
		 *
		 **/
		PupilDiameterPacket& operator= ( float pupilDiameter );

		/**
		 * \brief Assignment operator (const PupilDiameterPacket&)
		 * \param gazePacket (const PupilDiameterPacket&) the DataPacket to copy.
		 * \return (PupilDiameterPacket&) reference to this PupilDiameterPacket once set.
		 *
		 * Sets this PupilDiameterPacket record time stamp, status, name and pupil diameter, using a given PupilDiameterPacket.
		 *
		 **/
		PupilDiameterPacket& operator= ( const PupilDiameterPacket& gazePacket );
	
	}; // END CLASS PUPIL-DIAMETER-PACKET
	// ##################################################

	// ##################################################
	// ###         CLASS GAZE-DIRECTION-PACKET        ###
	// ##################################################

	/**
	 * \class GazeDirectionPacket
	 * \brief Describes Tobii Glasses 2 live gaze direction data packets.
	 *
	 * The gaze direction packets describe the direction in which one eye looks. This is a 3D unit vector in the space of the camera.
	 *
	 **/
	class GazeDirectionPacket : public EyeDataPacket, public Vector3
	{
	public:
		/**
		 * \brief Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param gazeDirection (const Vector3&) the gaze direction unit vector
		 * \param eye (Eye) The eye flag that this packets describes
		 *
		 * Creates and initialize an object of type GazeDirectionPacket, setting its time stamp, status, gaze direction, and eye
		 *
		 **/
		GazeDirectionPacket ( Int64 timeStamp = 0, UInt8 status = 0, const Vector3& gazeDirection = Vector3::zero(), Eye eye = Eye::NONE );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const GazeDirectionPacket&) The GazeDirectionPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type GazeDirectionPacket.
		 *
		 **/
		GazeDirectionPacket ( const GazeDirectionPacket& gazePacket );

		/**
		 * \brief Getter (const Vector3&)
		 * \return (const Vector3&) The gaze direction described by this packet.
		 *
		 * Returns which the gaze direction described by this packet.
		 *
		 **/
		const Vector3& gazeDirection () const;

		/**
		 * \brief Getter / Setter (Vector3&)
		 * \return (Vector3&) The reference of gaze direction described by this packet.
		 *
		 * Returns the reference of gaze direction described by this packet, that can later be used to modified.
		 *
		 **/
		Vector3& gazeDirection ();

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of this PupilDiameterPacket
		 *
		 * Creates a std::string representation of this PupilDiameterPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"pd":value}
		 *
		 **/
		virtual operator std::string () const;
	
		/**
		 * \brief Assignment operator (const Vector3&)
		 * \param gazeDirection (const Vector3&) vector describing the gaze direction
		 * \return (GazeDirectionPacket&) reference to this GazeDirectionPacket once set.
		 *
		 * Sets this GazeDirectionPacket gaze direction using the given vector
		 *
		 **/
		GazeDirectionPacket& operator= ( const Vector3& gazeDirection );

		/**
		 * \brief Assignment operator (const GazeDirectionPacket&)
		 * \param gazePacket (const GazeDirectionPacket&) the DataPacket to copy.
		 * \return (GazeDirectionPacket&) reference to this GazeDirectionPacket once set.
		 *
		 * Sets this GazeDirectionPacket record time stamp, status, name and gaze direction, using a given GazeDirectionPacket.
		 *
		 **/
		GazeDirectionPacket& operator= ( const GazeDirectionPacket& gazePacket );
	
	}; // END CLASS GAZE-DIRECTION-PACKET
	// ##################################################

	// ##################################################
	// ###       CLASS GAZE-2D-POSITION-PACKET        ###
	// ##################################################

	/**
	 * \class Gaze2DPositionPacket
	 * \brief Describes Tobii Glasses 2 live 2D gaze position data packets.
	 *
	 * The 2D gaze position packets describe the position of the gaze fixation, projected to the camera plan.
	 * It is a 2D point where (0,0) is the top-left corner of the camera plan, and (1,1) its bottom-right corner.
	 *
	 **/
	class Gaze2DPositionPacket : public DataPacket, public Point2
	{
	public:
		/**
		 * \brief Gaze2DPositionPacket Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param gazePosition (const Point2&) the gaze fixation position on the camera plan.
		 *
		 * Creates and initialize an object of type Gaze2DPositionPacket, setting its time stamp, status, gaze position 2D.
		 *
		 **/
		Gaze2DPositionPacket ( Int64 timeStamp = 0, UInt8 status = 0, const Point2& gazePosition = Point2::zero() );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const Gaze2DPositionPacket&) The Gaze2DPositionPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type Gaze2DPositionPacket.
		 *
		 **/
		Gaze2DPositionPacket ( const Gaze2DPositionPacket& gazePacket );

		/**
		 * \brief Getter (const Point3&)
		 * \return (const Point2&) The gaze position that this packet describes.
		 *
		 * Returns The gaze position that this packet describes.
		 *
		 **/
		const Point2& gazePosition () const;

		/**
		 * \brief Getter / Setter (const Point2&)
		 * \return (Point2&) The reference of the gaze position that this packet describes.
		 *
		 * Returns The reference of the gaze position that this packet describes, whiwh can later be used to modified its position.
		 *
		 **/
		Point2& gazePosition ();

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of this Gaze2DPositionPacket
		 *
		 * Creates a std::string representation of this Gaze2DPositionPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"gp":[x,y]}
		 *
		 **/
		virtual operator std::string () const;

		/**
		 * \brief Assignment operator (const Point2&)
		 *
		 * \param position (const Point2&) Describing the 2D gaze position on the camera plan
		 * \return (Gaze2DPositionPacket&) reference to this Gaze2DPositionPacket once set.
		 *
		 * Sets this Gaze2DPositionPacket gaze position using the given 2D point.
		 *
		 **/
		Gaze2DPositionPacket& operator= ( const Point2& position );

		/**
		 * \brief Assignment operator (const Gaze2DPositionPacket&)
		 * \param gazePacket (const Gaze2DPositionPacket&) the DataPacket to copy.
		 * \return (Gaze2DPositionPacket&) reference to this Gaze2DPositionPacket once set.
		 *
		 * Sets this Gaze2DPositionPacket record time stamp, status, name and gaze 2D position, using a given Gaze2DPositionPacket.
		 *
		 **/
		Gaze2DPositionPacket& operator= ( const Gaze2DPositionPacket& gazePacket );
	
	}; // END CLASS GAZE-2D-POSITION-PACKET
	// ##################################################

	// ##################################################
	// ###       CLASS GAZE-3D-POSITION-PACKET        ###
	// ##################################################

	/**
	 * \class Gaze3DPositionPacket
	 * \brief Describes Tobii Glasses 2 live 3D gaze position data packets.
	 *
	 * The 3D gaze position packets describe the position of the gaze fixation in the camera space, in millimeters.
	 *
	 **/
	class Gaze3DPositionPacket : public DataPacket, public Point3
	{
	public:
		/**
		 * \brief Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param gazePosition (const Point3&) the gaze fixation position in the camera space.
		 *
		 * Creates and initialize an object of type Gaze3DPositionPacket, setting its time stamp, status, gaze position 3D
		 *
		 **/
		Gaze3DPositionPacket ( Int64 timeStamp = 0, UInt8 status = 0, const Point3& gazePosition = Point3::zero() );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const Gaze3DPositionPacket&) The Gaze3DPositionPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type Gaze3DPositionPacket.
		 *
		 **/
		Gaze3DPositionPacket ( const Gaze3DPositionPacket& gazePacket );

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of this Gaze3DPositionPacket
		 *
		 * Creates a std::string representation of this Gaze3DPositionPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"gp3":[x,y,z]}
		 *
		 **/
		virtual operator std::string () const;
		
		/**
		 * \brief Getter (const Point3&).
		 * \return (const Point3&) The gaze position that this packet describes.
		 *
		 * Returns The gaze position that this packet describes.
		 *
		 **/
		const Point3& gazePosition () const;

		/**
		 * \brief Getter / Setter (Point3&)
		 * \return (Point3&) The reference of the gaze position that this packet describes.
		 *
		 * Returns The reference of the gaze position that this packet describes, whiwh can later be used to modified its position.
		 *
		 **/
		Point3& gazePosition ();
		
		/**
		 * \brief Assignment operator (const Point3&)
		 *
		 * \param gazePosition (const Point3&) Describing the 3D gaze position in the camera space
		 * \return (Gaze3DPositionPacket&) reference to this Gaze3DPositionPacket once set.
		 *
		 * Sets this Gaze3DPositionPacket 3D gaze position using the given 3D point.
		 *
		 **/
		Gaze3DPositionPacket& operator= ( const Point3& gazePosition );

		/**
		 * \brief Assignment operator (const Gaze3DPositionPacket&)
		 * \param gazePacket (const Gaze3DPositionPacket&) the DataPacket to copy.
		 * \return (Gaze3DPositionPacket&) reference to this Gaze3DPositionPacket once set.
		 *
		 * Sets this Gaze3DPositionPacket record time stamp, status, name and gaze 3D position, using a given Gaze3DPositionPacket.
		 *
		 **/
		Gaze3DPositionPacket& operator= ( const Gaze3DPositionPacket& gazePacket );
	
	}; // END CLASS GAZE-3D-POSITION-PACKET
	// ##################################################

	// ##################################################
	// ###          CLASS GYROSCOPE-PACKET           ###
	// ##################################################
	
	/**
	 * \class GyroscopePacket
	 * \brief Describes Tobii Glasses 2 live gyroscope data packets.
	 *
	 * The gyroscope packets describe the rotation recordings of the Tobii Glasses 2 using a 3D vector which values are in degre per seconds ( °/s ).
	 *
	 **/
	class GyroscopePacket : public DataPacket, public Vector3
	{
	public:
		/**
		 * \brief GyroscopePacket Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param orientation (const Vector3&) the orientation of the Tobii Glasses 2 in degre per seconds.
		 *
		 * Creates and initialize an object of type GyroscopePacket, setting its time stamp, status, orientation
		 *
		 **/
		GyroscopePacket ( Int64 timeStamp = 0, UInt8 status = 0, const Vector3& orientation = Vector3::zero() );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const GyroscopePacket&) The GyroscopePacket to copy.
		 *
		 * Creates and initialize a copy of an object of type GyroscopePacket.
		 *
		 **/
		GyroscopePacket ( const GyroscopePacket& gazePacket );

		/**
		 * \brief Getter (const Vector3&).
		 * \return (const Vector3&) The reference of the orientation that this packet describes.
		 *
		 * Returns The reference of the orientation that this packet describes.
		 *
		 **/
		const Vector3& orientation () const;

		/**
		 * \brief Getter / Setter (Vector3&).
		 * \return (Vector3&) The reference of the orientation that this packet describes.
		 *
		 * Returns The reference of the orientation that this packet describes, which content can later be modified.
		 *
		 **/
		Vector3& orientation ();

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of this GyroscopePacket
		 *
		 * Creates a std::string representation of this GyroscopePacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"gy":[dx,dy,dz]}
		 *
		 **/
		virtual operator std::string () const;

		/**
		 * \brief Assignment operator (const Vector3&)
		 *
		 * \param orientation (const Vector3&) Orientation vector.
		 * \return (GyroscopePacket&) reference to this GyroscopePacket once set.
		 *
		 * Sets this GyroscopePacket orientation using the given 3D vector.
		 *
		 **/
		GyroscopePacket& operator= ( const Vector3& orientation );

		/**
		 * \brief Assignment operator (const GyroscopePacket&)
		 * \param gazePacket (const GyroscopePacket&) the DataPacket to copy.
		 * \return (GyroscopePacket&) reference to this GyroscopePacket once set.
		 *
		 * Sets this GyroscopePacket record time stamp, status, name and orientation, using a given GyroscopePacket.
		 *
		 **/
		GyroscopePacket& operator= ( const GyroscopePacket& gazePacket );
	
	}; // END CLASS GYROSCOPE-PACKET
	// ##################################################

	// ##################################################
	// ###        CLASS ACCELEROMETER-PACKET          ###
	// ##################################################

	/**
	 * \class AccelerometerPacket
	 * \brief Describes Tobii Glasses 2 live accelerometer data packets.
	 *
	 * The accelerometer data packets describe the acceleration recordings of the Tobii Glasses 2 using a 3D vector which values are in meter per second squared ( m/s^2 ).
	 *
	 **/
	class AccelerometerPacket : public DataPacket, public Vector3
	{
	public:
		/**
		 * \brief AccelerometerPacket Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param acceleration (const Vector3&) the acceleration of the Tobii Glasses 2 in meter per second squared.
		 *
		 * Creates and initialize an object of type AccelerometerPacket, setting its time stamp, status, acceleration
		 *
		 **/
		AccelerometerPacket ( Int64 timeStamp = 0, UInt8 status = 0, const Vector3& acceleration = Vector3::zero() );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const AccelerometerPacket&) The AccelerometerPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type AccelerometerPacket.
		 *
		 **/
		AccelerometerPacket ( const AccelerometerPacket& gazePacket );

		/**
		 * \brief Getter (const Vector3&).
		 * \return (const Vector3&) The reference of the acceleration that this packet describes.
		 *
		 * Returns The reference of the acceleration that this packet describes.
		 *
		 **/
		const Vector3& acceleration () const;

		/**
		 * \brief Getter / Setter (Vector3&).
		 * \return (Vector3&) The reference of the acceleration that this packet describes.
		 *
		 * Returns The reference of the acceleration that this packet describes, which content can later be modified.
		 *
		 **/
		Vector3& acceleration ();

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of this AccelerometerPacket
		 *
		 * Creates a std::string representation of this AccelerometerPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"ac":Int64}
		 *
		 **/
		virtual operator std::string () const;

		/**
		 * \brief Assignment operator (const Vector3&)
		 *
		 * \param acceleration (const Vector3&) Acceleration vector.
		 * \return (AccelerometerPacket&) Reference to this AccelerometerPacket once set.
		 *
		 * Sets this AccelerometerPacket acceleration using the given 3D vector.
		 *
		 **/
		AccelerometerPacket& operator= ( const Vector3& acceleration );

		/**
		 * \brief Assignment operator (const AccelerometerPacket&)
		 * \param gazePacket (const AccelerometerPacket&) the DataPacket to copy.
		 * \return (AccelerometerPacket&) reference to this AccelerometerPacket once set.
		 *
		 * Sets this AccelerometerPacket record time stamp, status, name and acceleration, using a given AccelerometerPacket.
		 *
		 **/
		AccelerometerPacket& operator= ( const AccelerometerPacket& gazePacket );
	
	}; // END CLASS ACCELEROMETER-PACKET
	// ##################################################

	// ##################################################
	// ###          CLASS PTS-SYNC-PACKET             ###
	// ##################################################

	/**
	 * \class PTSSyncPacket
	 * \brief Describes Tobii Glasses 2 live PTS synchronisation data packets.
	 *
	 * The PTS synchronization data packets specify the correspondance between data packets time tamps and video pipeline time stamps.
	 * They are used for online synchronisation (see Synchronizer)
	 *
	 **/
	class PTSSyncPacket : public DataPacket
	{
	private:
		Int64 _pipelineTimeStamp;	/*!< Specifies the time stamp of the video relative to the video pipeline start */
		UInt8 _pipelineVersion;		/*!< Specifies the pipeline version */

	public:
		/**
		 * \brief PTSSyncPacket Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param pipelineTimeStamp (Int64) pipelineTimeStamp corresponding to this time stamp
		 * \param pipelineVersion (UInt8) pipelineVersion corresponding to this time stamp
		 *
		 * Creates and initialize an object of type PTSSyncPacket, setting its time stamp, status, acceleration
		 *
		 **/
		PTSSyncPacket ( Int64 timeStamp = 0, UInt8 status = 0, Int64 pipelineTimeStamp = 0, UInt8 pipelineVersion = 0 );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const PTSSyncPacket&) The PTSSyncPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type PTSSyncPacket.
		 *
		 **/
		PTSSyncPacket ( const PTSSyncPacket& gazePacket );

		/**
		 * \brief Getter (Int64).
		 * \return (Int64) The video time stamp given by the video pipeline, described by this packet
		 *
		 * Returns The video time stamp given by the video pipeline, described by this packet
		 *
		 **/
		Int64 pipelineTimeStamp () const;

		/**
		 * \brief Getter / Setter (Int64&).
		 * \return (Int64&) The reference of the video time stamp given by the video pipeline
		 *
		 * Returns The reference of the acceleration that this packet describes, that can later be modified.
		 *
		 **/
		Int64& pipelineTimeStamp ();		

		/**
		 * \brief Getter (UInt8).
		 * \return (UInt8) The video pipeline version, described by this packet
		 *
		 * Returns The video pipeline version, described by this packet
		 *
		 **/
		UInt8 pipelineVersion () const;

		/**
		 * \brief Getter / Setter (UInt8&).
		 * \return (UInt8&) The reference of the video pipeline version described by this packet
		 *
		 * Returns The reference of the video pipeline version, described by this packet, that can later be modified.
		 *
		 **/
		UInt8& pipelineVersion ();

		/**
		 * \brief Cast operator (std::string)
		 * \return (std::string) representation of this PTSSyncPacket
		 *
		 * Creates a std::string representation of this PTSSyncPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"pts":Int64,"pv":Int64}
		 *
		 **/
		virtual operator std::string () const;

		/**
		 * \brief Assignment operator (const PTSSyncPacket&)
		 * \param gazePacket (const PTSSyncPacket&) the DataPacket to copy.
		 * \return (PTSSyncPacket&) Reference to this PTSSyncPacket once set.
		 *
		 * Sets this PTSSyncPacket record time stamp, status, name, video time stamp and pipeline version, using a given PTSSyncPacket.
		 *
		 **/
		PTSSyncPacket& operator= ( const PTSSyncPacket& gazePacket );
	
	}; // END CLASS PTS-SYNC-PACKET
	// ##################################################

	// ##################################################
	// ###          CLASS VTS-SYNC-PACKET             ###
	// ##################################################

	/**
	 * \class VTSSyncPacket
	 * \brief Describes Tobii Glasses 2 VTS synchronisation data packets.
	 *
	 * The VTS synchronisation data packets specify the correspondance between data packets time stamps and the video time stamp of the first frame.
	 * They are used for offline synchronisation (see Synchronizer)
	 *
	 **/
	class VTSSyncPacket : public DataPacket
	{
	private:
		Int64 _videoTimeStamp; /*!< Specifies the video time stamp relative to the first frame */

	public:
		/**
		 * \brief VTSSyncPacket Constructor (default)
		 * \param timeStamp (Int64) time-stamp in miliseconds at which the data packet was sent by the Tobii Glasses 2.
		 * \param status (UInt8) status of the recording: 0 means ok, any other values means an error occured durring the recording.
		 * \param videoTimeStamp (Int64) video time-stamp corresponding to this time stamp
		 *
		 * Creates and initialize an object of type VTSSyncPacket, setting its time stamp, status, acceleration
		 *
		 **/
		VTSSyncPacket ( Int64 timeStamp = 0, UInt8 status = 0, Int64 videoTimeStamp = 0 );

		/**
		 * \brief Constructor (copy)
		 * \param gazePacket (const VTSSyncPacket&) The VTSSyncPacket to copy.
		 *
		 * Creates and initialize a copy of an object of type VTSSyncPacket.
		 *
		 **/
		VTSSyncPacket ( const VTSSyncPacket& gazePacket );

		/**
		 * \brief Getter (Int64).
		 * \return (Int64) The video time stamp, described by this packet
		 *
		 * Returns The video time stamp, described by this packet
		 *
		 **/
		Int64 videoTimeStamp () const;

		/**
		 * \brief Getter / Setter (Int64&).
		 * \return (Int64&) The reference of the video time stamp, described by this packet
		 *
		 * Returns The reference of video time stamp, described by this packet, that can later be modified.
		 *
		 **/
		Int64& videoTimeStamp ();

		/**
		 * \brief Cast operator (std::string)
		 * \return std::string representation of this VTSSyncPacket
		 *
		 * Creates a std::string representation of this VTSSyncPacket.
		 * The representation is in JSON format:
		 * {"ts":time-stamp,"s":status,"vts":Int64}
		 *
		 **/
		virtual operator std::string () const;

		/**
		 * \brief Assignment operator (const VTSSyncPacket&)
		 * \param videoTimeStamp (float) The video time stamp.
		 * \return (VTSSyncPacket&) Reference to this VTSSyncPacket once set.
		 *
		 * Sets this VTSSyncPacket record video time stamp, using the given time stamp.
		 *
		 **/
		VTSSyncPacket& operator= ( float videoTimeStamp );

		/**
		 * \brief Assignment operator (const VTSSyncPacket&)
		 * \param gazePacket (const VTSSyncPacket&) The DataPacket to copy.
		 * \return (VTSSyncPacket&) Reference to this VTSSyncPacket once set.
		 *
		 * Sets this VTSSyncPacket record time stamp, status, name, and video time stamp, using a given VTSSyncPacket.
		 *
		 **/
		VTSSyncPacket& operator= ( const VTSSyncPacket& gazePacket );
	
	}; // END CLASS VTS-SYNC-PACKET
	// ##################################################

	// ##################################################
	// ###      SHARED POINTER TYPE DEFINITION        ###
	// ##################################################

	/**
	 * \typedef DataPacketPtr
	 * \brief Shared pointer to a DataPacket
	 *
	 * Defines the type shared pointer to an DataPacket.
	 *
	 **/
	using DataPacketPtr = std::shared_ptr< DataPacket >;

	/**
	 * \typedef PupilCenterPacketPtr
	 * \brief Shared pointer to a PupilCenterPacket
	 *
	 * Defines the type shared pointer to an DataPacket.
	 *
	 **/
	using PupilCenterPacketPtr = std::shared_ptr< PupilCenterPacket >;

	/**
	 * \typedef PupilDiameterPacketPtr
	 * \brief Shared pointer to a PupilDiameterPacket
	 *
	 * Defines the type shared pointer to an PupilDiameterPacket.
	 *
	 **/
	using PupilDiameterPacketPtr = std::shared_ptr< PupilDiameterPacket >;

	/**
	 * \typedef GazeDirectionPacketPtr
	 * \brief Shared pointer to a GazeDirectionPacket
	 *
	 * Defines the type shared pointer to an GazeDirectionPacket.
	 *
	 **/
	using GazeDirectionPacketPtr = std::shared_ptr< GazeDirectionPacket >;

	/**
	 * \typedef Gaze2DPositionPacketPtr
	 * \brief Shared pointer to a Gaze2DPositionPacket
	 *
	 * Defines the type shared pointer to an Gaze2DPositionPacket.
	 *
	 **/
	using Gaze2DPositionPacketPtr = std::shared_ptr< Gaze2DPositionPacket >;

	/**
	 * \typedef Gaze3DPositionPacketPtr
	 * \brief Shared pointer to a Gaze3DPositionPacket
	 *
	 * Defines the type shared pointer to an Gaze3DPositionPacket.
	 *
	 **/
	using Gaze3DPositionPacketPtr = std::shared_ptr< Gaze3DPositionPacket >;

	/**
	 * \typedef GyroscopePacketPtr
	 * \brief Shared pointer to a GyroscopePacket
	 *
	 * Defines the type shared pointer to an GyroscopePacket.
	 *
	 **/
	using GyroscopePacketPtr = std::shared_ptr< GyroscopePacket >;

	/**
	 * \typedef AccelerometerPacketPtr
	 * \brief Shared pointer to a AccelerometerPacket
	 *
	 * Defines the type shared pointer to an AccelerometerPacket.
	 *
	 **/
	using AccelerometerPacketPtr = std::shared_ptr< AccelerometerPacket >;

	/**
	 * \typedef PTSSyncPacketPtr
	 * \brief Shared pointer to a PTSSyncPacket
	 *
	 * Defines the type shared pointer to an PTSSyncPacket.
	 *
	 **/
	using PTSSyncPacketPtr = std::shared_ptr< PTSSyncPacket >;

	/**
	 * \typedef VTSSyncPacketPtr
	 * \brief Shared pointer to a VTSSyncPacket
	 *
	 * Defines the type shared pointer to an VTSSyncPacket.
	 *
	 **/
	using VTSSyncPacketPtr = std::shared_ptr< VTSSyncPacket >;

	// END SHARED POINTER TYPE DEFINITION
	// ##################################################

	// ##################################################
	// ###      STREAM CONCATENATION OPERATORS        ###
	// ##################################################

	/**
	 * \brief Right-hand side output stream concatenation operator for DataPacket.
	 * \param stream (std::ostream&) stream where to concatenate the std::string representation of the DataPacket record.
	 * \param packet (const DataPacket&) the DataPacket record to represent.
	 * \return (std::ostream&) the output stream once the DataPacket record representation has been concatenated to it.
	 *
	 * Concatenate the std::string representation of the DataPacket record in the output stream.
	 *
	 **/
	std::ostream& operator<< ( std::ostream& stream, const DataPacket& packet );

	/**
	 * \brief Left-hand side output stream concatenation operator for DataPacket.
	 * \param packet (const DataPacket&) the DataPacket record to represent.
	 * \param stream (std::ostream&) stream where to concatenate the std::string representation of the DataPacket record.
	 * \return (std::ostream&) the output stream once the DataPacket record representation has been concatenated to it.
	 *
	 * Concatenate the std::string representation of the DataPacket record in the output stream.
	 *
	 **/
	std::ostream& operator<< ( const DataPacket& packet, std::ostream& stream );
	
	/**
	 * \brief Right-hand side output stream concatenation operator for DataPacketPtr.
	 * \param stream (std::ostream&) stream where to concatenate the std::string representation of the DataPacketPtr record.
	 * \param packet (const DataPacketPtr&) the DataPacketPtr record to represent.
	 * \return (std::ostream&) the output stream once the DataPacketPtr record representation has been concatenated to it.
	 *
	 * Concatenate the std::string representation of the DataPacketPtr record in the output stream.
	 *
	 **/
	std::ostream& operator<< ( std::ostream& stream, const DataPacketPtr& packet );

	/**
	 * \brief Left-hand side output stream concatenation operator for DataPacketPtr.
	 * \param packet (const DataPacketPtr&) the DataPacketPtr record to represent.
	 * \param stream (std::ostream&) stream where to concatenate the std::string representation of the DataPacketPtr record.
	 * \return (std::ostream&) the output stream once the DataPacketPtr record representation has been concatenated to it.
	 *
	 * Concatenate the std::string representation of the DataPacketPtr record in the output stream.
	 *
	 **/
	std::ostream& operator<< ( const DataPacketPtr& packet, std::ostream& stream );

	// END STREAM CONCATENATION OPERATORS
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################

#endif // TOBII_DATAPACKETS_HPP
// ##################################################