// ##################################################
// ###                DEPENDANCES                 ###
// ##################################################

// STANDARD DEPENDANCES:
#include "WorkspaceSampler.hpp"

// ##################################################
// ###           NAMESPACE SAMPLING               ###
// ##################################################

namespace Sampling {

	// ##################################################
	// ###         CLASS WORKSPACE SAMPLER            ###
	// ##################################################

	// --------------------------------------------------
	// CONSTRUCTOR

	WorkspaceSampler::WorkspaceSampler ( const QString& workspacePath, const QString& outputPath, int nbThreads )
	: IndexManager(workspacePath, outputPath)
	{
		qRegisterMetaType< Sampling::PendingProjectPtr >();
		this->initThreads(nbThreads);
	}
	
	// --------------------------------------------------
	// DESSTRUCTOR

	WorkspaceSampler::~WorkspaceSampler ()
	{
		this->stopThreads();
	}

	// --------------------------------------------------
	// GETTERS - SETTERS

	int						WorkspaceSampler::nbThreads				() const			{ return this->_nbThreads;				}
	int&					WorkspaceSampler::nbThreads				()					{ return this->_nbThreads;				}
	const ThreadArray&		WorkspaceSampler::threads				() const			{ return this->_threads;				}
	ThreadArray&			WorkspaceSampler::threads				()					{ return this->_threads;				}
	const SamplerArray&		WorkspaceSampler::samplers				() const			{ return this->_samplers;				}
	SamplerArray&			WorkspaceSampler::samplers				()					{ return this->_samplers;				}
	const PendingProjects&	WorkspaceSampler::pendingProjects		() const			{ return this->_pendingProjects;		}
	PendingProjects&		WorkspaceSampler::pendingProjects		()					{ return this->_pendingProjects;		}
	const cv::Size&			WorkspaceSampler::samplingResolution	() const			{ return this->_samplingResolution;		}
	cv::Size&				WorkspaceSampler::samplingResolution	()					{ return this->_samplingResolution;		}

	const QThread*			WorkspaceSampler::threadAt				( int index ) const	{ return this->threads().at(index);		}
	QThread*&				WorkspaceSampler::threadAt				( int index )		{ return this->threads().at(index);		}
	const ProjectSampler*	WorkspaceSampler::samplerAt				( int index ) const	{ return this->samplers().at(index);	}
	ProjectSampler*&		WorkspaceSampler::samplerAt				( int index )		{ return this->samplers().at(index);	}

	// --------------------------------------------------
	// METHODS

	void WorkspaceSampler::initThreads ( int nbThreads )
	{
		// init arrays
		if ( nbThreads == -1 )
			this->nbThreads() = 8;
		else
			this->nbThreads() = nbThreads;

		this->threads()  =  ThreadArray( this->nbThreads(), NULL );
		this->samplers() = SamplerArray( this->nbThreads(), NULL );

		// Message user
		syncPrint( QString("Starting %1 threads (ideal count is %2)").arg(this->nbThreads()).arg(QThread::idealThreadCount()) );
		syncPrint();

		// Init theads, and create samplers
		for ( int index(0) ; index < this->nbThreads() ; ++index )
		{
			// Create the the thread
			this->threadAt(index)  = new QThread();
			this->samplerAt(index) = new ProjectSampler();
			this->samplerAt(index)->moveToThread( this->threadAt(index) );

			// Connect it:
			connect( this->threadAt(index),  SIGNAL( finished() ), this->samplerAt(index), SLOT( deleteLater() ), Qt::AutoConnection );
	        connect( this->samplerAt(index), SIGNAL(    samplingFailed(const QString&, const QString&) ), this, SLOT(     samplingFailed(const QString&, const QString&) ), Qt::DirectConnection );
	        connect( this->samplerAt(index), SIGNAL( samplingProgessed(const QString&, int, int)       ), this, SLOT( samplingProgressed(const QString&, int, int)       ), Qt::DirectConnection );
	        connect( this->samplerAt(index), SIGNAL(  samplingComplete(const QString&)                 ), this, SLOT(   samplingComplete(const QString&)                 ), Qt::DirectConnection );
			this->threadAt(index)->start();
		}
	}
	
	void WorkspaceSampler::stopThreads ()
	{
		for ( QThread*& thread : this->threads() )
		{
			if ( thread != NULL )
			{
				thread->quit();
				thread->wait();
				thread = NULL;
			}
		}
	}

	bool WorkspaceSampler::samplingFinished ()
	{
		if ( ! this->pendingProjects().empty() )
			return false;

		int index(0);
		for ( const ProjectSampler* sampler : this->samplers() )
		{
			if ( sampler->isBusy() )
				return false;
		}

		return true;
	}

	// --------------------------------------------------
	// PUBLIC SLOTS

	void WorkspaceSampler::sampleProject ( const QString& videoPath )
	{
		// This pointer is used to comuncate with the sapler that will process the video
		static ProjectSampler* connectedSampler = NULL;
		PendingProjectPtr ptrProject;

		// If the path is empty, pick a pending one:
		if ( videoPath.isEmpty() )
		{
			// If none is pending, return
			if ( this->pendingProjects().empty() )
			{
				syncPrint( "All videos are processed, or being precessed." );
				return;
			}

			ptrProject = this->pendingProjects().back();
			this->pendingProjects().pop_back();
		}

		// Remember the last one connected:
		if ( connectedSampler != NULL )
		{
			disconnect( this, SIGNAL(startSamplingProject(Sampling::PendingProjectPtr, const QString&)), connectedSampler, SLOT(sampleProject(Sampling::PendingProjectPtr, const QString&)) );
			connectedSampler = NULL;
		}

		// Find a sampler that is not busy, block if none are available
		while ( connectedSampler == NULL )
		{
			for ( ProjectSampler* sampler : this->samplers() )
				if ( ! sampler->isBusy() )
				{
					connectedSampler = sampler;
					break;
				}
		}

		// Connect it:
		connect( this, SIGNAL( startSamplingProject(Sampling::PendingProjectPtr, const QString&) ), connectedSampler, SLOT( sampleProject(Sampling::PendingProjectPtr, const QString&) ), Qt::AutoConnection );
		
		// Set its index file:
		connectedSampler->ptrIndexManager()    = this;
		connectedSampler->samplingResolution() = this->samplingResolution();

		// Start sampling
		emit this->startSamplingProject( ptrProject, this->outputPath() );

		// Sleep
		QThread::msleep(120);
	}

	void WorkspaceSampler::sampleWorkspace ( const cv::Size& samplingResolution )
	{
		// ---------------------------------------------------------------
		// Settings
		this->samplingResolution() = samplingResolution;

		// ---------------------------------------------------------------
		// Open all the project:
		std::map< QString, int > counters;
		for ( const QString& category : this->categories() )
			if ( category != "Background" )
				counters.emplace( category, 0 );

		Array< PendingProjectPtr > pedingProjects;

		for ( const QFileInfo& videoInfo : QDir( this->workspacePath() ).entryInfoList( QDir::Dirs|QDir::NoDotAndDotDot ) )
		{
		 	// Open the project:
		 	PendingProjectPtr ptrProject = std::make_shared<PendingProject>( videoInfo.absoluteFilePath(), SamplingMode::Test() );
		 	QString category = ptrProject->categories()[0];

		 	// Check the project:
			if ( ! ptrProject->isOpen() )
				continue;
			if ( category == "Bad" || category == "Unknown" )
				continue;
			if ( this->categories().labelOfCategory(category) == -1 )
			{
				syncWarning( QString("Project %1 has an unregistered category: %2, ignoring it !").arg(ptrProject->name()).arg(ptrProject->categories()[0]) );
				continue;
			}

			// Add it to the pending list
			pedingProjects.push_back(ptrProject);

			// Count it
			counters.at( ptrProject->categories()[0] ) += 1;
		}

		// ---------------------------------------------------------------
		// Split them in Train / Test / Valisation

		// Create a progress index per category (max is the number of videos in counters)
		std::map< QString, int > indexes;
		for ( const QString& category : this->categories() )
			if ( category != "Background" )
				indexes.emplace( category, 0 );

		std::random_shuffle( pedingProjects.begin(), pedingProjects.end() );

		for ( PendingProjectPtr project : pedingProjects )
		{
			// Read the project's category and index
			QString category = project->categories()[0];
			int		index    = indexes.at( category );

			// If its to small (< 3) skip this category
			if ( counters.at(category) < 3 )
			{
				syncWarning( QString("This project's category appears less than 3 time, it will be train-only ! [project=%1, category=%2]").arg(project->name()).arg(category) );
				project->mode() = SamplingMode::Train();
			}
			else
			{
				if ( index < round( counters.at(category)*0.8 ) )
				{
					project->mode() = SamplingMode::Validation();
				}
				if ( index < round( counters.at(category)*0.6 ) )
				{
					project->mode() = SamplingMode::Train();
				}
			}
			this->pendingProjects().push_back(project);

			// Up the count
			indexes.at( category ) += 1;
		}

		syncPrint();
		syncPrint( "Projects:" );
		for ( const PendingProjectPtr project : this->pendingProjects() )
			syncPrint( QString("\t%1, %2, %3").arg(project->name()).arg( (QString)project->mode() ).arg(project->categories()[0]) );
		syncPrint();

		// ---------------------------------------------------------------
		// Start the sampling works:
		for ( int index(0) ; index < this->nbThreads() ; ++index )
		 	this->sampleProject();

		// ---------------------------------------------------------------
		// Wait for completion:
		while ( ! this->samplingFinished() )
		{
			QThread::msleep(1000);
		}

		// ---------------------------------------------------------------
		//  Save log

		QFile logFile( this->outputPath()+"/SamplingLog.txt" );
		if ( ! logFile.open(QIODevice::WriteOnly|QIODevice::Text) )
			syncError( "Could not open SamplingLog.txt" );
		QTextStream logStream( &logFile );

		logStream << static_cast< IndexManager >( *this );

		QTextStream out( stdout );
		out << static_cast< IndexManager >( *this );

	}

	// --------------------------------------------------
	// PRIVATE SLOTS

	void WorkspaceSampler::samplingFailed ( const QString& videoName, const QString& error )
	{
		// Display the error message
		syncError( QString("%1\t Sampling failed ! error: %2").arg(videoName).arg(error) );

		// Sample a pending video if any:
		this->sampleProject();
	}
	
	void WorkspaceSampler::samplingProgressed ( const QString& videoName, int currentIndex, int lastIndex )
	{
		syncPrint( QString("%1\t Sampling progession: %2/%3:%4\%").arg(videoName).arg(currentIndex).arg(lastIndex).arg(currentIndex*100.0/lastIndex) );
	}
		
	void WorkspaceSampler::samplingComplete ( const QString& videoName )
	{
		// Display the success message:
		syncPrint( QString("%1\t Sampling completed").arg(videoName) );

		// Sample a pending video if any:
		this->sampleProject();
	}

	// END CLASS WORKSPACE SAMPLER
	// ##################################################

} // END NAMESPACE SAMPLING
// ##################################################
