// ##################################################
// ###            MATCHING_UTILS_HPP              ###
// ##################################################
#ifndef MATCHING_UTILS_HPP
#define MATCHING_UTILS_HPP

// ##################################################
// ###                DEPENDANCES                 ###
// ##################################################

// STANDARD DEPENDANCES:
#include <vector>
#include <list>
#include <iostream>
#include <map>
#include <memory>

// QT5 CORE DEPENDANCES:
#include <QDir>
#include <QString>
#include <QMutex>

// OPENCV 2.4.11 DEPENDANCES:
#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

// ##################################################
// ###           NAMESPACE SAMPLING               ###
// ##################################################

namespace Sampling {

	/**
	 * \brief Typedef for std::vector<  >
	 */
	template< typename Type >
	using Array = std::vector< Type >;

	/**
	 * \brief Typedef for std::map<  >
	 */
	template< typename KeyType, typename ValueType >
	using HashMap = std::map< KeyType, ValueType >;

	/**
	 * \brief Typedef for Array< cv::Rect >
	 */
	using RectArray = Array< cv::Rect >;
	/**
	 * \brief Typedef for Array< cv::Mat >
	 */
	using MatArray = Array< cv::Mat >;
	/**
	 * \brief Typedef for unsigned char
	 */
	using UChar = unsigned char;
	/**
	 * \brief Typedef for unsigned int
	 */
	using UInt = unsigned int;

	/**
	 * \brief Compute a radom integet in the interval [a, b]
	 * \param a (int) Minimum bound.
	 * \param b (int) Maximum bound.
	 * \return (int) Ransom number in [a, b]
	 */
	int randIn ( int a, int b );

	/**
	 * \breif Compute a random rectangle.
	 * \param minSize (const cv::Size&) minimum size of the random rectangle.
	 * \param limits (const cv::Rect&) The random rectangle is included in the 'limit' rectangle.
	 * \param square (bool) if true the random rectangle is square.
	 * \return A random rectangle.
	 */
	cv::Rect randomRect ( const cv::Size& minSize, const cv::Rect& limits, bool square = false );

	/**
	 * \brief Compute a radom integet in the interval [a, b]
	 * \param rectangles (const RectArray&) Rectangles to compute the union of
	 * \return (cv::Rect) Union rectangle
	 */
	cv::Rect unionOfAll ( const RectArray& rectangles );

	/**
	 * \brief Computed the intersection over union.
	 * \param A (const cv::Rect& A) First rectangle considered.
	 * \param B (const cv::Rect& A) Second rectangle considered.
	 * \return (double) Ratio of the area of the intersection of A and B, by the area of the union of A and B.
	 */
	double IoU ( const cv::Rect& A, const cv::Rect& B );

	static QMutex cerrMutex;
	void syncPrint	 ( const QString& text = "" );
	void syncWarning ( const QString& text = "" );
	void syncError	 ( const QString& text = "" );

} // END NAMESPACE SAMPLING
// ##################################################

std::ostream& operator<<( std::ostream& stream, const cv::Size& size );
std::ostream& operator<<( std::ostream& stream, const QString& string );
std::ostream& operator<<( std::ostream& stream, const QStringList& lst );

#endif // MATCHING_UTILS_HPP
// ##################################################