// ##################################################
// ###         MATCHING_SAMPLINGTOOLS_HPP         ###
// ##################################################
#ifndef MATCHING_SAMPLINGTOOLS_HPP
#define MATCHING_SAMPLINGTOOLS_HPP

// ##################################################
// ###                DEPENDANCES                 ###
// ##################################################

// STANDARD DEPENDANCES:
#include <vector>
#include <list>

// OPENCV 2.4.11 DEPENDANCES:
#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

// INTERNAL MODULES:
#include "Utils.hpp"

// ##################################################
// ###           NAMESPACE SAMPLING               ###
// ##################################################

namespace Sampling {

	/**
	 * \brief Compute the bounding box of samples in the binary iamge.
	 * \param binaryImage (const cv::Mat&) The binary image with samples in white (255) and background in black (0).
	 * \return (RectArray) The list of bounding boxes, i.e. std::vector<cv::Rect>
	 *
	 * Go through the binary image, looking for white pixels. When one is found its 8-coonected neighborhood is visited.
	 * The most left-top, and bottom-right are kept to compute the bounding box of this sample.
	 */
	RectArray sampleObjectsBoundingBoxes ( const cv::Mat& binaryImage );

	/**
	 * \brief Sample bounding boxes on the background, excluding the band on the left and right of the union of all object bounding boxes.
	 * \param imageSize (const cv::Rect&) The size of the image.
	 * \param forbiddenRects (const RectArray&) Sample cannot be taken from these rectangles, this is usefull to exclude objects locations.
	 * \param number (UInt) This function will extract 'number' background bounding boxes if possible.
	 * \param minSize (UInt) The minimum width of the bounding box to extract. This should not be bigger than the remaining space on top/bottom.
	 * \param maxIoU (double) Controle the maximum overlapp between two background samples.
	 * \param maxNbIter (int) Maximum number of iteration.
	 * \return (RectArray) The list of bounding boxes.
	 * \ref sampleObjectsBoundingBoxes
	 *
	 * Randomly sample bounding boxes on the background, excluding the area containing objects.
	 */
	RectArray sampleBackgroundBoundingBoxes ( const cv::Size& imageSize, const RectArray& forbiddenRects, UInt number, UInt minSize, double maxIoU = 100.0, int maxNbIter = -1 );

	/**
	 * \brief Sample bounding boxes on the background, excluding the band on the left and right of the union of all object bounding boxes.
	 * \param imageSize (const cv::Rect&) The size of the image.
	 * \param forbiddenRect (const cv::Rect&) Sample cannot be taken from this rectangle, this is usefull to exclude the object location.
	 * \param number (UInt) This function will extract 'number' background bounding boxes.
	 * \param minSize (UInt) The minimum width of the bounding box to extract. This should not be bigger than the remaining space on top/bottom.
	 * \param maxIoU (double) Controle the maximum overlapp between two background samples.
	 * \param maxNbIter (int) Maximum number of iteration.
	 * \return (RectArray) The list of bounding boxes.
	 * \ref sampleObjectsBoundingBoxes
	 *
	 * Randomly sample bounding boxes on the background, excluding the area containing objects.
	 */
	RectArray sampleBackgroundBoundingBoxes ( const cv::Size& imageSize, const cv::Rect& forbidden, UInt number, UInt minSize, double maxIoU = 100.0, int maxNbIter = -1 );

	/**
	 * \brief Copy the image in the given bounding-box, with the given padding added to the bounding-box sides.
	 * \param image (const cv::Mat&) The image to copy from.
	 * \param boundingBox (const cv::Rect&) The bounding box.
	 * \param resolution (const cv::Size&) If specified, the sample is resize to fit the given resolution.
	 * \param padding (int) The padding to add on the side of the bounding box.
	 * \retrun A copy of the image in the bounding box.
	 *
	 * Extract the given sample on the image.
	 */
	cv::Mat extractSample ( const cv::Mat& image, const cv::Rect& boundingBox, cv::Size resolution = cv::Size(0,0), int padding = 0 );

	/**
	 * \brief Extract every sample of every given bounding boxes, with the given padding, and resize them all to the given resolution.
	 * \param image (const cv::Mat&) The image to extract the sample from.
	 * \param bbs (const RectArray&) The list of bounding box.
	 * \param resolution (const cv::Size&) The final resolution of every samples.
	 * \param padding (int) The number of pixel that is kept at the border of the bounding box. (can be negative)
	 * \return (MatArray) The list of bounding box.
	 *
	 * Extract the given samples on the image.
	 *
	 * \ref extractSample
	 */
	MatArray extractSamples ( const cv::Mat& image, const RectArray& bbs, cv::Size resolution = cv::Size(0,0), int padding = 0 );

	/**
	 * \brief Extract the samples in the image, given the binary mask of same size.
	 * \param image (const cv::Mat&) The image to extract the sample from.
	 * \param binaryImage (const cv::Mat&) The mask where samples appear in white (255) and the rest in black (0).
	 * \param resolution (const cv::Size&) The resolution of the samples once extracted.
	 * \param padding (int) The padding, addeed to the bounding box of each sample.
	 *
	 * Extract samples in the image based on the image mask.
	 *
	 * \ref extractBoundingBoxList
	 * \ref extractSample
	 * \ref extractSamples
	 */
	MatArray extractSamples ( const cv::Mat& image, const cv::Mat& binaryImage, cv::Size resolution = cv::Size(0,0), int padding = 0 );

	/**
	 * \brief Performs multiple rotatation of the given sample.
	 * \param sample (const cv::Mat&)  The sample to rotate
	 * \param angleStep (double) The cumulative angle to each rotation. 360 modulus 'angleStep' must be be equal to 0.
	 * \return (MatArray) An array containing all the rotated samples.
	 *
	 * Perform successive rotation of angles 0, 0+angleStep, 0+2*angleStep, ..., 360-angleStep
	 * The original sample is kept as is in the output as well (rotated with and angle of 0).
	 */
	MatArray rotateSample ( const cv::Mat& sample, double angleStep = 90.0 );

	/**
	 * \brief Performs multiple rotatation on each of the given samples.
	 * \param samples (const MatArray&)  The list of samples to rotate.
	 * \param angleStep (double) The cumulative angle to each rotation. 360 modulus 'angleStep' must be be equal to 0.
	 * \return (MatArray) An array containing all the rotated samples.
	 *
	 * Perform successive rotation of angles 0, 0+angleStep, 0+2*angleStep, ..., 360-angleStep to each samples in the list
	 *
	 * \ref rotateBlob
	 */
	MatArray rotateSamples ( const MatArray& samples, double angleStep = 90.0 );

	/**
	 * \brief Blur multiple times the given samples with increasing kernel size.
	 * \param sample (const cv::Mat&) The sample to blur.
	 * \param nbLevels (int) The number of times the object is blurred.
	 * \param initKernelSize (int) The first kernel size that is used.
	 * \param kernelSizeStep (int) The incremental size, added to the kernel size, between each blurring.
	 * \return (MatArray) The list of blurred samples.
	 *
	 * Computes different blurring of the given sample. The original sample is kept as is in the output as well.
	 */
	MatArray blurSample ( const cv::Mat& sample, int nbLevels = 3, int initKernelSize = 3, int kernelSizeStep = 2 );

	/**
	 * \brief Blur the list of samples with the function 'blurBlob'.
	 * \param sample (const MatArray&) The list of sample to blur.
	 * \param nbLevels (int) The number of times the object is blurred.
	 * \param initKernelSize (int) The first kernel size that is used.
	 * \param kernelSizeStep (int) The incremental size, added to the kernel size, between each blurring.
	 * \return (MatArray) The list of blurred samples.
	 * \ref blurBlob
	 *
	 * Blur every given sample using the function blurBlob, exta arguments are forwarded to this function.
	 */
	MatArray blurSamples ( const MatArray& samples, int nbLevels = 3, int initKernelSize = 3, int kernelSizeStep = 2 );

	/**
	 * \breif Rotate and blurr each of the given samples.
	 * \param samples (const MatArray&) The list of samples to augment.
	 * \param angleStep (double) The cumulative angle to each rotation. 360 modulus 'angleStep' must be be equal to 0.
	 * \param nbLevels (int) The number of times the object is blurred.
	 * \param initKernelSize (int) The first kernel size that is used.
	 * \param kernelSizeStep (int) The incremental size, added to the kernel size, between each blurring.
	 * \return (MatArray) The list of augmented samples.
	 * \ref rotateBlobs
	 * \ref blurBlobs
	 *
	 * Rotate and blur each samples using the functions 'rotateBlobs' and 'blurBlobs'.
	 * Extra arguments are forwarded to these functions.
	 */
	MatArray augmentSamples ( const MatArray& samples, double angleStep = 90.0, int nbLevels = 3, int initKernelSize = 3, int kernelSizeStep = 2 );

} // END NAMESPACE SAMPLING
// ##################################################

#endif // MATCHING_SAMPLINGTOOLS_HPP
// ##################################################