// ##################################################
// ###        MATCHING_SAMPLINGTOOLS_CPP          ###
// ##################################################
#ifndef MATCHING_PATCHTOOLS_CPP
#define MATCHING_PATCHTOOLS_CPP

// ##################################################
// ###                DEPENDANCES                 ###
// ##################################################

#include "SamplingTools.hpp"

// ##################################################
// ###            NAMESPACE SAMPLING              ###
// ##################################################

namespace Sampling {

	RectArray sampleObjectsBoundingBoxes ( const cv::Mat& binaryImage )
	{
		// --------------------------------------------------
		// Binarize the label image, it should already be
		// But with image compression, better to be carefull
		cv::Mat binaryMask = binaryImage;
		if ( binaryMask.type() == CV_8UC3 )
			cv::cvtColor( binaryMask, binaryMask, CV_BGR2GRAY );

		cv::threshold( binaryMask, binaryMask, 127, 255, cv::THRESH_BINARY );

		// Blobs are sorted in a std::map, indexed by the area of the associated blob
		RectArray samples;

		if ( cv::sum(binaryMask) == cv::Scalar(0.0) )
			return samples;

		// Label image, to account for pixels already processed
		cv::Mat labelImage = cv::Mat::zeros( binaryMask.size(), CV_8UC1 );
		int label( 1 );

		for ( int x(0) ; x < binaryMask.cols ; ++x )
		for ( int y(0) ; y < binaryMask.rows ; ++y )
		{
			// If we encounter a blob:
			if ( binaryMask.at<UChar>(y, x) == 255 && labelImage.at<UChar>(y, x) == 0 )
			{
				// Init a new blob:
				cv::Point topLeft(x, y), bottomRight(x, y);
				std::list< cv::Point > lstToProcess( 1, cv::Point(x, y) );

				// go in the neigh
				while ( lstToProcess.size() != 0 )
				{
					// Pop a point
					cv::Point current = lstToProcess.front();
					lstToProcess.pop_front();

					// Update min point if needed
					if ( current.x < topLeft.x )
						topLeft.x = current.x;
					if ( current.y < topLeft.y )
						topLeft.y = current.y;

					// Update max point if needed
					if ( current.x > bottomRight.x )
						bottomRight.x = current.x;
					if ( current.y > bottomRight.y )
						bottomRight.y = current.y;

					// Update label image
					labelImage.at<UChar>(current.y, current.x) = label;

					// Look in the neighborhood
					for ( int dx(-1) ; dx <= 1 ; ++dx )
					for ( int dy(-1) ; dy <= 1 ; ++dy )
					{
						cv::Point neib(current.x+dx, current.y+dy);

						// check the image bounds
						if( 0 <= neib.x && neib.x < binaryMask.cols && 0 <= neib.y && neib.y < binaryMask.rows )
						{
							// check the image and label value:
							if ( binaryMask.at<UChar>(neib.y, neib.x) == 255 && labelImage.at<UChar>(neib.y, neib.x) == 0 )
							{
								if ( std::find(lstToProcess.begin(), lstToProcess.end(), neib) == lstToProcess.end() )
								{
									// std::cout << " -> pushing it";
									lstToProcess.push_back( neib );
								}
							}
						}
					} // end for for
				} // end while

				// Remind this blob
				cv::Rect sample( topLeft.x, topLeft.y, bottomRight.x-topLeft.x+1, bottomRight.y-topLeft.y+1 );
				samples.push_back( sample );

				// In our case their is only one blob,
				// so we exit here
				// return samples;

				// Update label
				++label;
			
			} // end if == 255

		} // end image loop
		return samples;
	}

	RectArray sampleBackgroundBoundingBoxes ( const cv::Size& imageSize, const RectArray& forbiddenRects, UInt number, UInt minSize, double maxIoU, int maxNbIter )
	{
		// Init the output resut array
		RectArray results;

		// Check nb iter:
		if ( maxNbIter == -1 )
			maxNbIter = std::numeric_limits<int>::max();

		// Check if it is possible:
		cv::Rect forbidden = unionOfAll( forbiddenRects );
		if ( minSize > forbidden.y && minSize > (imageSize.height-(forbidden.y+forbidden.height)) )
			return results;

		// Get them:
		UInt nb( 0 ), nbIter( 0 );
		srand( time(NULL) ); // move this
		while ( nb < number && nbIter < maxNbIter )
		{
			// Compute a random rectangle
			cv::Rect randRect = randomRect( cv::Size(minSize, minSize), cv::Rect(cv::Point(0,0),imageSize), true );

			// Chech if this rectangle intersect with the forbidden area
			bool ok( true );
			for ( const cv::Rect forbidden : forbiddenRects )
			{
				if ( (randRect & forbidden).area() != 0 )
				{
					ok = false;
					break;
				}
			}
			
			// Check the IoU with other background samples (if its is not totaly allow)
			for ( int index(0) ; maxIoU != 100.0 && ok && index < nb ; ++index )
				if ( IoU(randRect, results.at(index)) > maxIoU )
					ok = false;

			if ( ok )
			{
				results.push_back( randRect );
				++nb;
			}
			++nbIter;
		}
		return results;
	}

	RectArray sampleBackgroundBoundingBoxes ( const cv::Size& imageSize, const cv::Rect& forbidden, UInt number, UInt minSize, double maxIoU, int maxNbIter )
	{
		return sampleBackgroundBoundingBoxes( imageSize, RectArray(1, forbidden), number, minSize, maxIoU, maxNbIter );
	}

	cv::Mat extractSample ( const cv::Mat& image, const cv::Rect& boundingBox, cv::Size resolution, int padding )
	{
		// Pad the bounding box
		cv::Rect bb = boundingBox;
		bb.x      = std::max( bb.x - padding, 0 );
		bb.y      = std::max( bb.y - padding, 0 );
		bb.width  = std::min( bb.width  - padding, image.cols );
		bb.height = std::min( bb.height - padding, image.rows );

		// Copy
		cv::Mat blob = image( bb );

		// Resize ?
		if ( resolution.width != 0 && resolution.height != 0 )
			cv::resize(blob, blob, resolution);

		return blob;
	}

	MatArray extractSamples ( const cv::Mat& image, const RectArray& bbs, cv::Size resolution, int padding )
	{
		MatArray results( bbs.size() );
		for ( int index(0) ; index < bbs.size() ; ++index )
			results.at(index) = extractSample( image, bbs.at(index), resolution, padding );
		return results;
	}

	MatArray extractSamples ( const cv::Mat& image, const cv::Mat& binaryImage, cv::Size resolution, int padding )
	{
		return extractSamples( image, sampleObjectsBoundingBoxes(binaryImage), resolution, padding );
	}

	MatArray rotateSample ( const cv::Mat& sample, double angleStep )
	{
		CV_Assert( std::fmod(360.0, angleStep) == 0 );
		int nb = std::floor( 360.0 / angleStep );

		MatArray results( nb );
		cv::Point2f	center ( sample.cols/2, sample.rows/2 );

		double angle( 0 );
		for ( int index(0) ; index < nb ; ++index )
		{
			cv::warpAffine( sample, results.at(index), cv::getRotationMatrix2D(center, angle, 1.0), sample.size() );
			angle += angleStep;
		}

		return results;
	}

	MatArray rotateSamples ( const MatArray& samples, double angleStep )
	{
		MatArray results;
		for ( const cv::Mat& sample : samples )
		{
			MatArray tmp = rotateSample( sample, angleStep );
			std::move( tmp.begin(), tmp.end(), std::back_inserter(results) );
		}
		return results;
	}

	MatArray blurSample ( const cv::Mat& sample, int nbLevels, int initKernelSize, int kernelSizeStep )
	{
		MatArray results( nbLevels+1 );
		results.at(0) = sample;

		int kernelSize = initKernelSize;
		for ( int index(1) ; index <= nbLevels ; ++index )
		{
			cv::GaussianBlur( sample, results.at(index), cv::Size(kernelSize,kernelSize), 0 );
			kernelSize += kernelSizeStep;
		}
		return results;
	}

	MatArray blurSamples ( const MatArray& samples, int nbLevels, int initKernelSize, int kernelSizeStep )
	{
		MatArray results;
		for ( const cv::Mat& sample : samples )
		{
			MatArray tmp = blurSample( sample, nbLevels, initKernelSize, kernelSizeStep );
			std::move( tmp.begin(), tmp.end(), std::back_inserter(results) );
		}
		return results;
	}

	MatArray augmentSamples ( const MatArray& samples, double angleStep, int nbLevels, int initKernelSize, int kernelSizeStep )
	{
		return blurSamples( rotateSamples( samples, angleStep ), nbLevels, initKernelSize, kernelSizeStep );
	}

} // END NAMESPACE SAMPLING
// ##################################################

#endif // MATCHING_SAMPLINGTOOLS_CPP
// ##################################################