// ##################################################
// ###            PROJECTSAMPLER_CPP              ###
// ##################################################
#ifndef PROJECTSAMPLER_CPP
#define PROJECTSAMPLER_CPP

// ##################################################
// ###                DEPENDANCES                 ###
// ##################################################

#include "ProjectSampler.hpp"

// ##################################################
// ###           NAMESPACE SAMPLING               ###
// ##################################################

namespace Sampling {

	// ##################################################
	// ###           CLASS PROJECTSAMPLER             ###
	// ##################################################

	ProjectSampler::ProjectSampler ( IndexManager* ptrIndexManager )
	: _ptrIndexManager( ptrIndexManager )
	, _isBusy( false )
	{}

	const IndexManager*			ProjectSampler::ptrIndexManager		() const	{ return this->_ptrIndexManager;	}
	IndexManager*&				ProjectSampler::ptrIndexManager		()			{ return this->_ptrIndexManager;	}
	const PendingProjectPtr&	ProjectSampler::ptrPendingProject	() const	{ return this->_ptrPendingTask;		}
	PendingProjectPtr&			ProjectSampler::ptrPendingProject	()			{ return this->_ptrPendingTask;		}
	const cv::Size&				ProjectSampler::samplingResolution	() const	{ return this->_samplingResolution;	}
	cv::Size&					ProjectSampler::samplingResolution	()			{ return this->_samplingResolution;	}
	bool						ProjectSampler::isBusy				() const	{ return this->_isBusy;				}
	bool&						ProjectSampler::isBusy				()			{ return this->_isBusy;				}


	void ProjectSampler::sampleFrame ( const cv::Mat& image, const cv::Mat& mask, const QString& prefixFG, const QString& prefixBG )
	{
		// --------------------------------------------------
		// Get the objectsblobs
		RectArray objectsBB = sampleObjectsBoundingBoxes( mask );

		// Get the biggest:
		if ( ! objectsBB.empty() )
		{
			cv::Rect bigger = objectsBB.at( 0 );

			for ( int id(1) ; id < objectsBB.size() ; ++id )
				if ( objectsBB.at(id).area() > bigger.area() )
					bigger = objectsBB.at(id);

			objectsBB = RectArray(1, bigger);
		}


		int nbBB = objectsBB.size();
		int ratio = 1;
		
		if ( ! objectsBB.empty() )
		{
			MatArray  objectsSamples = extractSamples( image, objectsBB, this->samplingResolution() );

			if ( this->ptrPendingProject()->mode() != SamplingMode::Test() )
			{
				objectsSamples = augmentSamples( objectsSamples, 90.0, 3, 3, 2 );
			}

			// --------------------------------------------------
			// Save them into png files
			for ( int index(0) ; index < objectsSamples.size() ; ++index )
			{
				QString fullpath = QString("%1_fg%2.png").arg(prefixFG).arg(index, 2, 10, QLatin1Char('0'));
				
			 	cv::imwrite( fullpath.toStdString(), objectsSamples.at(index), {CV_IMWRITE_PNG_COMPRESSION, 0} );

		 		if ( this->ptrPendingProject()->mode() == SamplingMode::Test() )
		 		{
		 			this->ptrIndexManager()->addIndex(
		 				this->ptrPendingProject()->name(),
		 				this->ptrPendingProject()->categories()[0],
		 				this->ptrPendingProject()->currentTimeStamp(),
		 				fullpath
		 			);
		 		}
		 		else
		 			this->ptrIndexManager()->addIndex( this->ptrPendingProject()->mode(), fullpath, this->ptrPendingProject()->categories()[0] );
			}

			
			if ( this->ptrPendingProject()->mode() != SamplingMode::Test() )
			{
				// --------------------------------------------------
				// Compute the forbidden rectangle
				cv::Rect forbidden = unionOfAll( objectsBB );
				forbidden.x        = 0;
				forbidden.width    = image.cols;
				
				// --------------------------------------------------
				// Get the background blobs
				RectArray backgroundBB = sampleBackgroundBoundingBoxes( image.size(), forbidden, nbBB*ratio, 95, 20.0, nbBB*100 );

				if ( ! backgroundBB.empty() )
				{
					MatArray  backgroundSamples = extractSamples( image, backgroundBB, this->samplingResolution() );
							  backgroundSamples = augmentSamples( backgroundSamples, 90.0, 3, 3, 2 );

					// --------------------------------------------------
					// Save them into png files
					for ( int index(0) ; index < backgroundSamples.size() ; ++index )
					{
						QString fullpath = QString("%1_bg%2.png").arg(prefixBG).arg(index, 4, 10, QLatin1Char('0'));

						cv::imwrite( fullpath.toStdString(), backgroundSamples.at(index), {CV_IMWRITE_PNG_COMPRESSION, 0} );

						this->ptrIndexManager()->addIndex( this->ptrPendingProject()->mode(), fullpath, "Background" );
					}
				}
			}

			// --------------------------------------------------
			// Display
			// cv::Mat copy = image.clone();
			// for ( const cv::Rect rect : objectsBB )
			// 	cv::rectangle( copy, rect, cv::Scalar(0,255,0) );
			// for ( const cv::Rect rect : backgroundBB )
			// 	cv::rectangle( copy, rect, cv::Scalar(0,0,255) );
			// 
			// cv::resize( copy, copy, cv::Size(0,0), 0.5, 0.5 );
			// cv::imshow( "image.png", copy );//, {CV_IMWRITE_PNG_COMPRESSION, 0} );
			// cv::waitKey( 1 );

		} // if ( ! objectsBB.empty() )
	}
	
	void ProjectSampler::sampleProject ( Sampling::PendingProjectPtr ptrProject, const QString& outputPath )
	{
		using namespace Tobii;
		
		// Set the busy flag:
		this->isBusy() = true;

		this->ptrPendingProject() = ptrProject;
		syncPrint( QString("Starting sampling of project \"%1\", category=%2")
			.arg(this->ptrPendingProject()->name())
			.arg(this->ptrPendingProject()->categories()[0])
		);

		// Create the outputs directory for the FG / BG
		QString pathOutFG = QString("%1/%2/%3").arg(outputPath).arg(this->ptrPendingProject()->categories()[0]).arg(this->ptrPendingProject()->name());
		QDir dirFG( pathOutFG );
		if ( !dirFG.exists() )
		{
			dirFG.mkpath( dirFG.absolutePath() );
		}

		QString pathOutBG = QString("%1/Background/%2").arg(outputPath).arg(this->ptrPendingProject()->name());
		if ( this->ptrPendingProject()->mode() != SamplingMode::Test() )
		{
			QDir dirBG( pathOutBG );
			if ( !dirBG.exists() )
			{
				dirBG.mkpath( dirBG.absolutePath() );
			}
		}

		// Sample each frame of the project:
		Int64 timeStamp = -1;
		while ( this->ptrPendingProject()->currentTimeStamp() <= this->ptrPendingProject()->lastTimeStamp() )
		{
			// Read a set of frame+annotation
			FramesMap frames = this->ptrPendingProject()->getCurrentFrames();
			timeStamp        = this->ptrPendingProject()->currentTimeStamp();

			// Create the prefix for the output images:
			QString prefixFG = QString("%1/Frame_%2").arg(pathOutFG).arg(timeStamp, 4, 10, QLatin1Char('0'));
			QString prefixBG = QString("%1/Frame_%2").arg(pathOutBG).arg(timeStamp, 4, 10, QLatin1Char('0'));

			this->sampleFrame( frames.at(Project::Video), frames.at(Project::Annotation), prefixFG, prefixBG );

			// Signal current progess
			emit this->samplingProgessed( this->ptrPendingProject()->name(), this->ptrPendingProject()->currentIndex(), this->ptrPendingProject()->lastIndex() );

			// Sample next frame if any
			this->ptrPendingProject()->next();
		}

		// Signal the completion
		this->isBusy() = false;
		emit this->samplingComplete( this->ptrPendingProject()->name() );

		// Stop the thread until the job is restart
		return QThread::yieldCurrentThread();
	}

	// END CLASS PROJECTSAMPLER
	// ##################################################

} // END NAMESPACE SAMPLING
// ##################################################

#endif // VIDEOSAMPLER_CPP
// ##################################################
