// ##################################################
// ###             INDEXMANAGER_CPP               ###
// ##################################################
#ifndef INDEXMANAGER_CPP
#define INDEXMANAGER_CPP

// ##################################################
// ###                DEPENDANCES                 ###
// ##################################################

#include "IndexManager.hpp"

// ##################################################
// ###           NAMESPACE SAMPLING               ###
// ##################################################

namespace Sampling {

	// ##################################################
	// ###           CLASS INDEXMANAGER               ###
	// ##################################################

	// --------------------------------------------------
	// CONSTRUCTOR

	IndexManager::IndexManager ( const QString& workspacePath, const QString& outputPath )
	{
		// ---------------------------------------------------------------
		// Check that the workspace folder exists:
		QDir dirWorkspace( workspacePath );
		if ( ! dirWorkspace.exists() )
			syncError( QString("The Workspace folder \"%1\" does not exist !").arg(workspacePath) );

		// Check that the workspace isn't empty:
		if ( dirWorkspace.entryInfoList( QDir::Dirs|QDir::NoDotAndDotDot ).isEmpty() )
			syncError( QString("The Workspace folder \"%1\"is empty !").arg(workspacePath) );

		this->workspacePath() = workspacePath;

		// Check that the categories file exist
		QFile categoriesFile( workspacePath+"/Categories.txt" );
		if ( ! categoriesFile.exists() )
			syncError( QString( "Categories file does not exist: %1/Categories.txt").arg(workspacePath) );

		// Try to open it
		if( ! categoriesFile.open(QIODevice::ReadOnly|QIODevice::Text) )
    		syncError( QString( "Categories file culd not be openned: %1/Categories.txt").arg(workspacePath) );

		QTextStream categoriesStream( &categoriesFile );
		categoriesStream >> this->categories();

		syncPrint( QString("Categories: (%1)").arg(this->categories().size()) );
		for ( int id(0) ; id < this->categories().size() ; ++id )
			syncPrint( QString("\t%1:%2").arg(id).arg(this->categories().at(id)) );
		syncPrint();

		// ---------------------------------------------------------------
		// Ouput dir:
		QDir outputDir( outputPath );
		if ( outputPath.split(' ', QString::SkipEmptyParts).size() != 1 )
			syncError( QString("Ouput path should not containt spaces (caffe does not support it): %1").arg(outputDir.absolutePath()) );

		if ( ! outputDir.exists() )
			outputDir.mkpath( outputDir.absolutePath() );

		this->outputPath() = outputDir.absolutePath();

		// ---------------------------------------------------------------
		// Create the 3 index files:
		this->addIndexFile( "IndexTrain" );
		this->addIndexFile( "IndexValidation" );
		this->addIndexFile( "IndexTest" );

		// ---------------------------------------------------------------
		// Create the SynsetWords.txt:
		QFile synsetFile( outputPath+"/SynsetWords.txt" );
		if ( ! synsetFile.open( QIODevice::WriteOnly | QIODevice::Text) )
			syncError( "Could not create file "+outputPath+"/SynsetWords.txt" );

		QTextStream synsetStream( &synsetFile );
		synsetStream << this->categories();
	}

	// --------------------------------------------------
	// GETTERS - SETTERS

	const QString&		IndexManager::workspacePath	() const	{ return this->_workspacePath;	}
	QString&			IndexManager::workspacePath	()			{ return this->_workspacePath;	}
	const QString&		IndexManager::outputPath	() const	{ return this->_outputPath;		}
	QString&			IndexManager::outputPath	()			{ return this->_outputPath;		}
	const Categories&	IndexManager::categories	() const	{ return this->_categories;		}
	Categories&			IndexManager::categories	()			{ return this->_categories;		}
	const IndexFiles&	IndexManager::indexFiles	() const	{ return this->_indexFiles;		}
	IndexFiles&			IndexManager::indexFiles	()			{ return this->_indexFiles;		}

	void IndexManager::setCategories ( const Categories& categories )
	{
		this->categories() = categories;
	}

	const IndexFilePtr	IndexManager::indexFileByName ( const QString& name ) const	{ return this->indexFiles().at(name);	}	
	IndexFilePtr		IndexManager::indexFileByName ( const QString& name )		{ return this->indexFiles().at(name);	}

	const IndexFilePtr IndexManager::indexFileByMode ( const SamplingMode& mode ) const
	{
		if ( mode == SamplingMode::Train() )
			return this->indexFileByName( (QString)mode );
		
		else if ( mode == SamplingMode::Validation() )
			return this->indexFileByName( (QString)mode );
		
		return this->indexFileByName( (QString)mode );
	}

	IndexFilePtr IndexManager::indexFileByMode ( const SamplingMode& mode )
	{
		if ( mode == SamplingMode::Train() )
			return this->indexFileByName( "Index"+(QString)mode );
		
		else if ( mode == SamplingMode::Validation() )
			return this->indexFileByName( "Index"+(QString)mode );
		
		return this->indexFileByName( "Index"+(QString)mode );
	}

	void IndexManager::addIndexFile ( const QString& name )
	{
		IndexFilePtr index = std::make_shared< IndexFile >( this->outputPath()+"/"+name+".txt" );
		
		for ( const QString& category : this->categories() )
			index->addCounter( this->categories().labelOfCategory(category) );

		this->indexFiles().emplace( name, index );
	}

	void IndexManager::addIndexFile ( const SamplingMode& mode )
	{
		this->addIndexFile( "Index"+(QString)mode );
	}

	void IndexManager::addIndex ( const SamplingMode& mode, const QString& fullpath, const QString& category )
	{
		this->indexFileByMode(mode)->addIndex(
			fullpath,
			this->categories().labelOfCategory(category)
		);
	}

	void IndexManager::addIndex ( const QString& projectName, const QString& category, int timestamp, const QString& fullpath )
	{
		this->indexFileByMode(SamplingMode::Test())->addIndex(
			projectName,
			category,
			this->categories().labelOfCategory(category),
			timestamp,
			fullpath
		);
	}

	// END CLASS INDEXMANAGER
	// ##################################################

	QTextStream& operator<< ( QTextStream& stream, const IndexManager& indexManager )
	{
		stream
			<< "Categories" << endl
			<< indexManager.categories();

		for ( auto entry : indexManager.indexFiles() )
			stream << entry.first << endl << *entry.second;

		return stream;
	}

} // END NAMESPACE SAMPLING
// ##################################################

#endif // INDEXMANAGER_CPP
// ##################################################