#include "AnnotationParser.hpp"
#include "SMIDataParser.hpp"
#include "Stats.hpp"
namespace SMI
{
static Tobii::Project createProject(const std::string& name, const std::string& path){
  QString projectName = QString::fromStdString(path) + QString::fromStdString(name);
  QStringList streamNames;
  streamNames
      << Tobii::Project::Video
      << Tobii::Project::Saliency
      /*<< Tobii::Project::Annotation*/;
  
  Tobii::Project prj;
  prj.open(projectName, true, streamNames);
  std::cout << "Project " << name << " created." << std::endl;
  return prj;
}

static void skipFrames(Tobii::InputVideoStream& inputStream, int from, int to)
{
  if(from == to) { return; }

  std::cout << "Skipping unsed frames ..." << std::endl;
  int cptTmp(from);
  while(cptTmp < to)
  {
    inputStream.getNextFrame();
    cptTmp++;
  }
  std::cout << "Skipping finished successfully." << std::endl;
}


static unsigned long msToFrameId(unsigned long time, int fps, bool sup) 
{
  if(sup)
  {
    return ceil(time*fps/1000);
  }
  return time*fps/1000; 
}

static std::string nameWithoutExt(std::string name)
{
  std::string fileName = name.substr(name.find_last_of('/')+1, name.length());
  return fileName.substr(0, fileName.find_last_of("."));
}
}

int main(int argc, char** argv)
{
  using namespace SMI;

  if( argc != 5){
    std::cerr << "usage: annotation file : ." << SMI_DATA_FILE_EXTENSION 
    << " video file : ." << VIDEO_FILE_EXTENSION 
    << "SMI Eyetracker data file : ." << SMI_DATA_FILE_EXTENSION
    << "& result folder (without /)" 
    << std::endl;
    return EXIT_FAILURE;
  }

  std::string annotationFile(argv[1]);
  std::string videoName(argv[2]);
  std::string fixationData(argv[3]);
  std::string path(argv[4]);

  if(   !checkExtension(videoName, VIDEO_FILE_EXTENSION)
     || !checkExtension(fixationData, SMI_DATA_FILE_EXTENSION))
  {
    std::cerr << "[ERROR] bad file extension: (." <<  SMI_DATA_FILE_EXTENSION
              << "|data) or (." << VIDEO_FILE_EXTENSION
              << "|video)" << std::endl;
    return EXIT_FAILURE;
  }

  /** 
     GETTING VIDEO DATAS
  **/
  Tobii::InputVideoStream inputStream;
  inputStream.open(QString::fromStdString(videoName));

  int vHeight(inputStream.resolution().height);
  int vWidth(inputStream.resolution().width);
  int fps(inputStream.fps());
  int frameCount(inputStream.frameCount());

  std::cout << "[***VIDEO DATA***]" << std::endl
            << "   *name : " << videoName << std::endl
            << "   *resolution :" << vWidth << "x" << vHeight << std::endl 
            << "   *fps : " << fps << std::endl
            << "   *total fps count :" << inputStream.frameCount() << std::endl
            << "[****************]" << std::endl;

  std::ifstream file(annotationFile);
  if(!file.is_open())
  {
    std::cerr << "Error while we were trying to open " << argv[1] << std::endl;
    return EXIT_FAILURE;
  }
  //------------- ANNOTATIONS PARSER --------------
  Action action;
  std::vector<Action> actionVector;
  std::string line;
  while(std::getline(file, line))
  {
    action = parseAction(line);
    actionVector.push_back(action);
  }
  std::sort(actionVector.begin(), actionVector.end());
  //----------------------------------------------

  //------------ RESULT DIR ---------------
  QDir dir(QString::fromStdString(path));
  if(!dir.exists()) { dir.mkpath(QString::fromStdString(path)); }
  //---------------------------------------

  //------------- STATS PROCESSING ----------------
  Stats stats(frameCount);
  for(auto& act : actionVector)
  {
    int frameStart = msToFrameId(act.startTime(), fps, false);
    int frameEnd = msToFrameId(act.endTime(), fps, true);
    stats.insert(act.object().toStdString(), frameEnd - frameStart);
  }
  stats.processStats(videoName, path + '/' + nameWithoutExt(videoName) + std::string("_stats.txt"));
  //------------------------------------------------

  //------------- FIXATION PARSER ---------------
  Tobii::GazeRecordings gazeRecordings(parseMe(fixationData, vWidth, vHeight)); 
  //---------------------------------------------- 

  int lastFrameRead(0);
  int nextFrameToRead(msToFrameId(actionVector[0].startTime(), fps, false));
  for(auto &act :actionVector)
  {
    std::string categoryName(act.object().toStdString());
    std::string projectName = path + '/' + nameWithoutExt(videoName) + '_' + categoryName;
    
    int cpt(0);
    while(QDir(QString::fromStdString(projectName)).exists())
    {
      cpt++;
      projectName = path + '/' + nameWithoutExt(videoName) + '_' + categoryName + std::to_string(cpt); 
    }
    Tobii::Project project = createProject(projectName, "");

    std::string ext(".txt");
    std::string fullName = path + '/' + nameWithoutExt(videoName) + ext;
    std::ofstream categoryFile(std::string(path + '/' + nameWithoutExt(videoName) + ext), std::ios_base::app);
    if(!categoryFile)
    {
      std::cerr << "Couldn't open Categories file " << fullName << std::endl;
      return EXIT_FAILURE;
    }

    /** Each time I'll skip (perhaps) non-used frame by saving the last frame read index, 
        and by getting the nextFrameToRead index (ie frameStart) for this loop
    **/

    int frameStart = msToFrameId(act.startTime(), fps, false);
    int frameEnd = msToFrameId(act.endTime(), fps, true);
    
    //-------------------- FAST FORWARDING ------------------------
    nextFrameToRead = frameStart;
    skipFrames(inputStream, lastFrameRead, nextFrameToRead);
    lastFrameRead = frameEnd;
    //-----------------------------------------------------------

    bool isEmpty(true);
    for(int i = frameStart ; i < frameEnd ; i++)
    {
       Tobii::Frame vFrame = inputStream.getNextFrame();
       if(vFrame.size().width && vFrame.size().height)
       {   
        Tobii::GazeFixation gf = Tobii::Interpolate::live(gazeRecordings, i, 40, true, false);
        Tobii::Frame saliencyFrame(doSaliencyMap(gf, vWidth, vHeight));
        Tobii::FramesMap framesMap;
        framesMap.emplace(Tobii::Project::Video, vFrame);
        framesMap.emplace(Tobii::Project::Saliency, saliencyFrame);
        project.addFrames(framesMap);
        isEmpty = false;
      }
    }

    if(isEmpty) { project.setCategory("Bad"); }
    else        { project.setCategory(act.object()); }

    categoryFile << act.object().toStdString() << std::endl; // add it anyway to the catFile
  }
}
