// ###################################################################### //
// ###                           DEPENDANCES                          ### //
// ###################################################################### //

#include <algorithm>
#include <iosfwd>
#include <memory>
#include <string>
#include <utility>
#include <vector>
#include <map>
#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <caffe/caffe.hpp>

// ###################################################################### //
// ###                           NAMESPACES                           ### //
// ###################################################################### //

using namespace caffe;  // NOLINT(build/namespaces)

using cv::Scalar;
using cv::Size;

// ###################################################################### //
// ###                            TYPEDEFS                            ### //
// ###################################################################### //

using String   = std::string;
using IFStream = std::ifstream;
using OFStream = std::ofstream;
using OStream  = std::ostream;

template< typename T >
using Array = std::vector<T>;

template< typename T1, typename T2 >
using Pair = std::pair<T1,T2>;

template< typename T1, typename T2 >
using HashMap = std::map< T1, T2 >;

using Scores   = Array< float >;
using Matrix   = cv::Mat;
using Matrices = Array< Matrix >;
using Indices  = Array< int >;
using Labels   = Array< String >;
using Network  = Net < float >;

using NetworkPtr = std::shared_ptr< Network >;

OStream& operator<< ( OStream& stream, const Scores& scores );

// ###################################################################### //
// ###                           PREDICTION                           ### //
// ###################################################################### //

struct Prediction
{
	float score;
	union
	{
		int label;
		const char* category;
	};

	Prediction ( float score, int label );
	Prediction ( float score, String category );
};

using Predictions = Array< Prediction >;

OStream& operator<< ( OStream& stream, const Prediction&  p  );
OStream& operator<< ( OStream& stream, const Predictions& ps );

// ###################################################################### //
// ###                             QUERY                              ### //
// ###################################################################### //

class Query
{
public:
	Query ( const String& fullpath = "", int timestamp = 0 );

	String	fullpath;
	int		timestamp;
	Scores	scores;
	Indices results;
};
using Queries = Array< Query >;

OStream& operator<< ( OStream& stream, const Query& query );
OStream& operator<< ( OStream& stream, const Queries& queries );

class VideoQuery
{
public:
	VideoQuery ( const String& name, const String& category, int label );

	String	name;
	String	category;
	int		label;
	Queries queries;
	Scores	summedScores;
};

using VideoQueries = HashMap< String, VideoQuery >;

OStream&  operator<< ( OStream& stream, const VideoQuery& query );
IFStream& operator>> ( IFStream& stream, VideoQueries& queries );


// ###################################################################### //
// ###                        CONFUSION MATRIX                        ### //
// ###################################################################### //

class ConfusionMatrix : public Array< float >
{
public:
	int nbCategories;
	ConfusionMatrix ( int nbCategories = 0 );
	ConfusionMatrix ( const ConfusionMatrix& mat );
	ConfusionMatrix& operator= ( const ConfusionMatrix& mat );

	void normalize ();

	float  count ( int label, int predicted ) const;
	float& count ( int label, int predicted );

	float ap  ( int label ) const;
	float map () const;
};

OStream& operator<< ( OStream& stream, const ConfusionMatrix& matrix );

// ###################################################################### //
// ###                           CLASSIFIER                           ### //
// ###################################################################### //

class Classifier
{
public:
	Classifier ( const String& model_file, const String& trained_file, const String& mean_file, const String& label_file );
	Predictions Classify ( const Matrix& img, int N = 10 );

	void SetMean ( const string& mean_file );
	Scores Predict ( const Matrix& img );
	void WrapInputLayer ( Matrices* input_channels );
	void Preprocess ( const Matrix& img, Matrices* input_channels );

	static bool PredictionCompare ( const Prediction& lhs, const Prediction& rhs );
	static Indices Argmax ( const Scores& scores );

	NetworkPtr network;
	Size input_geometry;
	int num_channels;
	Matrix mean;
	Labels labels;
};

// ###################################################################### //
// ###                               MAIN                             ### //
// ###################################################################### //

#define ARG( id, def ) (id < argc ? argv[id] : def)

int main ( int argc, char** argv )
{
	if ( argc < 5 )
	{
		std::cerr << "Usage: " << argv[0] << " [deploy.prototxt] [network.caffemodel] [mean.binaryproto] [labels.txt] [img.jpg / list.txt] [output.txt]" << std::endl;
		return 1;
	}

	::google::InitGoogleLogging( argv[0] );

	String model_file   ( argv[1] );
	String trained_file ( argv[2] );
	String mean_file    ( argv[3] );
	String label_file   ( argv[4] );
	String output_dir = ARG( 5, "./result.txt" );

	std::cout << "----------------------------------------------------------------------" << std::endl;
	std::cout << "> Loading classifier: ... " << std::flush;
	Classifier classifier( model_file, trained_file, mean_file, label_file );
	std::cout << "done!" << std::endl;

	String filename = argv[5];

	size_t pos = filename.find_last_of(".");
	String extention = filename.substr( pos+1 );

	// ---------------------------------------------------------------------- //
	// SINGLE IMAGE PREDICTION
	if ( extention == "jpg" || extention == "png" )
	{
		std::cout << "> Prediction for image \"" << filename << "\":" << std::endl;
		std::cout << "> Loading image ... ";

		Matrix img = cv::imread( filename, -1 );
		if( img.empty() )
		{
			std::cout << "failed: could not open file !" << std::endl;
			std::exit(EXIT_FAILURE);
		}
		else
			std::cout << "done !" << std::endl;

		std::cout << "> Computing predictions ... ";
		Predictions predictions = classifier.Classify( img );
		std::cout << "done !" << std::endl;

		// Print the top N predictions
		std::cout << predictions << std::endl;
	}

	// ---------------------------------------------------------------------- //
	// LIST FILE PREDICTION
	if ( extention == "txt" )
	{
		std::cout << "> Prediction for list \"" << filename << "\":" << std::endl;
		
		// OPEN QUERY LIST FILE
		std::cout << "> Opening query list file ... ";
		IFStream listFile( filename );
		if ( ! listFile )
		{
			std::cout << "failed: could not open file !" << std::endl;
			std::exit(EXIT_FAILURE);
		}
		std::cout << "done !" << std::endl;

		// OPEN OUTPUT TXT CONFUSION MATRIX 1
		String namePatchConfu ( "./ConfusionPatch.txt" );
		if ( argc >= 7 )
			namePatchConfu = argv[6];

		std::cout << "> Opening output file \""<< namePatchConfu << "\" ... ";
		OFStream txtPatchConfu( namePatchConfu );
		if ( ! txtPatchConfu.is_open() )
		{
			std::cout << "failed !" << std::endl;
			std::exit(EXIT_FAILURE);
		}
		else
			std::cout << "done !" << std::endl;

		// OPEN OUTPUT TXT CONFUSION MATRIC 2
		String nameVideoConfu ( "./ConfusionVideo.txt" );
		if ( argc >= 8 )
			nameVideoConfu = argv[7];

		std::cout << "> Opening output file \""<< nameVideoConfu << "\" ... ";
		OFStream txtVideoConfu( nameVideoConfu );
		if ( ! txtVideoConfu.is_open() )
		{
			std::cout << "failed !" << std::endl;
			std::exit(EXIT_FAILURE);
		}
		else
			std::cout << "done !" << std::endl;

		// LOAD QUERY LIST
		std::cout << "> Laoding query list ... "; 
		VideoQueries queries;
		listFile >> queries;
		listFile.close();
		int nvVideo = queries.size();
		std::cout << "done ! " << nvVideo << " video queries" << std::endl;

		// CLASSIFY ALL OF THEM:
		std::cout << "> Classifying:" << std::endl;

		ConfusionMatrix patchConfusion( classifier.labels.size() );
		ConfusionMatrix videoConfusion( classifier.labels.size() );

		int idVideo(0);
		for ( auto videoEntry : queries )
		{
			VideoQuery& videoQuery = videoEntry.second;
			int         label      = videoQuery.label;
			String      category   = videoQuery.category;

			std::cout << "> Video \"" << videoQuery.name << "\", category: \"" << category << "\" (" << idVideo+1 << "/" << nvVideo << "), progress: " << std::endl;
			std::cout << ">    ";

			// Summed scores over video:
			videoQuery.summedScores = Scores(classifier.labels.size(), 0);

			// Classify queries:
			int idQuery(0);
			int nbQuery(videoQuery.queries.size());
			int progress(0);
			for ( Query& imageQuery : videoQuery.queries )
			{
				// Open the image
				Matrix img = cv::imread( imageQuery.fullpath );
				if( img.empty() )
				{
					std::cerr << "failed: could not open file: " << imageQuery.fullpath << " !" << std::endl;
					std::exit(EXIT_FAILURE);
				}

				// Predict
				imageQuery.scores  = classifier.Predict( img );
				imageQuery.results = Classifier::Argmax( imageQuery.scores );
				int max_label      = imageQuery.results[0];

				// Summed scores over video:
				for ( int id(0) ; id < imageQuery.scores.size() ; ++id )
					videoQuery.summedScores.at(id) += imageQuery.scores.at(id);

				patchConfusion.count( label, max_label ) += 1.0f;

				// Progress manager
				++idQuery;
				int new_progress = idQuery / (float)nbQuery * 100.0f;
				if ( new_progress > progress )
				{
					progress = new_progress;
					std::cout << progress << ' ';
					if ( fmod(progress, 10) == 0 && progress != 100 )
						std::cout << std::endl << ">    ";

				}

			} // end for videoQueries.queries

			// Fusion over the video:
			int max_label = Classifier::Argmax(videoQuery.summedScores)[0];
			videoConfusion.count( label, max_label ) += 1.0f;

			std::cout << std::endl;
			std::cout << ">    scores:" << videoQuery.summedScores << ", max = " << videoQuery.summedScores.at(max_label) << std::endl;
			std::cout << ">    max category: " << classifier.labels.at(max_label) << (max_label == label ? ", Good" : ", Bad") << std::endl;
			std::cout << ">" << std::endl;

			++idVideo;
		} // end for videoQueries
		std::cout << "> done !" << std::endl;
		std::cout << ">" << std::endl;

		int nbGoodPatch(0);
		int nbGoodVideo(0);
		for ( int label(0) ; label < patchConfusion.nbCategories ; ++label )
		{
			nbGoodPatch += patchConfusion.count(label, label);
			nbGoodVideo += videoConfusion.count(label, label);
		}
		int nbPatchs = std::accumulate( patchConfusion.begin(), patchConfusion.end(), 0 );
		int nbVideos = std::accumulate( videoConfusion.begin(), videoConfusion.end(), 0 );

		patchConfusion.normalize();
		videoConfusion.normalize();
		
		std::cout << "> Patch confusion matrix:" << std::endl;
		std::cout << patchConfusion;
		std::cout << ">     mAp: " << patchConfusion.map() << std::endl;
		std::cout << ">     result: " << nbGoodPatch << " / " << nbPatchs << " = " << (float)nbGoodPatch / (float)nbPatchs * 100.f << std::endl;
		std::cout << ">" << std::endl;
		std::cout << "> Video confusion matrix:" << std::endl;
		std::cout << videoConfusion;
		std::cout << ">     mAp: " << videoConfusion.map() << std::endl;
		std::cout << ">     result: " << nbGoodVideo << " / " << nbVideos << " = " << (float)nbGoodVideo / (float)nbVideos * 100.f << std::endl;
		std::cout << ">" << std::endl;
		
		txtPatchConfu << patchConfusion;
		txtPatchConfu << "mAp\t" << patchConfusion.map() << std::endl;
		txtPatchConfu << "nbGood\tnbTotal\tRatio" << std::endl << nbGoodPatch << '\t' << nbPatchs << '\t' << (float)nbGoodPatch / (float)nbPatchs << std::endl;
		txtPatchConfu.close();
		txtVideoConfu << videoConfusion;
		txtVideoConfu << videoConfusion.map()<< std::endl;
		txtVideoConfu << "nbGood\tnbTotal\tRatio" << std::endl << nbGoodVideo << '\t' << nbVideos << '\t' << (float)nbGoodVideo / (float)nbVideos << std::endl;
		txtVideoConfu.close();

	} // end if txt
}

OStream& operator<< ( OStream& stream, const Scores& scores )
{
	stream << '[';
	for ( int id(0) ; id < scores.size()-1 ; ++id )
		stream << scores.at(id) << ", ";
	return stream << scores.back() << ']';
}

// ###################################################################### //
// ###                           PREDICTION                           ### //
// ###################################################################### //

Prediction::Prediction( float _score, int _label )
: score( _score ), label( _label )
{}

Prediction::Prediction( float _score, String _category )
: score( _score ), category( _category.c_str() )
{}

OStream& operator<< ( OStream& stream, const Prediction& p )
{
	return stream << std::fixed << std::setprecision(4) << p.score << " - \"" << p.category << "\"";
}

OStream& operator<< ( OStream& stream, const Predictions& ps )
{
	for ( const Prediction& p : ps )
		stream << '\t' << p << std::endl;
	return stream;
}

// ###################################################################### //
// ###                             QUERY                              ### //
// ###################################################################### //

Query::Query ( const String& _fullpath, int _timestamp )
: fullpath( _fullpath ), timestamp( _timestamp )
{}

OStream& operator<< ( OStream& stream, const Query& query )
{
	return stream << query.fullpath << query.timestamp;
}

OStream& operator<< ( OStream& stream, const Queries& queries )
{
	stream << '[';

	for ( int id(0) ; id < queries.size()-1 ; ++id )
		stream << queries.at(id) << ", ";

	return stream << queries.back() << ']';
}

VideoQuery::VideoQuery ( const String& _name, const String& _category, int _label )
: name( _name ), category( _category ), label( _label )
{}

OStream& operator<< ( OStream& stream, const VideoQuery& query )
{
	return stream << '(' << query.name << ", " << query.category << ", " << query.label << ", " << query.queries.size() << ')';
}

IFStream& operator>> ( IFStream& stream, VideoQueries& queries )
{
	queries.clear();

	while ( stream.good() )
	{
		// Read a line
		String videoName;
		String category;
		int label;
		int timestamp;
		String fullpath;
		stream >> videoName;

		if ( ! videoName.empty() && videoName[0] != '#' )
		{
			stream >> category >> label >> timestamp >> fullpath;

			// find the video if it exist:
			if ( queries.find(videoName) == queries.end() )
				queries.emplace( videoName, VideoQuery(videoName, category, label) );

			queries.at(videoName).queries.push_back( Query(fullpath,timestamp) );
		}
	}
	return stream;
}

// ###################################################################### //
// ###                        CONFUSION MATRIX                        ### //
// ###################################################################### //


ConfusionMatrix::ConfusionMatrix ( int _nbCategories )
: Array< float >( _nbCategories*_nbCategories, 0 ), nbCategories( _nbCategories )
{}

ConfusionMatrix::ConfusionMatrix ( const ConfusionMatrix& mat )
: Array< float >( mat ), nbCategories( mat.nbCategories )
{}

ConfusionMatrix& ConfusionMatrix::operator= ( const ConfusionMatrix& mat )
{
	this->Array< float >::operator=( mat );
	this->nbCategories = mat.nbCategories;
	return (*this);
}

float  ConfusionMatrix::count ( int label, int predicted ) const	{ return this->at( label + predicted * this->nbCategories );	}
float& ConfusionMatrix::count ( int label, int predicted )			{ return this->at( label + predicted * this->nbCategories );	}

void ConfusionMatrix::normalize ()
{
	for ( int label(0) ; label < this->nbCategories ; ++label )
	{
		auto it1 = this->begin() + label     * this->nbCategories;
		auto it2 = this->begin() + (label+1) * this->nbCategories;
		float sum = std::accumulate(it1, it2, 0);

		if ( sum != 0 && sum != 1 )
		{
			while ( it1 <= it2 )
			{
				*it1 /= sum;
				++it1;
			}
		}
	}
}

float ConfusionMatrix::ap  ( int label ) const
{
	return this->count( label, label );
}

float ConfusionMatrix::map () const
{
	float map = 0;
	for ( int label(0) ; label < this->nbCategories ; ++label )
		map += this->ap( label ); 
	return map/this->nbCategories;
}

OStream& operator<< ( OStream& stream, const ConfusionMatrix& matrix )
{
	stream << "Label";
	for ( int label(0) ; label < matrix.nbCategories ; ++label )
		stream << '\t' << label;
	stream << "\tAP" << std::endl;

	for ( int label(0) ; label < matrix.nbCategories ; ++label )
	{
		stream << label;
		for ( int predicted(0) ; predicted < matrix.nbCategories ; ++predicted )
		{
			stream << '\t' << matrix.count( label, predicted );
		}
		stream << '\t' << matrix.ap(label) << std::endl;
	}
	return stream;
}

// ###################################################################### //
// ###                           CLASSIFIER                           ### //
// ###################################################################### //

Classifier::Classifier ( const String& model_file, const String& trained_file, const String& mean_file, const String& label_file )
{
#ifdef CPU_ONLY
	Caffe::set_mode( Caffe::CPU );
#else
	Caffe::set_mode( Caffe::GPU );
#endif

	/* Load the network. */
	this->network.reset(new Net<float>(model_file, TEST));
	this->network->CopyTrainedLayersFrom(trained_file);

	CHECK_EQ( this->network->num_inputs(),  1 ) << "Network should have exactly one input.";
	CHECK_EQ( this->network->num_outputs(), 1 ) << "Network should have exactly one output.";

	Blob<float>* input_layer = this->network->input_blobs()[0];

	this->num_channels = input_layer->channels();
	CHECK( this->num_channels == 3 || this->num_channels == 1 ) << "Input layer should have 1 or 3 channels.";

	this->input_geometry = Size( input_layer->width(), input_layer->height() );

	/* Load the binaryproto mean file. */
	this->SetMean( mean_file );

	/* Load labels. */
	IFStream labels( label_file.c_str() );
	CHECK(labels) << "Unable to open labels file " << label_file;

	String line;
	while (std::getline(labels, line))
	{
		this->labels.push_back(string(line));
	}

	Blob<float>* output_layer = this->network->output_blobs()[0];
	CHECK_EQ( this->labels.size(), output_layer->channels()) << "Number of labels is different from the output layer dimension.";
}

/* Return the top N predictions. */
Predictions Classifier::Classify( const Matrix& img, int N )
{
	Scores scores = this->Predict( img );

	N = std::min<int>( this->labels.size(), N );

	Indices maxN = Argmax( scores );
	Predictions predictions;

	for ( int i = 0 ; i < N ; ++i )
	{
		int idx = maxN[i];
		predictions.push_back( Prediction(scores[idx], this->labels[idx]) );
	}

	return predictions;
}

/* Load the mean file in binaryproto format. */
void Classifier::SetMean( const string& mean_file )
{
	BlobProto blob_proto;
	ReadProtoFromBinaryFileOrDie( mean_file.c_str(), &blob_proto );

	/* Convert from BlobProto to Blob<float> */
	Blob<float> mean_blob;
	mean_blob.FromProto( blob_proto );
	CHECK_EQ(mean_blob.channels(), this->num_channels) << "Number of channels of mean file doesn't match input layer.";

	/* The format of the mean file is planar 32-bit float BGR or grayscale. */
	Matrices channels;
	float* data = mean_blob.mutable_cpu_data();
	for ( int i = 0; i < this->num_channels; ++i )
	{
		/* Extract an individual channel. */
		Matrix channel( mean_blob.height(), mean_blob.width(), CV_32FC1, data );
		channels.push_back( channel );
		data += mean_blob.height() * mean_blob.width();
	}

	/* Merge the separate channels into a single image. */
	Matrix mean;
	cv::merge( channels, mean );

	/* Compute the global mean pixel value and create a mean image
	* filled with this value. */
	Scalar channel_mean = cv::mean(mean);
	this->mean = cv::Mat(this->input_geometry, mean.type(), channel_mean);
}

Scores Classifier::Predict( const cv::Mat& img )
{
	Blob<float>* input_layer = this->network->input_blobs()[0];
	input_layer->Reshape( 1, this->num_channels, this->input_geometry.height, this->input_geometry.width );

	/* Forward dimension change to all layers. */
	this->network->Reshape();

	Matrices input_channels;
	this->WrapInputLayer( &input_channels );
	this->Preprocess( img, &input_channels );

	this->network->Forward();

	/* Copy the output layer to a std::vector */
	Blob<float>* output_layer = this->network->output_blobs()[0];
	const float* begin        = output_layer->cpu_data();
	const float* end          = begin + output_layer->channels();

	return Scores( begin, end );
}

/* Wrap the input layer of the network in separate cv::Mat objects
 * (one per channel). This way we save one memcpy operation and we
 * don't need to rely on cudaMemcpy2D. The last preprocessing
 * operation will write the separate channels directly to the input
 * layer. */
void Classifier::WrapInputLayer ( Matrices* input_channels )
{
	Blob<float>* input_layer = this->network->input_blobs()[0];

	int    width      = input_layer->width();
	int    height     = input_layer->height();
	float* input_data = input_layer->mutable_cpu_data();

	for ( int i = 0; i < input_layer->channels(); ++i )
	{
		Matrix channel( height, width, CV_32FC1, input_data );
		input_channels->push_back( channel );
		input_data += width * height;
	}
}

void Classifier::Preprocess( const Matrix& img, Matrices* input_channels )
{
	/* Convert the input image to the input image format of the network. */
	Matrix sample;
	if (img.channels() == 3 && this->num_channels == 1)
		cv::cvtColor(img, sample, cv::COLOR_BGR2GRAY);
	else if (img.channels() == 4 && this->num_channels == 1)
		cv::cvtColor(img, sample, cv::COLOR_BGRA2GRAY);
	else if (img.channels() == 4 && this->num_channels == 3)
		cv::cvtColor(img, sample, cv::COLOR_BGRA2BGR);
	else if (img.channels() == 1 && this->num_channels == 3)
		cv::cvtColor(img, sample, cv::COLOR_GRAY2BGR);
	else
		sample = img;

	Matrix sample_resized;
	if (sample.size() != this->input_geometry)
		cv::resize(sample, sample_resized, this->input_geometry);
	else
		sample_resized = sample;

	Matrix sample_float;
	if (this->num_channels == 3)
		sample_resized.convertTo(sample_float, CV_32FC3);
	else
		sample_resized.convertTo(sample_float, CV_32FC1);

	Matrix sample_normalized;
	cv::subtract(sample_float, this->mean, sample_normalized);

	/* This operation will write the separate BGR planes directly to the
	* input layer of the network because it is wrapped by the cv::Mat
	* objects in input_channels. */
	cv::split( sample_normalized, *input_channels );

	CHECK( reinterpret_cast<float*>( input_channels->at(0).data ) == this->network->input_blobs()[0]->cpu_data() ) << "Input channels are not wrapping the input layer of the network.";
}

bool Classifier::PredictionCompare( const Prediction& lhs, const Prediction& rhs )
{
	return lhs.score > rhs.score;
}

Indices Classifier::Argmax( const Scores& scores )
{
	Predictions predictions;
	
	for ( int i = 0 ; i < scores.size() ; ++i )
		predictions.push_back( Prediction( scores[i], i ) );

	std::sort( predictions.begin(), predictions.end(), Classifier::PredictionCompare );

	Indices result;
	
	for ( Prediction& prediction : predictions )
		result.push_back( prediction.label );
	
	return result;
}
