#include "TobiiSDK/Stream/Project.hpp"
#include <iostream>
#include <QDir>
#include <QString>

using namespace Tobii;

#define EXIT_IF( condition, message ) do{ if( condition == true ){ std::cout << message << std::endl; std::exit(EXIT_FAILURE); } }while(0)
#define GetArg( index, defaultValue ) ( (index < argc) ? QString(argv[index]) : QString(defaultValue) )

std::ostream& operator<<( std::ostream& stream, const cv::Size& size );
std::ostream& operator<<( std::ostream& stream, const QString& string );
std::ostream& operator<<( std::ostream& stream, const QStringList& lst );

void CHECK_TIMESTAMP  ( Frame frame, const QString& name, Int64 timeStamp );
void CHECK_MATRIXDATA ( Frame frame, const QString& name );
void CHECK_RESOLUTION ( Frame frame, const QString& name, const cv::Size& resolution );

int main ( const int argc, const char* argv[] )
{
	// Usage ?
	EXIT_IF( argc < 1, "usage: DataSetManager <dataset-folder-path> <check frames=y/n>" );

	// Arguments
	QString	folderPath( GetArg( 1, "./Workspace/Dataset" ) );
	bool	checkDatas( GetArg( 2, "y" ) == "y"            );

	// Open the dataset folder
	QDir datasetFolder( folderPath );

	// Check it
	EXIT_IF( !datasetFolder.exists(),     QString("Dataset folder \"%1\" does not exist."         ).arg(folderPath).toStdString() );
	EXIT_IF(  datasetFolder.count() == 2, QString("Dataset folder \"%1\" is empty, nothing to do.").arg(folderPath).toStdString() );

	QFileInfoList lst = datasetFolder.entryInfoList(QDir::Dirs|QDir::NoDotAndDotDot);
	for ( const QFileInfo& entry : lst )
	{
		Tobii::Project project( entry.filePath() );
		std::cout
			<< "-------------------------------------------------------" << std::endl
			<< project.name()		<< std::endl
			<< "\tCategories   : "	<< project.categories()	<< std::endl
			<< "\tStream names : "	<< project.streamNames()<< std::endl
			<< "\tFrame counts : ";
			for ( auto& stream : project.streams() )
				std::cout << stream.second->frameCount() << ' ';
		std::cout << std::endl	
			<< "\tResolution   : "	<< project.resolution()	<< std::endl
			<< std::endl;
		
		if ( checkDatas )
		{
			std::cout
				<< "-------------------------------------------------------" << std::endl
				<< "\tCHECKING FRAMES:" << std::endl
				<< std::endl;

			// Sample each frame of the project:
			int nbInstanceFrames( 0 );
			while ( project.currentTimeStamp() <= project.lastTimeStamp() )
			{
				// Read a set of frame+saliency+annotation
				Int64     timeStamp = project.currentTimeStamp();
				FramesMap frames    = project.getCurrentFrames();

				std::cout << project.name() << " : Frames at " << timeStamp << "/" << project.lastTimeStamp() << "ms (" << project.currentIndex() << "/" << project.lastIndex() << "):" << std::endl;
				
				for ( const QString& streamName : project.streamNames() )
				{
					// Checking timestamps:
					CHECK_TIMESTAMP( frames.at(streamName), streamName, timeStamp );
					// Checking matrix data:
					CHECK_MATRIXDATA( frames.at(streamName), streamName );
					// Checking resolution:
					CHECK_RESOLUTION( frames.at(streamName), streamName, project.resolution() );
				}

				// Sample next frame if any
				project.next();
				std::cout << std::endl;
			}
		}
	}
	return 0;
}

void CHECK_TIMESTAMP ( Frame frame, const QString& name, Int64 timeStamp )
{
	if ( frame.timeStamp() != timeStamp )
	{
		std::cout << name << "\t: Time stamps FAIL : " << frame.timeStamp() << " !" << timeStamp << std::endl;
		std::exit( EXIT_FAILURE );
	}
	else
		std::cout << name << "\t: Time stamps OK" << std::endl;
}

void CHECK_MATRIXDATA ( Frame frame, const QString& name )
{
	if ( frame.empty() )
	{
		std::cout << name << "\t: Frame could not be read !" << std::endl;
		std::exit( EXIT_FAILURE );
	}
	else
		std::cout << name << "\t: Frame read \tOK" << std::endl;
}

void CHECK_RESOLUTION ( Frame frame, const QString& name, const cv::Size& resolution )
{
	if ( frame.size() != resolution )
	{
		std::cout << name << "\t: Frame resolution, FAIL : " << frame.size() << " != " << resolution << " !" << std::endl;
		std::exit(EXIT_FAILURE);
	}
	else
		std::cout << name << "\t: Frame resolution,\tOK" << std::endl;
}

std::ostream& operator<<( std::ostream& stream, const cv::Size& size )
{
	stream << size.width << "x" << size.height;
	return stream;
}

std::ostream& operator<<( std::ostream& stream, const QString& string )
{
	stream << string.toStdString();
	return stream;
}

std::ostream& operator<<( std::ostream& stream, const QStringList& lst )
{
	stream << "[";
	for ( int index(0) ; index < lst.size()-1 ; ++index )
	{
		stream << lst.at(index) << ", ";
	}
	stream << lst.at(lst.size()-1) << "]";
	return stream;
}