// ##################################################
// ###             TOBII_MAINWINDOW_CPP           ###
// ##################################################
#ifndef TOBII_MAINWINDOW_CPP
#define TOBII_MAINWINDOW_CPP

// ##################################################
// ###                 DEPENDENCES                ###
// ##################################################

#include "MainWindow.hpp"

// ##################################################
// ###              NAMESPACE TOBII               ###
// ##################################################

namespace Tobii {

	// ##################################################
	// ###             CLASS MAINWINDOW               ###
	// ##################################################
	
	// --------------------------------------------------
	// CONSTRUCTOR - DESTRUCTOR
	
	MainWindow::MainWindow ()
	: QMainWindow()
	, _workspace(  this->settings().value( "workspace", QDir::homePath() ).toString() )
	, _categoryFile( this->settings().value( "categories" ).toString() )
	, _ptrStartPage( new StartPage( this->workspace(), this->categoryFile(), this ) )
	{
		this->setupUi( this );
		this->centralWidget()->hide();
		this->setTabPosition( Qt::TopDockWidgetArea, QTabWidget::North );
		this->setWindowTitle( "Annotation Tool" );

		// MENU BAR
		this->initMenuProject();
		this->initMenuEdit();

		// START PAGE
		connect( this->ptrStartPage(), SIGNAL(        newProject()               ), this, SLOT(    projectNewEvent()               ) );
		connect( this->ptrStartPage(), SIGNAL(       openProject()               ), this, SLOT(   projectOpenEvent()               ) );
		connect( this->ptrStartPage(), SIGNAL(     importProject()               ), this, SLOT( projectImportEvent()               ) );
		connect( this->ptrStartPage(), SIGNAL(  workspaceChanged(const QString&) ), this, SLOT(       setWorkspace(const QString&) ) );
		connect( this->ptrStartPage(), SIGNAL( categoriesChanged(const QString&) ), this, SLOT(      setCategories(const QString&) ) );
		this->addDockWidget ( Qt::TopDockWidgetArea, this->ptrStartPage() );
	}

	MainWindow::~MainWindow ()
	{
		for ( QDockWidget* page : this->tabifiedDockWidgets(this->ptrStartPage()) )
			page->close();
	}

	// --------------------------------------------------
	// GETTERS - SETTERS

	const QSettings&	MainWindow::settings		() const	{ return this->_settings;		}
	QSettings&			MainWindow::settings		()			{ return this->_settings;		}
	const QString&		MainWindow::workspace		() const	{ return this->_workspace;		}
	QString&			MainWindow::workspace		()			{ return this->_workspace;		}
	const QString&		MainWindow::categoryFile	() const	{ return this->_categoryFile;	}
	QString&			MainWindow::categoryFile	()			{ return this->_categoryFile;	}
	const StartPage*	MainWindow::ptrStartPage	() const	{ return this->_ptrStartPage;	}
	StartPage*&			MainWindow::ptrStartPage	()			{ return this->_ptrStartPage;	}

	void MainWindow::addProjectPage ( ProjectPage* page )
	{
		this->tabifyDockWidget( this->ptrStartPage(), page );
		connect( page, SIGNAL(sizeChanged(const QSize&)), this, SLOT(fitToPage(const QSize&)) );
		page->show();
		page->raise();
	}

	ProjectPage* MainWindow::getProjectPage ()
	{
		for ( QDockWidget* page : this->tabifiedDockWidgets(this->ptrStartPage()) )
		{
			if ( page->isVisible() )
				return dynamic_cast< ProjectPage* >( page );
		}
		return NULL;
	}

	// --------------------------------------------------
	// INIT MENUS

	void MainWindow::initMenuProject ()
	{
		// Project -> New
		actNew->setShortcut( QKeySequence::New );
		actNew->setIcon( QApplication::style()->standardIcon(QStyle::SP_FileIcon) );
		connect( actNew, SIGNAL(triggered()), this, SLOT(projectNewEvent()) );

		// Project -> Open
		actOpen->setShortcut( QKeySequence::Open );
		actOpen->setIcon( QApplication::style()->standardIcon(QStyle::SP_DialogOpenButton) );
		connect( actOpen, SIGNAL(triggered()), this, SLOT(projectOpenEvent()) );

		// Project -> Save
		actSave->setShortcut( QKeySequence::Save );
		actSave->setIcon( QApplication::style()->standardIcon(QStyle::SP_DialogSaveButton) );
	//	connect( ptrActionFileSave, SIGNAL(triggered()), this, SLOT(projectSaveEvent()) );

		// Project -> SaveAs
		actSaveAs->setShortcut( QKeySequence::SaveAs );
		actSaveAs->setIcon( QApplication::style()->standardIcon(QStyle::SP_DialogSaveButton) );
	//	connect( actSaveAs, SIGNAL(triggered()), this, SLOT(projectSaveAsEvent()) );

		// Project -> Import
		actImport->setShortcut( QKeySequence("Ctrl+I") );
		actImport->setIcon( QApplication::style()->standardIcon(QStyle::SP_DialogOpenButton) );
		connect( actImport, SIGNAL(triggered()), this, SLOT(projectImportEvent()) );

		// Project -> Export
		actExport->setShortcut( QKeySequence("Ctrl+E"));
		actExport->setIcon( QApplication::style()->standardIcon(QStyle::SP_DialogSaveButton) );
	//	connect( actExport, SIGNAL(triggered()), this, SLOT(projectExportEvent()) );

		// Project -> Close
		actClose->setShortcut( QKeySequence("Ctrl+W"));
		actClose->setIcon( QApplication::style()->standardIcon(QStyle::SP_DialogCloseButton) );
		connect( actClose, SIGNAL(triggered()), this, SLOT(projectCloseEvent()) );

		// Project -> Quit
		actQuit->setShortcut( QKeySequence::Quit );
		actQuit->setIcon( QApplication::style()->standardIcon(QStyle::SP_DialogCloseButton) );
		connect( actQuit, SIGNAL(triggered()), this, SLOT(quitEvent()) );
	}

	void MainWindow::initMenuEdit ()
	{
		actUndo->setShortcut( QKeySequence::Undo );
	//	connect( actUndo, SIGNAL(triggered()), this, SLOT(editUndoEvent()) );

		actRedo->setShortcut( QKeySequence::Redo );
	//	connect( actRedo, SIGNAL(triggered()), this, SLOT(editRedoEvent()) );

		actCopy->setShortcut( QKeySequence::Copy );
	//	connect( actCopy, SIGNAL(triggered()), this, SLOT(editCopyEvent()) );

		actCut->setShortcut( QKeySequence::Cut );
	//	connect( actCut, SIGNAL(triggered()), this, SLOT(editCutEvent()) );

		actPaste->setShortcut( QKeySequence::Paste );
	//	connect( actPaste, SIGNAL(triggered()), this, SLOT(editPastEvent()) );
	}

	void MainWindow::projectNewEvent ()
	{
		static int count( 0 );

		QString dir = QString("%1/New Project %2").arg(this->workspace()).arg(++count);

		this->addProjectPage( new ProjectPage(dir) );
	}

	void MainWindow::projectOpenEvent ()
	{
		QString dir = QFileDialog::getExistingDirectory( this, tr("Open an existing project"), this->workspace(), QFileDialog::ShowDirsOnly );
		if ( ! dir.isEmpty() )
		{
			ProjectPage* ptrNewPage = new ProjectPage( dir, this->categoryFile() );
			this->addProjectPage( ptrNewPage );
		}
		// TODO - START TIMER HERE
	}

	void MainWindow::projectImportEvent ()
	{
		// Create a new blank Project Page (and its underlying project)
		ProjectPtr ptrProject  = std::make_shared< Project >();

		// Create the imortation wizard
		ImportationWizard* ptrWizard = new ImportationWizard( this->workspace() );

		// Create the importer, connect it to the wizard (to pass options)
		Importer* ptrImporter = new Importer( ptrProject );
		connect( ptrWizard, SIGNAL( accepted(const ImportationOptions&) ), ptrImporter, SLOT( setOptions(const ImportationOptions&) ) );

		// run the wizard, and if accepted, import 
		if ( ptrWizard->exec() == QDialog::Accepted )
		{
			// Run the importer
			ptrImporter->run( this->workspace() );

			// Add the page to the UI
			ProjectPage* ptrImportedPage = new ProjectPage( ptrProject->fullpath(), this->categoryFile() );
			this->addProjectPage( ptrImportedPage );
		}
	}

	void MainWindow::projectCloseEvent ()
	{
		ProjectPage* page = this->getProjectPage();

		if ( page != NULL )
		{
			page->close();
			this->removeDockWidget( page );
			delete page;
		}	
	}

	void MainWindow::quitEvent ()
	{
		this->close();
	}

	void MainWindow::setWorkspace ( const QString& folder )
	{
		this->workspace() = folder;
		this->settings().setValue( "workspace", folder );
	}

	void MainWindow::setCategories ( const QString& file )
	{
		this->categoryFile() = file;
		this->settings().setValue( "categories", file );
	}

	void MainWindow::fitToPage ( const QSize& size )
	{
		this->QWidget::resize( size );
	}

 	// END CLASS MAINWINDOW
	// ##################################################

} // END NAMESPACE TOBII
// ##################################################


#endif // TOBII_MAINWINDOW_CPP
// ##################################################
