# --------------------------------------------------------
# Fast R-CNN
# Copyright (c) 2015 Microsoft
# Licensed under The MIT License [see LICENSE for details]
# Written by Ross Girshick
# --------------------------------------------------------

"""Compute minibatch blobs for training a Fast R-CNN network."""

import numpy as np
import numpy.random as npr
import cv2
import os
from fast_rcnn.config import cfg
from utils.blob import prep_im_for_blob, im_list_to_blob
import pdb

def get_minibatch(roidb, num_classes):
   
    #Only positives and video level labels
    if cfg.TRAIN.BG_SAMPLING==0:
        im_blob, rois_blob, labels_blob, weights_blob=minibatch_video(roidb)            
    #Positives and negatives sampled from different frames
    elif cfg.TRAIN.BG_SAMPLING==1:
        #im_blob, rois_blob, labels_blob=minibatch_pos_neg_difframes(roidb)
        im_blob, rois_blob, labels_blob, weights_blob=minibatch_pos_neg_difframes(roidb)
    #Positives and negatives sampled from any frame    
    elif cfg.TRAIN.BG_SAMPLING==2:
        im_blob, rois_blob, labels_blob, weights_blob=minibatch_pos_neg_allframes(roidb)     
    #By video and by category
    elif cfg.TRAIN.BG_SAMPLING==3:
        im_blob, rois_blob, labels_blob, weights_blob=minibatch_category(roidb)                   
    # For debug visualizations
    # _vis_minibatch(im_blob, rois_blob, labels_blob, all_overlaps)

    blobs = {'data': im_blob,
             'rois': rois_blob,
             'labels': labels_blob.transpose(),
             'weights': weights_blob}


    return blobs

def _get_image_blob(roidb,vfv_indexes,scale_inds,im_rois):
    """Builds an input blob from the images with the indexes of video and frames.
    """
    num_images = vfv_indexes.shape[0]
    processed_ims = []
    im_scales = []
    for i in xrange(num_images):
        frFile=roidb[vfv_indexes[i,0]]['frames'][vfv_indexes[i,1]]['images'][vfv_indexes[i,2]]
        #print frFile[-70:]
        im = cv2.imread(frFile)
        target_size = cfg.TRAIN.SCALES[scale_inds[vfv_indexes[i,0]]]
        height=im.shape[0]
        width=im.shape[1]
        if cfg.TRAIN.CROPPING:
            center=np.zeros((2,),dtype=np.uint16)
            #Get the limits
            aux_bb=roidb[vfv_indexes[i,0]]['frames'][vfv_indexes[i,1]]['boxes'][vfv_indexes[i,2],...]
            hbb=np.round(0.5*cfg.TRAIN.CROPSIZE).astype(int)
            center[0]=((aux_bb[...,0]+0.5*aux_bb[...,2]).mean()).astype(int)
            center[1]=((aux_bb[...,1]+0.5*aux_bb[...,3]).mean()).astype(int)
            center[0]=np.minimum(np.maximum(center[0],hbb),width-hbb)
            center[1]=np.minimum(np.maximum(center[1],hbb),height-hbb)
            initp=[center[0]-hbb,center[1]-hbb]
            endp=[center[0]+hbb,center[1]+hbb]
            im=im[initp[1]:endp[1],initp[0]:endp[0],...]
            #if os.path.exists('tests')==0:
            #    os.makedirs('tests')
            #cv2.imwrite('tests/%d.jpg'%i,im)
            im, im_scale = prep_im_for_blob(im, cfg.PIXEL_MEANS, np.maximum(im.shape[0],im.shape[1]),
                                            cfg.TRAIN.MAX_SIZE)
            #Modify the rois to the cropped version of the image
            imidx=np.where(im_rois[...,0]==i)[0]
            im_rois[imidx,1] = im_rois[imidx,1]-initp[0]
            im_rois[imidx,2] = im_rois[imidx,2]-initp[1]
            im_scale=1
        else:
            imidx=np.where(im_rois[...,0]==i)[0]
            
            im, im_scale = prep_im_for_blob(im, cfg.PIXEL_MEANS, target_size,
                                            cfg.TRAIN.MAX_SIZE)
            im_rois[imidx][1:] = im_rois[imidx][1:] * im_scale
        
        im_scales.append(im_scale)
        processed_ims.append(im)

    # Create a blob to hold the input images
    blob = im_list_to_blob(processed_ims)

    return blob,im_scales,im_rois

def _project_im_rois(im_rois, im_scale_factor):
    """Project image RoIs into the rescaled training image."""
    for i in xrange(im_rois.shape[0]):
        imidx=int(im_rois[i][0])
        im_rois[i][1:] = im_rois[i][1:] * im_scale_factor[imidx]
    return im_rois


def _vis_minibatch(im_blob, rois_blob, labels_blob, overlaps):
    """Visualize a mini-batch for debugging."""
    import matplotlib.pyplot as plt
    for i in xrange(rois_blob.shape[0]):
        rois = rois_blob[i, :]
        im_ind = rois[0]
        roi = rois[1:]
        im = im_blob[im_ind, :, :, :].transpose((1, 2, 0)).copy()
        im += cfg.PIXEL_MEANS
        im = im[:, :, (2, 1, 0)]
        im = im.astype(np.uint8)
        cls = labels_blob[i]
        plt.imshow(im)
        print 'class: ', cls, ' overlap: ', overlaps[i]
        plt.gca().add_patch(
            plt.Rectangle((roi[0], roi[1]), roi[2] - roi[0],
                          roi[3] - roi[1], fill=False,
                          edgecolor='r', linewidth=3)
            )
        plt.show()

"""Function that implements the selection of a minibatch at the video level"""
def minibatch_video(roidb):
    
    """Given a roidb, construct a minibatch sampled from it."""
    num_videos = len(roidb)
     # Sample random scales to use for each image in this batch
    random_scale_inds = npr.randint(0, high=len(cfg.TRAIN.SCALES),size=num_videos) 
    images_per_video = cfg.TRAIN.IMAGES_PER_VIDEO
    assert(cfg.TRAIN.BATCH_SIZE % (num_videos*images_per_video) == 0), \
    'num_videos*images_per_video ({}) must divide BATCH_SIZE ({})'. \
    format(num_videos*images_per_video, cfg.TRAIN.BATCH_SIZE)
    
    rois_per_image = cfg.TRAIN.BATCH_SIZE / (num_videos*images_per_video)
    
    
    #Variable with the video and frame indexes of the batch
    vid_fr_var_indexes=np.zeros((0,3),dtype=np.int)
    rois_blob = np.zeros((0, 5), dtype=np.float32)
    labels_blob = np.zeros((1, 0), dtype=np.float32)
    weights_blob = np.zeros((0,), dtype=np.float32)
    vtypes=(npr.rand(num_videos)<cfg.TRAIN.FG_FRACTION).astype(np.int);
    #We need to sample things
    for v in xrange(num_videos):
         #We need to select if the video is going to be positive or negative
         
         
         variations = np.array([],dtype=int)
         #Get the number of variations
         num_variations=roidb[v]['frames'][0]['boxes'].shape[0]
         
         #Sample the frames of the vtype
         nFr=len(roidb[v]['frames'])
         fg_frames=np.where((roidb[v]['frame_types'].ravel()>0).astype(np.int)==vtypes[v])[0]
         nvFr=len(fg_frames)
         
         #if(vtypes[v]==0):
         #    #Quite the last 10 frames
         #    fg_frames=fg_frames[:nvFr-10]
         #    nvFr=len(fg_frames)
         
         if nvFr==0:
             #print "Correcting Bad annotated video " + roidb[v]['video']
             fg_frames=range(int(0.5*nFr),nFr)
             nvFr=len(fg_frames)
             
         fg_frames=np.array(fg_frames)
         if images_per_video<=nvFr:
             fr_idxs=npr.choice(fg_frames,images_per_video,False)
         else:
             #fr_idxs=npr.choice(fg_frames,images_per_video,True)
             reps=int(images_per_video/nvFr)
             fr_idxs=fg_frames.repeat(reps)
             dif=images_per_video-fr_idxs.shape[0]
             if dif>0:
                 fr_idxs=np.concatenate((fr_idxs,npr.choice(fg_frames,dif,False)))
        
         #Sample the variations
         variations=np.concatenate((variations, npr.randint(0,num_variations,(len(fr_idxs),))))
                     
         nFr=fr_idxs.shape[0]
         vfi=np.vstack((v*np.ones((nFr,),dtype=np.int),fr_idxs,variations)).transpose()
         vid_fr_var_indexes=np.vstack((vid_fr_var_indexes,vfi))
            
         for f in xrange(nFr):
             idxf=fr_idxs[f]
             nvar=variations[f]
             #Index of fg boxes
             fgbbidx=(np.where((roidb[v]['frames'][idxf]['fg'][nvar,...].ravel()>0).astype(np.int)==vtypes[v]))[0]
             if len(fgbbidx)==0:
                 pdb.set_trace()
                 print "Weird, there are not bbs where they should be"
                 fgbbidx=range(rois_per_image)
             bb_idxs=npr.choice(fgbbidx,rois_per_image,False)
             nBBs=len(bb_idxs)
             rois=roidb[v]['frames'][idxf]['boxes'][nvar,bb_idxs,...]
             #Add random shifts
             if cfg.TRAIN.SHIFT_STD>0:
                 rois[...,0:2]=(rois[...,0:2]+cfg.TRAIN.SHIFT_STD*npr.randn(nBBs,2)).astype(np.int)
             bbs=np.hstack((f*np.ones((nBBs,1),dtype=np.float32),rois))
             rois_blob=np.vstack((rois_blob,bbs))
         
         #Generate the weights
         distances=roidb[v]['frame_types'][fr_idxs].astype(np.float).transpose()
         if distances.max()>0:
             distances=distances/distances.max()
             distances=1.0/distances;
             distances=distances-max(distances)
             weights=np.exp(-cfg.TRAIN.WEIGHTS_GAMMA*distances)
         else:
             weights=np.ones_like(distances)
         weights=weights/weights.sum()    
         weights_blob = np.hstack((weights_blob, weights))
    
         #We can generate labels at every bb
         labels=roidb[v]['label']*vtypes[v]*np.ones((1,1),dtype=np.float32)
         labels_blob = np.hstack((labels_blob, labels))
   
    # Get the input image blob, formatted for caffe
    im_blob, im_scales, rois_blob = _get_image_blob(roidb,vid_fr_var_indexes,random_scale_inds,rois_blob)
    
    return im_blob, rois_blob, labels_blob, weights_blob



"""Function that implements the selection of a minibatch with positives 
   and negatives samples from different frames"""
def minibatch_pos_neg_difframes(roidb):
    
    """Given a roidb, construct a minibatch sampled from it."""
    num_videos = len(roidb)
     # Sample random scales to use for each image in this batch
    random_scale_inds = npr.randint(0, high=len(cfg.TRAIN.SCALES),size=num_videos) 
    images_per_video = cfg.TRAIN.IMAGES_PER_VIDEO
    pos_images_per_video = int(cfg.TRAIN.IMAGES_PER_VIDEO*cfg.TRAIN.FG_FRACTION)
    neg_images_per_video = images_per_video - pos_images_per_video
    
    assert(cfg.TRAIN.BATCH_SIZE % (num_videos*images_per_video) == 0), \
    'num_videos*images_per_video ({}) must divide BATCH_SIZE ({})'. \
    format(num_videos*images_per_video, cfg.TRAIN.BATCH_SIZE)
    
    rois_per_image = cfg.TRAIN.BATCH_SIZE / (num_videos*images_per_video)
    #Variable with the video and frame indexes of the batch
    vid_fr_var_indexes=np.zeros((0,3),dtype=np.int)
    rois_blob = np.zeros((0, 5), dtype=np.float32)
    weights_blob = np.zeros((1, 0),dtype=np.float32)
    labels_blob = np.zeros((1, 0),dtype=np.float32)
    #We need to sample positives and negatives
    for v in xrange(num_videos):
         fr_idxs = np.array([])
         variations = np.array([],dtype=int)
         #Get the number of variations
         num_variations=roidb[v]['frames'][0]['boxes'].shape[0]
             
         #Sample the positive and negative frames
         nFr=len(roidb[v]['frames'])
         pos_frames=np.where(roidb[v]['frame_types']==1)[0]
         neg_frames=np.where(roidb[v]['frame_types']==0)[0]
         
         des_npos=pos_images_per_video
         des_nneg=neg_images_per_video
         
         npFr=len(pos_frames)
         nnFr=len(neg_frames)
         if npFr==0:
             print "Video with no positives " + roidb[v]['video']
             #pos_frames=range(int(0.5*nFr),nFr)
             des_npos=0
             des_nneg=images_per_video
                 
         if nnFr==0:
             print "Video with no negatives" + roidb[v]['video']
             des_npos=images_per_video
             des_nneg=0
         
         #Sampling positive data   
         if des_npos>0:
             pos_frames=np.array(pos_frames)
             if des_npos<=npFr:
                 fr_idxs=np.concatenate((fr_idxs,npr.choice(pos_frames,des_npos,False)))
             else:
                 #fr_idxs=npr.choice(fg_frames,images_per_video,True)
                 reps=int(des_npos/npFr)
                 fr_idxs=np.concatenate((fr_idxs,pos_frames.repeat(reps)))
                 dif=des_npos-fr_idxs.shape[0]
                 if dif>0:
                     fr_idxs=np.concatenate((fr_idxs,npr.choice(pos_frames,dif,False)))		        
    		    
         #Sampling negative data   
         if des_nneg>0:
             neg_frames=np.array(neg_frames)
             if des_nneg<=nnFr:
                 fr_idxs=np.concatenate((fr_idxs,npr.choice(neg_frames,des_nneg,False)))
                 
             else:
                 #fr_idxs=npr.choice(fg_frames,images_per_video,True)
                 reps=int(des_nneg/nnFr)
                 fr_idxs=np.concatenate((fr_idxs,neg_frames.repeat(reps)))
                 dif=des_nneg-fr_idxs.shape[0]
                 if dif>0:
                     fr_idxs=np.concatenate((fr_idxs,npr.choice(neg_frames,dif,False)))		
         
         #Sample the variations
         variations=np.concatenate((variations, npr.randint(0,num_variations,(len(fr_idxs),))))
                
         nFr=fr_idxs.shape[0]
         vfi=np.vstack((v*np.ones((nFr,),dtype=np.int),fr_idxs,variations)).transpose()
         vid_fr_var_indexes=np.vstack((vid_fr_var_indexes,vfi))
            
         for f in xrange(nFr):
             idxf=np.int(fr_idxs[f])
             nvar=variations[f]
             
             #get the frame type
             ftype = int(f < des_npos)
             #Get the boxes of the same type
             fgbbidx=np.where(roidb[v]['frames'][idxf]['fg'][nvar,...]==ftype)[0]
             if len(fgbbidx)==0:
                 print "Error there are not bounding boxes on a frame for which there should be"
                 fgbbidx=range(rois_per_image)
             #Randomly selecting the boxes
             if rois_per_image<=len(fgbbidx):
                 bb_idxs=npr.choice(fgbbidx,rois_per_image,False)
             else:
                 bb_idxs=npr.choice(fgbbidx,rois_per_image,True)
                 
             nBBs=len(bb_idxs)
             rois=roidb[v]['frames'][idxf]['boxes'][nvar,bb_idxs,...]
             #Add random shifts
             if cfg.TRAIN.SHIFT_STD>0:
                 rois[...,0:1]=np.int(rois[...,0:1]+cfg.TRAIN.SHIFT_STD*npr.randn(nBBs,2))
             bbs=np.hstack((f*np.ones((nBBs,1),dtype=np.float32),rois))
             rois_blob=np.vstack((rois_blob,bbs))
             #We can generate labels at every bb taking into account the ftype
             labels=ftype*roidb[v]['label']*np.ones((1,1),dtype=np.float32)
             labels_blob=np.hstack((labels_blob,labels))
         
         weights=np.ones((nFr,),dtype=np.float32)
         weights_blob = np.hstack((weights_blob, weights))
         
    # Get the input image blob, formatted for caffe
    im_blob, im_scales, rois_blob = _get_image_blob(roidb,vid_fr_var_indexes.astype(int),random_scale_inds,rois_blob)
    #print np.bincount(labels_blob[0,...].astype(int)).nonzero()
   
    
    return im_blob, rois_blob, labels_blob, weights_blob


"""Function that implements the selection of a minibatch with positives 
   and negatives samples from different frames"""
def minibatch_pos_neg_allframes(roidb):
    
    """Given a roidb, construct a minibatch sampled from it."""
    num_videos = len(roidb)
     # Sample random scales to use for each image in this batch
    random_scale_inds = npr.randint(0, high=len(cfg.TRAIN.SCALES),size=num_videos) 
    images_per_video = cfg.TRAIN.IMAGES_PER_VIDEO
    pos_images_per_video = int(cfg.TRAIN.IMAGES_PER_VIDEO*cfg.TRAIN.FG_FRACTION)
    neg_images_per_video = images_per_video - pos_images_per_video
    
    assert(cfg.TRAIN.BATCH_SIZE % (num_videos*images_per_video) == 0), \
    'num_videos*images_per_video ({}) must divide BATCH_SIZE ({})'. \
    format(num_videos*images_per_video, cfg.TRAIN.BATCH_SIZE)
    
    rois_per_image = cfg.TRAIN.BATCH_SIZE / (num_videos*images_per_video)
    #Variable with the video and frame indexes of the batch
    vid_fr_var_indexes=np.zeros((0,3),dtype=np.int)
    rois_blob = np.zeros((0, 5), dtype=np.float32)
    labels_blob = np.zeros((1, 0), dtype=np.float32)
    weights_blob = np.zeros((1, 0), dtype=np.float32)
    #We need to sample positives and negatives
    for v in xrange(num_videos):
         fr_idxs = np.array([])
    
         variations = np.array([],dtype=int)
         #Get the number of variations
         num_variations=roidb[v]['frames'][0]['boxes'].shape[0]
         
         #Sample the positive and negative frames
         nFr=len(roidb[v]['frames'])
         pos_frames=np.where(roidb[v]['frame_types']==1)[0]
         neg_frames=range(nFr)
         
         des_npos=pos_images_per_video
         des_nneg=neg_images_per_video
         
         npFr=len(pos_frames)
         nnFr=len(neg_frames)
         if npFr==0:
             #print "Video with no positives " + roidb[v]['video']
             #pos_frames=range(int(0.5*nFr),nFr)
             des_npos=0
             des_nneg=images_per_video
                 
         if nnFr==0:
             #print "Video with no negatives" + roidb[v]['video']
             des_npos=images_per_video
             des_nneg=0
         
         #Sampling positive data   
         if des_npos>0:
             pos_frames=np.array(pos_frames)
             if des_npos<=npFr:
                 fr_idxs=np.concatenate((fr_idxs,npr.choice(pos_frames,des_npos,False)))
             else:
                 reps=int(des_npos/npFr)
                 fr_idxs=np.concatenate((fr_idxs,pos_frames.repeat(reps)))
                 dif=des_npos-fr_idxs.shape[0]
                 if dif>0:
                     fr_idxs=np.concatenate((fr_idxs,npr.choice(pos_frames,dif,False)))		        
    		        
            
         #Sampling negative data   
         if des_nneg>0:
             neg_frames=np.array(neg_frames)
             if des_nneg<=nnFr:
                 fr_idxs=np.concatenate((fr_idxs,npr.choice(neg_frames,des_nneg,False)))
             else:
                 reps=int(des_nneg/nnFr)
                 fr_idxs=np.concatenate((fr_idxs,neg_frames.repeat(reps)))
                 dif=des_nneg-fr_idxs.shape[0]
                 if dif>0:
                     fr_idxs=np.concatenate((fr_idxs,npr.choice(neg_frames,dif,False)))		
              
         #Sample the variations
         variations=np.concatenate((variations, npr.randint(0,num_variations,(len(fr_idxs),))))
                    
         nFr=fr_idxs.shape[0]
         vfi=np.vstack((v*np.ones((nFr,),dtype=np.int),fr_idxs,variations)).transpose()
         vid_fr_var_indexes=np.vstack((vid_fr_var_indexes,vfi))
            
         for f in xrange(nFr):
             idxf=np.int(fr_idxs[f])
             nvar=variations[f]
             #get the frame type
             ftype = int(f < des_npos)
             #Get the boxes of the same type
             fgbbidx=np.where(roidb[v]['frames'][idxf]['fg'][nvar,...]==ftype)[0]
             if len(fgbbidx)==0:
                 print "Error there are not bounding boxes on a frame for which there should be"
                 fgbbidx=range(rois_per_image)
             #Randomly selecting the boxes
             bb_idxs=npr.choice(fgbbidx,rois_per_image,False)
             nBBs=len(bb_idxs)
             rois=roidb[v]['frames'][idxf]['boxes'][nvar,bb_idxs,...]
             #Add random shifts
             if cfg.TRAIN.SHIFT_STD>0:
                 rois[...,0:1]=np.int(rois[...,0:1]+cfg.TRAIN.SHIFT_STD*npr.randn(nBBs,2))
             bbs=np.hstack((f*np.ones((nBBs,1),dtype=np.float32),rois))
             rois_blob=np.vstack((rois_blob,bbs))
             #We can generate labels at every bb taking into account the ftype
             labels=ftype*roidb[v]['label']*np.ones((1,nBBs),dtype=np.float32)
             labels_blob = np.hstack((labels_blob, labels))
         
         weights=np.ones((nFr,),dtype=np.float32)
         weights_blob = np.hstack((weights_blob, weights))
         
    # Get the input image blob, formatted for caffe
    im_blob, im_scales, rois_blob = _get_image_blob(roidb,vid_fr_var_indexes.astype(int),random_scale_inds,rois_blob)
    #print np.bincount(labels_blob[0,...].astype(int)).nonzero()
   
    
    return im_blob, rois_blob, labels_blob


"""Function that implements the selection of a minibatch at the video level"""
def minibatch_category(roidb):
    
    """Given a roidb, construct a minibatch sampled from it."""
    num_videos = len(roidb)
     # Sample random scales to use for each image in this batch
    random_scale_inds = npr.randint(0, high=len(cfg.TRAIN.SCALES),size=num_videos) 
    images_per_video = cfg.TRAIN.IMAGES_PER_VIDEO
    assert(cfg.TRAIN.BATCH_SIZE % (num_videos*images_per_video) == 0), \
    'num_videos*images_per_video ({}) must divide BATCH_SIZE ({})'. \
    format(num_videos*images_per_video, cfg.TRAIN.BATCH_SIZE)
    
    rois_per_image = cfg.TRAIN.BATCH_SIZE / (num_videos*images_per_video)
    
    
    #Variable with the video and frame indexes of the batch
    vid_fr_var_indexes=np.zeros((0,3),dtype=np.int)
    rois_blob = np.zeros((0, 5), dtype=np.float32)
    labels_blob = np.zeros((1, 0), dtype=np.float32)
    weights_blob = np.zeros((0,), dtype=np.float32)
    contf=0
    #We need to sample things
    for v in xrange(num_videos):
         #We need to select if the video is going to be positive or negative
         
         
         variations = np.array([],dtype=int)
         #Get the number of variations
         num_variations=roidb[v]['frames'][0]['boxes'].shape[0]
         
         #Sample the frames of the vtype
         nFr=len(roidb[v]['frames'])
         fg_frames=np.where((roidb[v]['frame_types'].ravel()>0).astype(np.int)==1)[0]
         nvFr=len(fg_frames)
         
         #if(vtypes[v]==0):
         #    #Quite the last 10 frames
         #    fg_frames=fg_frames[:nvFr-10]
         #    nvFr=len(fg_frames)
         
         if nvFr==0:
             #print "Correcting Bad annotated video " + roidb[v]['video']
             fg_frames=range(int(0.5*nFr),nFr)
             nvFr=len(fg_frames)
             
         fg_frames=np.array(fg_frames)
         if images_per_video<=nvFr:
             fr_idxs=npr.choice(fg_frames,images_per_video,False)
         else:
             #fr_idxs=npr.choice(fg_frames,images_per_video,True)
             reps=int(images_per_video/nvFr)
             fr_idxs=fg_frames.repeat(reps)
             dif=images_per_video-fr_idxs.shape[0]
             if dif>0:
                 fr_idxs=np.concatenate((fr_idxs,npr.choice(fg_frames,dif,False)))
        
         #Sample the variations
         variations=np.concatenate((variations, npr.randint(0,num_variations,(len(fr_idxs),))))
                     
         nFr=fr_idxs.shape[0]
         vfi=np.vstack((v*np.ones((nFr,),dtype=np.int),fr_idxs,variations)).transpose()
         vid_fr_var_indexes=np.vstack((vid_fr_var_indexes,vfi))
            
         for f in xrange(nFr):
             idxf=fr_idxs[f]
             nvar=variations[f]
             #Index of fg boxes
             fgbbidx=(np.where((roidb[v]['frames'][idxf]['fg'][nvar,...].ravel()>0).astype(np.int)==1))[0]
             if len(fgbbidx)==0:
                 pdb.set_trace()
                 print "Weird, there are not bbs where they should be"
                 fgbbidx=range(rois_per_image)
             bb_idxs=npr.choice(fgbbidx,rois_per_image,False)
             nBBs=len(bb_idxs)
             rois=roidb[v]['frames'][idxf]['boxes'][nvar,bb_idxs,...]
             #Add random shifts
             if cfg.TRAIN.SHIFT_STD>0:
                 rois[...,0:2]=(rois[...,0:2]+cfg.TRAIN.SHIFT_STD*npr.randn(nBBs,2)).astype(np.int)
             bbs=np.hstack((contf*np.ones((nBBs,1),dtype=np.float32),rois))
             contf+=1
             rois_blob=np.vstack((rois_blob,bbs))
         
         #Generate the weights
         distances=roidb[v]['frame_types'][fr_idxs].astype(np.float).transpose()
         if distances.max()>0:
             distances=distances/distances.max()
             distances=1.0/distances;
             distances=distances-max(distances)
             weights=np.exp(-cfg.TRAIN.WEIGHTS_GAMMA*distances)
         else:
             weights=np.ones_like(distances)
         weights=weights/weights.sum()    
         weights_blob = np.hstack((weights_blob, weights))
    
    #We can generate label for the whole set
    labels=roidb[0]['label']*np.ones((1,1),dtype=np.float32)
    labels_blob = np.hstack((labels_blob, labels))
   
    # Get the input image blob, formatted for caffe
    im_blob, im_scales, rois_blob = _get_image_blob(roidb,vid_fr_var_indexes,random_scale_inds,rois_blob)
    
    return im_blob, rois_blob, labels_blob, weights_blob