# --------------------------------------------------------
# Fast R-CNN
# Copyright (c) 2015 Microsoft
# Licensed under The MIT License [see LICENSE for details]
# Written by Ross Girshick
# --------------------------------------------------------

"""The data layer used during training to train a Fast R-CNN network.

RoIDataLayer implements a Caffe Python layer.
"""

import caffe
from fast_rcnn.config import cfg
from roi_data_layer.minibatch import get_minibatch
import numpy as np
import yaml
from multiprocessing import Process, Queue
import pdb
import numpy.random as npr


class RoIDataLayer(caffe.Layer):
    """Fast R-CNN data layer used for training."""

    def _shuffle_roidb_inds(self):
        """Randomly permute the training roidb."""
        self._perm = np.random.permutation(np.arange(len(self._roidb)))
        self._cur = 0
        if cfg.TRAIN.BG_SAMPLING==3:
            self._labels = np.zeros((len(self._roidb),))
            for v in range(len(self._roidb)):
                self._labels[v]=self._roidb[v]['label']
                
    def _get_next_minibatch_inds(self):
        """Return the roidb indices for the next minibatch."""
        
        """If we find the end of the dataset, shuffle again"""
        if cfg.TRAIN.BG_SAMPLING<3:
            if self._cur + cfg.TRAIN.VIDEOS_PER_BATCH > len(self._roidb):
                self._shuffle_roidb_inds()
                
                """We do a premutation of the selected images"""
                db_inds = self._perm[self._cur:self._cur + cfg.TRAIN.VIDEOS_PER_BATCH]
                self._cur += cfg.TRAIN.VIDEOS_PER_BATCH
        #We select various videos with the same category
        else:
            if self._cur==(self._num_classes-1):
                self._cur=0
            idx_cat=self._cur+1
            self._cur+=1
            #print "%d/%d"%(idx_cat,self._num_classes)
            idx_videos=np.where(self._labels==idx_cat)[0]
            if cfg.TRAIN.VIDEOS_PER_BATCH>idx_videos.shape[0]:
                db_inds=npr.choice(idx_videos,cfg.TRAIN.VIDEOS_PER_BATCH,True)
            else:
                db_inds=npr.choice(idx_videos,cfg.TRAIN.VIDEOS_PER_BATCH,False)
            
            #if self._cur==self._roidb
        return db_inds

    def _get_next_minibatch(self):
        """Return the blobs to be used for the next minibatch.

        If cfg.TRAIN.USE_PREFETCH is True, then blobs will be computed in a
        separate process and made available through self._blob_queue.
        """
        if cfg.TRAIN.USE_PREFETCH:
            return self._blob_queue.get()
        else:
            db_inds = self._get_next_minibatch_inds()
            minibatch_db = [self._roidb[i] for i in db_inds]
            return get_minibatch(minibatch_db, self._num_classes)

    def set_roidb(self, roidb):
        """Set the roidb to be used by this layer during training."""
        self._roidb = roidb
        self._shuffle_roidb_inds()
        if cfg.TRAIN.USE_PREFETCH:
            self._blob_queue = Queue(10)
            self._prefetch_process = BlobFetcher(self._blob_queue,
                                                 self._roidb,
                                                 self._num_classes)
            self._prefetch_process.start()
            # Terminate the child process when the parent exists
            def cleanup():
                print 'Terminating BlobFetcher'
                self._prefetch_process.terminate()
                self._prefetch_process.join()
            import atexit
            atexit.register(cleanup)

    def setup(self, bottom, top):
        """Setup the RoIDataLayer."""

        # parse the layer parameter string, which must be valid YAML
        layer_params = yaml.load(self.param_str)

        self._num_classes = layer_params['num_classes']

        self._name_to_top_map = {
            'data': 0,
            'rois': 1,
            'labels': 2,
            'weights': 3}

        # data blob: holds a batch of N images, each with 3 channels
        # The height and width (100 x 100) are dummy values
        top[0].reshape(1, 3, 100, 100)

        # rois blob: holds R regions of interest, each is a 5-tuple
        # (n, x1, y1, x2, y2) specifying an image batch index n and a
        # rectangle (x1, y1, x2, y2)
        top[1].reshape(1, 5)

        # labels blob: R categorical labels in [0, ..., K] for K foreground
        # classes plus background
        top[2].reshape(1)


    def forward(self, bottom, top):
        """Get blobs and copy them into this layer's top blob vector."""
        blobs = self._get_next_minibatch()

        for blob_name, blob in blobs.iteritems():
            top_ind = self._name_to_top_map[blob_name]
            # Reshape net's input blobs
            top[top_ind].reshape(*(blob.shape))
            # Copy data into net's input blobs
            top[top_ind].data[...] = blob.astype(np.float32, copy=False)

    def backward(self, top, propagate_down, bottom):
        """This layer does not propagate gradients."""
        pass

    def reshape(self, bottom, top):
        """Reshaping happens during the call to forward."""
        pass

class BlobFetcher(Process):
    """Experimental class for prefetching blobs in a separate process."""
    def __init__(self, queue, roidb, num_classes):
        super(BlobFetcher, self).__init__()
        self._queue = queue
        self._roidb = roidb
        self._num_classes = num_classes
        self._perm = None
        self._cur = 0
        self._shuffle_roidb_inds()
        # fix the random seed for reproducibility
        np.random.seed(cfg.RNG_SEED)

    def _shuffle_roidb_inds(self):
        """Randomly permute the training roidb."""
        # TODO(rbg): remove duplicated code
        self._perm = np.random.permutation(np.arange(len(self._roidb)))
        self._cur = 0
        if cfg.TRAIN.BG_SAMPLING==3:
            self._labels = np.zeros((len(self._roidb),))
            for v in range(len(self._roidb)):
                self._labels[v]=self._roidb[v]['label']
                
    def _get_next_minibatch_inds(self):
        """Return the roidb indices for the next minibatch."""
        # TODO(rbg): remove duplicated code
        """If we find the end of the dataset, shuffle again"""
        if cfg.TRAIN.BG_SAMPLING<3:
            if self._cur + cfg.TRAIN.VIDEOS_PER_BATCH > len(self._roidb):
                self._shuffle_roidb_inds()
                
            """We do a premutation of the selected images"""
            db_inds = self._perm[self._cur:self._cur + cfg.TRAIN.VIDEOS_PER_BATCH]
            self._cur += cfg.TRAIN.VIDEOS_PER_BATCH
        #We select various videos with the same category
        else:
            if self._cur==(self._num_classes-1):
                self._cur=0
            idx_cat=self._cur+1
            self._cur+=1
            #print "%d/%d"%(idx_cat,self._num_classes)
            idx_videos=np.where(self._labels==idx_cat)[0]
            if cfg.TRAIN.VIDEOS_PER_BATCH>idx_videos.shape[0]:
                db_inds=npr.choice(idx_videos,cfg.TRAIN.VIDEOS_PER_BATCH,True)
            else:
                db_inds=npr.choice(idx_videos,cfg.TRAIN.VIDEOS_PER_BATCH,False)
        return db_inds

    def run(self):
        print 'BlobFetcher started'
        while True:
            db_inds = self._get_next_minibatch_inds()
            minibatch_db = [self._roidb[i] for i in db_inds]
            blobs = get_minibatch(minibatch_db, self._num_classes)
            self._queue.put(blobs)
