# --------------------------------------------------------
# Fast R-CNN
# Copyright (c) 2015 Microsoft
# Licensed under The MIT License [see LICENSE for details]
# Written by Ross Girshick
# --------------------------------------------------------

"""Train a Fast R-CNN network."""

import caffe
from fast_rcnn.config import cfg
import roi_data_layer.roidb as rdl_roidb
from utils.timer import Timer
import numpy as np
import os
import pdb
import matplotlib.pyplot as plt
import glob

from caffe.proto import caffe_pb2
import google.protobuf as pb2
import google.protobuf.text_format 


class SolverWrapper(object):
    """A simple wrapper around Caffe's solver.
    This wrapper gives us control over he snapshotting process, which we
    use to unnormalize the learned bounding-box regression weights.
    """

    def __init__(self, solver_prototxt, roidb, output_dir,
                 pretrained_model=None):
        
        """Initialize the SolverWrapper."""
        self.output_dir = output_dir
        self.solver = caffe.SGDSolver(solver_prototxt)
        
        self.solver.myiter=0
        
        if pretrained_model is not None:
            print ('Loading pretrained model '
                   'weights from {:s}').format(pretrained_model)
            self.solver.net.copy_from(pretrained_model)

        self.solver_param = caffe_pb2.SolverParameter()
        
        with open(solver_prototxt, 'rt') as f:
            google.protobuf.text_format.Merge(f.read(), self.solver_param)
            
        if os.path.exists(output_dir)==0:
            os.makedirs(output_dir)

        #Check for snapshots
        iter, snapshotFile=self.check_snapshots()
        if iter>0:
            print ('Loading snapshot model '
                   'weights from {:s}').format(snapshotFile)
            self.solver.restore(snapshotFile)
            
        self.solver.net.layers[0].set_roidb(roidb)

    def snapshot(self):
        """Take a snapshot of the network after unnormalizing the learned
        bounding-box regression weights. This enables easy use at test-time.
        """
        
        if not os.path.exists(self.output_dir):
            os.makedirs(self.output_dir)

        infix = ('_' + cfg.TRAIN.SNAPSHOT_INFIX
                 if cfg.TRAIN.SNAPSHOT_INFIX != '' else '')
        old_solverfilename = (self.snapshot_orig_prefix +
                    '_iter_{:d}'.format(self.solver.iter) + '.solverstate.h5')
        old_modelfilename = (self.snapshot_orig_prefix +
                    '_iter_{:d}'.format(self.solver.iter) + '.caffemodel.h5')
        
        solverfilename = (self.solver_param.snapshot_prefix + 
                    '_iter_{:d}'.format(self.solver.iter) + '.solverstate.h5')
        modelfilename = (self.solver_param.snapshot_prefix + 
                    '_iter_{:d}'.format(self.solver.iter) + '.caffemodel.h5')
        solverfilename = os.path.join(self.output_dir, solverfilename)
        modelfilename = os.path.join(self.output_dir, modelfilename)

        os.rename(old_solverfilename,solverfilename)
        os.rename(old_modelfilename,modelfilename)
        print 'Wrote snapshot to: {:s}'.format(solverfilename)
        


    def check_snapshots(self):
        """Check if a snapshot of the net exist"""
        
        #Look for snapshots
        pattern_snapshot = self.solver_param.snapshot_prefix + '_iter_*.solverstate.h5'
        list_snapshots=glob.glob(pattern_snapshot)
        snapshotFile=[]
        maxiter=0
            
        if len(list_snapshots)>0:
            for i in range(len(list_snapshots)):
                idxend=list_snapshots[i].find('.solverstate.h5')
                idxinit=list_snapshots[i].find('_iter_')
                iter=int(list_snapshots[i][idxinit+6:idxend])
                if iter>maxiter:
                    snapshotFile=list_snapshots[i].encode('utf-8')
                    maxiter=iter
        
        return maxiter,snapshotFile
            
    def train_model(self, max_iters):
        """Network training loop."""
       # last_snapshot_iter = -1
        timer = Timer()
        epochs=max_iters/self.solver_param.display
        tr_loss=np.zeros(epochs+1,)
        tr_acc=np.zeros(epochs+1,)
        iters_figures=np.zeros(epochs+1,)
        epoch=0
        videosEpoch=len(self.solver.net.layers[0]._perm)
        while self.solver.myiter < max_iters:
            # Make one SGD update
            timer.tic()
            self.solver.step(1)
            timer.toc()
            self.solver.myiter+=1
            #print self.solver.net.blobs['loss_cls'].data
            tr_loss[epoch]+=self.solver.net.blobs['loss_cls'].data
            tr_acc[epoch]+=self.solver.net.blobs['acc_cls'].data
            #
            if self.solver.iter % videosEpoch == 0 and self.solver.iter>=videosEpoch:
                tr_loss[epoch]=tr_loss[epoch]/videosEpoch
                tr_acc[epoch]=tr_acc[epoch]/videosEpoch
                iters_figures[epoch]=self.solver.myiter
                print 'epoch %d avg loss: %.3f avg acc: %.3f'%(epoch,tr_loss[epoch],tr_acc[epoch])
                print 'speed: {:.3f}s / iter'.format(timer.average_time)
                epoch+=1
                fig1 = plt.gcf()
                plt.plot(iters_figures[0:epoch],tr_loss[0:epoch])
                plt.plot(iters_figures[0:epoch],tr_acc[0:epoch],'r')
                plt.ylabel('Training loss')
                fig1.savefig(os.path.join(self.output_dir, 'train_loss.png'))
                plt.close('all')  
                outLogFile=os.path.join(self.output_dir, 'train_loss.txt')
                np.savetxt(outLogFile,np.vstack((iters_figures[0:epoch],tr_loss[0:epoch],tr_acc[0:epoch])).transpose(),"%d %.3f %.3f")    
            #if self.solver.iter % self.solver_param.snapshot == 0:
            #    self.snapshot()
                
        #if last_snapshot_iter != self.solver.myiter:
         #   self.snapshot()

def get_training_roidb(imdb):
    
    print 'Preparing training data...'
    rdl_roidb.prepare_roidb(imdb)
    print 'done'

    return imdb.roidb

def train_net(solver_prototxt, roidb, output_dir,
              pretrained_model=None, max_iters=40000):
    """Train a Fast R-CNN network."""
    sw = SolverWrapper(solver_prototxt, roidb, output_dir,pretrained_model=pretrained_model)

    print 'Solving...'
    sw.train_model(max_iters)
    print 'done solving'
