# --------------------------------------------------------
# Fast R-CNN
# Copyright (c) 2015 Microsoft
# Licensed under The MIT License [see LICENSE for details]
# Written by Ross Girshick
# --------------------------------------------------------

"""Test a Fast R-CNN network on an imdb (image database)."""

from fast_rcnn.config import cfg, get_output_dir
import roi_data_layer.roidb as rdl_roidb
import argparse
from utils.timer import Timer
import numpy as np
import cv2
import caffe
from utils.cython_nms import nms
import cPickle
import heapq
from utils.blob import im_list_to_blob
import os
import sklearn.metrics as skm
import pdb

def _get_image_blob(im):
    """Converts an image into a network input.

    Arguments:
        im (ndarray): a color image in BGR order

    Returns:
        blob (ndarray): a data blob holding an image pyramid
        im_scale_factors (list): list of image scales (relative to im) used
            in the image pyramid
    """
    im_orig = im.astype(np.float32, copy=True)
    im_orig -= cfg.PIXEL_MEANS

    im_shape = im_orig.shape
    im_size_min = np.min(im_shape[0:2])
    im_size_max = np.max(im_shape[0:2])

    processed_ims = []
    im_scale_factors = []

    if cfg.TEST.CROPPING==False:
        for target_size in cfg.TEST.SCALES:
            im_scale = float(target_size) / float(im_size_min)
            # Prevent the biggest axis from being more than MAX_SIZE
            if np.round(im_scale * im_size_max) > cfg.TEST.MAX_SIZE:
                im_scale = float(cfg.TEST.MAX_SIZE) / float(im_size_max)
            im = cv2.resize(im_orig, None, None, fx=im_scale, fy=im_scale,
                            interpolation=cv2.INTER_LINEAR)
            im_scale_factors.append(im_scale)
            processed_ims.append(im)
    else:
         processed_ims.append(im)
         im_scale_factors.append(1)
         
    # Create a blob to hold the input images
    blob = im_list_to_blob(processed_ims)

    return blob, np.array(im_scale_factors)

def _get_rois_blob(im_rois, im_scale_factors):
    """Converts RoIs into network inputs.

    Arguments:
        im_rois (ndarray): R x 4 matrix of RoIs in original image coordinates
        im_scale_factors (list): scale factors as returned by _get_image_blob

    Returns:
        blob (ndarray): R x 5 matrix of RoIs in the image pyramid
    """
    rois, levels = _project_im_rois(im_rois, im_scale_factors)
    rois_blob = np.hstack((levels, rois))
    return rois_blob.astype(np.float32, copy=False)

def _project_im_rois(im_rois, scales):
    """Project image RoIs into the image pyramid built by _get_image_blob.

    Arguments:
        im_rois (ndarray): R x 4 matrix of RoIs in original image coordinates
        scales (list): scale factors as returned by _get_image_blob

    Returns:
        rois (ndarray): R x 4 matrix of projected RoI coordinates
        levels (list): image pyramid levels used by each projected RoI
    """
    im_rois = im_rois.astype(np.float, copy=False)

    if len(scales) > 1:
        widths = im_rois[:, 2] - im_rois[:, 0] + 1
        heights = im_rois[:, 3] - im_rois[:, 1] + 1

        areas = widths * heights
        scaled_areas = areas[:, np.newaxis] * (scales[np.newaxis, :] ** 2)
        diff_areas = np.abs(scaled_areas - 224 * 224)
        levels = diff_areas.argmin(axis=1)[:, np.newaxis]
    else:
        levels = np.zeros((im_rois.shape[0], 1), dtype=np.int)

    rois = im_rois * scales[levels]

    return rois, levels

def _get_blobs(im, rois):
    """Convert an image and RoIs within that image into network inputs."""
    blobs = {'data' : None, 'rois' : None}
    
    height=im.shape[0]
    width=im.shape[1]
    if cfg.TEST.CROPPING:
        center=np.zeros((2,),dtype=np.uint16)
        #Get the limits
        #lim_aux_bb=[rois[...,0].min(), rois[...,1].min(), rois[...,2].max(), rois[...,3].max()]
        hbb=np.round(0.5*cfg.TEST.CROPSIZE).astype(int)
        center[0]=((rois[...,0]+0.5*rois[...,2]).mean()).astype(int)
        center[1]=((rois[...,1]+0.5*rois[...,3]).mean()).astype(int)
        center[0]=np.minimum(np.maximum(center[0],hbb),width-hbb)
        center[1]=np.minimum(np.maximum(center[1],hbb),height-hbb)
        initp=[center[0]-hbb,center[1]-hbb]
        endp=[center[0]+hbb,center[1]+hbb]
        im=im[initp[1]:endp[1],initp[0]:endp[0],...]
        #if os.path.exists('tests')==0:
        #    os.makedirs('tests')
        #cv2.imwrite('tests/%d.jpg'%i,im)
        blobs['data'], im_scale_factors = _get_image_blob(im)
        rois[:,0] = rois[:,0]-initp[0]
        rois[:,1] = rois[:,1]-initp[1]
        blobs['rois'] = _get_rois_blob(rois, im_scale_factors)
    else:
        blobs['data'], im_scale_factors = _get_image_blob(im)
        blobs['rois'] = _get_rois_blob(rois, im_scale_factors)
    return blobs, im_scale_factors


def im_detect(net, im, boxes):
    """Detect object classes in an image given object proposals.

    Arguments:
        net (caffe.Net): Fast R-CNN network to use
        im (ndarray): color image to test (in BGR order)
        boxes (ndarray): R x 4 array of object proposals

    Returns:
        scores (ndarray): R x K array of object class scores (K includes
            background as object category 0)
        boxes (ndarray): R x (4*K) array of predicted bounding boxes
    """
    blobs, unused_im_scale_factors = _get_blobs(im, boxes)

    # reshape network inputs
    net.blobs['data'].reshape(*(blobs['data'].shape))
    net.blobs['rois'].reshape(*(blobs['rois'].shape))
    net.forward(data=blobs['data'].astype(np.float32, copy=False),
                            rois=blobs['rois'].astype(np.float32, copy=False))

    #Get the frame scores
    if cfg.TEST.NMS==1:
        bbs_scores = net.blobs['bbs_score'].data
        #We generate scores at two levels: global and small scale for NMS
        fr_scores=np.exp(1.0*bbs_scores).mean(axis=0)
        fr_scores=fr_scores/sum(fr_scores)
        sfr_scores=np.exp(1.0*bbs_scores[0:3,...]).mean(axis=0)
        sfr_scores=sfr_scores/sfr_scores.sum()
    elif cfg.TEST.NMS==2:
        bbs_scores = net.blobs['bbs_score'].data
        nBBs=bbs_scores.shape[0]/2
        fr_scores=np.exp(1.0*bbs_scores[0:nBBs,...]).mean(axis=0)
        fr_scores=fr_scores/sum(fr_scores)
        sfr_scores=np.exp(1.0*bbs_scores[nBBs:,...]).mean(axis=0)
        sfr_scores=sfr_scores/sum(sfr_scores)
    else:
        fr_scores = net.blobs['frs_score'].data[0,:]
        fr_scores=np.exp(fr_scores)
        fr_scores=fr_scores/fr_scores.sum()
        sfr_scores = []
        #bbs_scores = net.blobs['bbs_score'].data
        #sfr_scores=bbs_scores
    return fr_scores, sfr_scores    
        
 
def vis_detections(im, class_name, dets, thresh=0.3):
    """Visual debugging of detections."""
    import matplotlib.pyplot as plt
    im = im[:, :, (2, 1, 0)]
    for i in xrange(np.minimum(10, dets.shape[0])):
        bbox = dets[i, :4]
        score = dets[i, -1]
        if score > thresh:
            plt.cla()
            plt.imshow(im)
            plt.gca().add_patch(
                plt.Rectangle((bbox[0], bbox[1]),
                              bbox[2] - bbox[0],
                              bbox[3] - bbox[1], fill=False,
                              edgecolor='g', linewidth=3)
                )
            plt.title('{}  {:.3f}'.format(class_name, score))
            plt.show()

def apply_nms(gscores,sfr_scores):
    
    orig_gscores=np.copy(gscores)
    #Global
    gscores[0]=0;
    gscores=gscores/gscores.sum()
                 
    #Check if we have two peaks
    maxscore=gscores.max();
    peaks=np.where(gscores>cfg.TEST.NMS_TH*maxscore)[0]
    #If we have another peak
    if(peaks.shape[0]>1):
        print "Applying non-maxima suppression"
	print gscores[peaks]        
	#pdb.set_trace()
        #We take the results of the small scale
        
        #totalprob=gscores[peaks].sum()
        #orig_gscores[peaks]=totalprob*(orig_gscores[peaks]/orig_gscores[peaks].sum())
        
        orig_gscores[peaks]=sfr_scores[peaks]
        orig_gscores=orig_gscores/orig_gscores.sum()
    return orig_gscores

             
def test_net(net, imdb):
    """Test a Fast R-CNN network on an video database."""

    output_dir = get_output_dir(imdb, net)
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)

    # timers
    _t = {'im_detect' : Timer(), 'eval' : Timer()}
    rdl_roidb.prepare_roidb(imdb)
    roidb = imdb.roidb
    
    
    num_videos=imdb.num_videos
    
    num_classes=imdb.num_classes
    
    #Loop of videos
    scores=np.zeros((num_videos,num_classes))
    labels=np.zeros((num_videos,1))
    for v in xrange(num_videos):
        #v=60
        #pdb.set_trace()
        #Get the number of frames
        #We take only valid frames
        fg_frames=roidb[v]['frame_types'].ravel().nonzero()[0]
        nFr=len(fg_frames)
        
        fr_scores=np.zeros((nFr,num_classes))
        if cfg.TEST.NMS:
            sfr_scores=np.zeros((nFr,num_classes))
            
        #Loop of frames
        for f in xrange(nFr):
            #Read the image
            im = cv2.imread(imdb.image_path_at(v,fg_frames[f],0))
            _t['im_detect'].tic()
            rois=roidb[v]['frames'][fg_frames[f]]['boxes'][0,...]#...]
            #print rois
            if cfg.TEST.NMS==2:
                nBB=rois.shape[0]
                rois=np.vstack((rois,rois))
                rois[nBB:,1]=rois[nBB:,1]+cfg.TEST.NMS_SHIFT;
            
            sc, ssc = im_detect(net, im, rois)
            
            if(np.isnan(sc).sum()>0):
                print "Error nan"
            #Remove BG from the computations
            fr_scores[f,...]=sc;
            if cfg.TEST.NMS>0:
                sfr_scores[f,...]=ssc;
            
            _t['im_detect'].toc()
            _t['eval'].tic()
            #Evaluation at bb level; Loop of categories, we consider the confussion matrix
            _t['eval'].toc()
        
        #Generate the video prediction
        #scores[v,...]=fr_scores.mean(axis=0)
        if cfg.TEST.TWEIGHTS==False:
            scores[v,...]=(np.exp(1.0*fr_scores)).mean(axis=0)
            scores[v,...]=scores[v,...]/(scores[v,...].sum()+1e-30)
            if cfg.TEST.NMS>0:
                sfr_scores=(np.exp(1.0*sfr_scores)).mean(axis=0)
                sfr_scores=sfr_scores/(sfr_scores.sum()+1e-30)
                scores[v,...]=apply_nms(scores[v,...],sfr_scores)
            
        else:
            tweights=np.power(range(1,nFr+1),cfg.TEST.TWEIGHTS_PARAM)#0.1+np.log(1+cfg.TEST.TWEIGHTS_PARAM*np.linspace(0,1,nFr))#
            tweights=tweights/tweights.sum()
            scores[v,...]=(np.kron(np.ones((17,1)),tweights).transpose()*np.exp(fr_scores)).mean(axis=0)
            scores[v,...]=scores[v,...]/(scores[v,...].sum()+1e-30)
            if cfg.TEST.NMS>0:
                sfr_scores=(np.kron(np.ones((17,1)),tweights).transpose()*np.exp(sfr_scores)).mean(axis=0)
                sfr_scores=sfr_scores/(sfr_scores.sum()+1e-30)
                scores[v,...]=apply_nms(scores[v,...],sfr_scores)
       
        labels[v]=imdb.get_label(v)
        print 'Video {:d}/{:d} with {:d} frames {:.3f}s {:.3f}s' \
            .format(v,num_videos,nFr, _t['im_detect'].average_time,
            _t['eval'].average_time)
        
    
    #Remove BG in the evaluation
    labels=labels-1
    scores=scores[...,1:]
    
    #Softmax??
    #pdb.set_trace()
    #scores=np.exp(scores)
    #scores=scores/np.expand_dims(scores.sum(axis=1),axis=1)
    
    evaluateResults(scores,labels,imdb.classes)
    
    #Write output data
    of=open('outFile.txt','w')
    ypred=scores.argmax(axis=1).astype(np.int)
    labels=labels[...,0].astype(np.int)
    for v in xrange(num_videos):
        videoname=imdb.video_index[v]
        if(ypred[v]!=labels[v]):
            outdata="%s %s %s\n"%(videoname,imdb.classes[labels[v]+1],imdb.classes[ypred[v]+1])
            of.write(outdata)
    return
    of.close()
    
def evaluateResults(scores,labels,classes_names):
    num_classes=scores.shape[1]
    #Computing scores per_category
    aps=np.zeros(num_classes,)
    ypred=scores.argmax(axis=1)
    cm=skm.confusion_matrix(labels,ypred).astype(np.float)
    ncm=np.transpose( np.transpose(cm) / np.maximum(cm.astype(np.float).sum(axis=1),1e-40) )
    print 'Final results:'
    print '*****************'    
    for cat in xrange(num_classes):
        scores_cat=scores[...,cat]
        labels_cat=(labels==cat).astype(np.float)
        if(labels_cat.sum()>0):
            aps[cat]=skm.average_precision_score(labels_cat,scores_cat)
        else:
            aps[cat]=0
        print '%25s:\tap: %.3f\tacc: %.3f'%(classes_names[cat+1],aps[cat],ncm[cat,cat])
    print '*****************'    
    print '%25s:\tap: %.3f\tacc: %.3f\tbacc: %.3f'%('Average',aps.mean(),cm.trace()/cm.sum(),ncm.trace()/ncm.sum())    
    print 'Confusion Matrix'
    print cm
    np.set_printoptions(precision=3,linewidth=150)
    print ncm
    
    
    
   
        
