# --------------------------------------------------------
# Fast R-CNN
# Copyright (c) 2015 Microsoft
# Licensed under The MIT License [see LICENSE for details]
# Written by Ross Girshick
# --------------------------------------------------------

import os
import os.path as osp
import PIL
from utils.cython_bbox import bbox_overlaps
import numpy as np
import scipy.sparse
import datasets

class imdb(object):
    """Image database."""

    def __init__(self, name):
        self._name = name
        self._bbname = name
        self._num_classes = 0
        self._classes = []
        self._video_index = []
        self._obj_proposer = 'selective-search'
        self._roidb = None
        self._roidb_handler = self.default_roidb
        # Use this dict for storing dataset specific config options
        self.config = {}
        self._numVar = 1
        
    @property
    def name(self):
        return self._name

    
    @property
    def num_classes(self):
        return len(self._classes)

    @property
    def classes(self):
        return self._classes

    @property
    def video_index(self):
        return self._video_index

    @property
    def roidb_handler(self):
        return self._roidb_handler

    @roidb_handler.setter
    def roidb_handler(self, val):
        self._roidb_handler = val

    @property
    def roidb(self):
        # A roidb is a list of dictionaries, each with the following keys:
        #   boxes
        #   gt_overlaps
        #   gt_classes
        #   flipped
        if self._roidb is not None:
            return self._roidb
        self._roidb = self.roidb_handler()
        return self._roidb

    @property
    def cache_path(self):
        cache_path = osp.abspath(osp.join(datasets.ROOT_DIR, 'data', 'cache'))
        if not os.path.exists(cache_path):
            os.makedirs(cache_path)
        return cache_path

    @property
    def num_videos(self):
      return len(self.video_index)

    def image_path_at(self, i):
        raise NotImplementedError

    def default_roidb(self):
        raise NotImplementedError

    def evaluate_detections(self, all_boxes, output_dir=None):
        """
        all_boxes is a list of length number-of-classes.
        Each list element is a list of length number-of-images.
        Each of those list elements is either an empty list []
        or a numpy array of detection.

        all_boxes[class][image] = [] or np.array of shape #dets x 5
        """
        raise NotImplementedError

    def append_flipped_images(self):
        num_videos = self.num_videos
        widths = [PIL.Image.open(self.image_path_at(i)).size[0]
                  for i in xrange(num_videos)]
        for i in xrange(num_videos):
            boxes = self.roidb[i]['boxes'].copy()
            oldx1 = boxes[:, 0].copy()
            oldx2 = boxes[:, 2].copy()
            boxes[:, 0] = widths[i] - oldx2 - 1
            boxes[:, 2] = widths[i] - oldx1 - 1
            assert (boxes[:, 2] >= boxes[:, 0]).all()
            entry = {'boxes' : boxes,
                     'gt_overlaps' : self.roidb[i]['gt_overlaps'],
                     'gt_classes' : self.roidb[i]['gt_classes'],
                     'flipped' : True}
            self.roidb.append(entry)
        self._video_index = self._video_index * 2


    def create_roidb_from_box_list(self, box_list, gt_roidb):
        assert len(box_list) == self.num_videos, \
                'Number of boxes must match number of ground-truth images'
        roidb = []
        for v in xrange(self.num_videos):
            posFr=0
            nFr = len(box_list[v])
            video = []
            for f in xrange(nFr): 
                boxes = box_list[v][f]
                #If we have a positive frame
                if boxes[...,0].sum()>0:
                    posFr=1
                else:
                    #Don't consider negatives after the positive
                    if posFr==1:
                        break
                video.append({'boxes' : boxes[...,1:],
                              'fg' : boxes[...,0],
                              'flipped' : False})     
            roidb.append({'frames' : video})      
        
        self.set_bb_numVar(box_list[0][0].shape[0])
        return roidb
    
   
        
    @staticmethod
    def merge_roidbs(a, b):
        assert len(a) == len(b)
        for i in xrange(len(a)):
            a[i]['boxes'] = np.vstack((a[i]['boxes'], b[i]['boxes']))
            a[i]['gt_classes'] = np.hstack((a[i]['gt_classes'],
                                            b[i]['gt_classes']))
            a[i]['gt_overlaps'] = scipy.sparse.vstack([a[i]['gt_overlaps'],
                                                       b[i]['gt_overlaps']])
        return a

    def competition_mode(self, on):
        """Turn competition mode on or off."""
        pass

    def set_bb_numVar(self,numVar):
        self._numVar=numVar