# --------------------------------------------------------
# Fast R-CNN
# Copyright (c) 2015 Microsoft
# Licensed under The MIT License [see LICENSE for details]
# Written by Ross Girshick
# --------------------------------------------------------

import datasets
import datasets.grasping_in_the_wild
import os
import datasets.imdb
import xml.dom.minidom as minidom
import numpy as np
import scipy.sparse
import scipy.io as sio
import utils.cython_bbox
import cPickle
import subprocess
import pdb

class grasping_in_the_wild(datasets.imdb):
    
    def __init__(self, image_set, devkit_path=None):
        datasets.imdb.__init__(self, 'gitw_' + image_set)
        self._image_set = image_set
        """I follow pascal terminology, but in our case devkit and datafolders are the same"""
        self._devkit_path = self._get_default_path() if devkit_path is None \
                            else devkit_path
        self._data_path = self._devkit_path #os.path.join(self._devkit_path, 'DATA')
        self._frame_pattern = 'Frame_'
        self._frames_subfolder = 'Frames'
        self._aug_frames_subfolder = 'FramesAug'
        self._frame_step = 40
        
        
        """Read the classes from the corresponding file"""
        catFile=os.path.join(self._devkit_path,'categories.txt')
        fc = open(catFile, "r")
        categories=fc.read().splitlines()
        categories.insert(0,'background')
        fc.close()
        
        self._classes = tuple(categories)
        self._class_to_ind = dict(zip(self.classes, xrange(self.num_classes)))
        self._image_ext = '.jpg'
        self._video_index = self._load_video_set_index()
        # Default to roidb handler
        self._roidb_handler = self.fixation_roidb

        # PASCAL specific config options
        self.config = {'cleanup'  : True,
                       'use_salt' : True,
                       'top_k'    : 2000}

        assert os.path.exists(self._devkit_path), \
                'Grasp in the Wild path does not exist: {}'.format(self._devkit_path)
        assert os.path.exists(self._data_path), \
                'Data Path does not exist: {}'.format(self._data_path)

    def get_label(self, v):
        """
        Return the label number for the video.
        """
        vcat=os.path.dirname(self._video_index[v])
        label=self._class_to_ind[vcat]
        return label
    
    def video_path_at(self, i):
        """
        Return the absolute path to image i in the image sequence.
        """
        return self.video_path_from_index(self._video_index[i])

    def video_path_from_index(self, index):
        """
        Construct an image path from the image's "index" identifier.
        """
        video_path = os.path.join(self._data_path, 'images',
                                  index )
        #assert os.path.exists(video_path), \
        #        'Path does not exist: {}'.format(video_path)
        if os.path.exists(video_path)==0:
            print 'Path does not exist: %s'%video_path
        return video_path
    
    def image_path_at(self, v, i, nv):
        """
        Return the absolute path to image i in the image sequence.
        """
        return self.image_path_from_index(self._video_index[v], i, nv)
    
    def image_path_from_index(self, vindex, index, nvar):
        """
        Construct an image path from the image's "index" identifier.
        """
        if nvar == 0:
            image_path = os.path.join(self._data_path, 'images',
                                  vindex, self._frames_subfolder, '%s%d%s'%(self._frame_pattern,index*self._frame_step,self._image_ext) )
        else:
            image_path = os.path.join(self._data_path, 'images',
                                  vindex, self._aug_frames_subfolder, '%s%d_v%d%s'%(self._frame_pattern,index*self._frame_step,nvar,self._image_ext) )
                
                
#        assert os.path.exists(image_path), \
#                'Path does not exist: {}'.format(image_path)
        if os.path.exists(image_path)==0:
            print 'Path does not exist: %s'%image_path
        
        return image_path

    def _load_video_set_index(self):
        """
        Load the indexes listed in this dataset's image set file.
        """
        # Example path to image set file:
        # self._devkit_path + /VOCdevkit2007/VOC2007/ImageSets/Main/val.txt
        video_set_file = os.path.join(self._data_path, 'ImageSets', self._image_set + '.txt')
        assert os.path.exists(video_set_file ), \
                'Path does not exist: {}'.format(video_set_file )
        with open(video_set_file ) as f:
            video_index = [x[0:-3].strip() for x in f.readlines()]
        return video_index 

    def _get_default_path(self):
        """
        Return the default path where Grasp in the Wild is expected to be installed.
        """
        return os.path.join(datasets.ROOT_DIR, '..','..','Data','Datasets', 'GITW')


    def fixation_roidb(self):
        """
        Return the database of selective search regions of interest.
        Ground-truth ROIs are also included.

        This function loads/saves from/to a cache file to speed up future calls.
        """
        
        cache_file = os.path.join(self.cache_path,
                                  self.bbname + '_fixation_roidb.pkl')

        if os.path.exists(cache_file):
            with open(cache_file, 'rb') as fid:
                roidb = cPickle.load(fid)
            print '{} ss roidb loaded from {}'.format(self.bbname, cache_file)
            return roidb

        roidb = self._load_fixation_roidb(None)
        with open(cache_file, 'wb') as fid:
            cPickle.dump(roidb, fid, cPickle.HIGHEST_PROTOCOL)
        print 'wrote fixations roidb to {}'.format(cache_file)

        return roidb

    #We need to propose the rois from the fixations/or load them???
    #Better to load them
    def _load_fixation_roidb(self, gt_roidb):
        
        filename = os.path.abspath(os.path.join(self._data_path,
                                                'BBs',
                                                self.bbname + '.npy'))
        assert os.path.exists(filename), \
               'Bounding boxes not found at: {}'.format(filename)
        boxes=np.load(filename);
        
        return self.create_roidb_from_box_list(boxes, gt_roidb)

   
    

    def _write_voc_results_file(self, all_boxes):
        use_salt = self.config['use_salt']
        comp_id = 'comp4'
        if use_salt:
            comp_id += '-{}'.format(os.getpid())

        # VOCdevkit/results/VOC2007/Main/comp4-44503_det_test_aeroplane.txt
        path = os.path.join(self._devkit_path, 'results', 'VOC' + self._year,
                            'Main', comp_id + '_')
        for cls_ind, cls in enumerate(self.classes):
            if cls == '__background__':
                continue
            print 'Writing {} VOC results file'.format(cls)
            filename = path + 'det_' + self._image_set + '_' + cls + '.txt'
            with open(filename, 'wt') as f:
                for im_ind, index in enumerate(self.image_index):
                    dets = all_boxes[cls_ind][im_ind]
                    if dets == []:
                        continue
                    # the VOCdevkit expects 1-based indices
                    for k in xrange(dets.shape[0]):
                        f.write('{:s} {:.3f} {:.1f} {:.1f} {:.1f} {:.1f}\n'.
                                format(index, dets[k, -1],
                                       dets[k, 0] + 1, dets[k, 1] + 1,
                                       dets[k, 2] + 1, dets[k, 3] + 1))
        return comp_id

    def _do_matlab_eval(self, comp_id, output_dir='output'):
        rm_results = self.config['cleanup']

        path = os.path.join(os.path.dirname(__file__),
                            'VOCdevkit-matlab-wrapper')
        cmd = 'cd {} && '.format(path)
        cmd += '{:s} -nodisplay -nodesktop '.format(datasets.MATLAB)
        cmd += '-r "dbstop if error; '
        cmd += 'voc_eval(\'{:s}\',\'{:s}\',\'{:s}\',\'{:s}\',{:d}); quit;"' \
               .format(self._devkit_path, comp_id,
                       self._image_set, output_dir, int(rm_results))
        print('Running:\n{}'.format(cmd))
        status = subprocess.call(cmd, shell=True)

    def evaluate_detections(self, all_boxes, output_dir):
        comp_id = self._write_voc_results_file(all_boxes)
        self._do_matlab_eval(comp_id, output_dir)

    def competition_mode(self, on):
        if on:
            self.config['use_salt'] = False
            self.config['cleanup'] = False
        else:
            self.config['use_salt'] = True
            self.config['cleanup'] = True
    
    def set_bb_db(self, bbname):
        self.bbname = bbname
    
    
if __name__ == '__main__':
    d = datasets.grasping_in_the_wild('train')
    res = d.roidb
    from IPython import embed; embed()
