#include <safeQueue.hpp>
#include <depth.hpp>
#include <traitement.hpp>
#include <calib.hpp>
#include <timer.hpp>

#include <opencv2/calib3d/calib3d.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <string>
#include <iostream>
#include <thread>
#include <unistd.h>

using namespace std;
using namespace cv;

bool GetUndistortFrames(SafeQueue<Mat>* queue_in,SafeQueue<Mat>* queue_out,unsigned short int flag,atomic_bool* end,StereoCalibration* calib)
{
  //thread::id this_id = this_thread::get_id();
  unsigned int i=0;
  Mat src,rectify;
  do
  {
    src=queue_in->dequeue();
    if(src.empty())
      break;
    if(calib != NULL)
    {
	calib->undistortion(src,&rectify,flag);
	queue_out->enqueue(rectify);
    }
    else
      queue_out->enqueue(src);
  }
  while(42);
  *end=true;
}

void depthmap(Mat& disp,Mat* dest,Mat& reprojectionMat,FILE* file)
{
  reprojectImageTo3D(disp,*dest, reprojectionMat,false);
  // if really high values -> incorrect value
  const double max = 1.0e4;
  for(int y = 0; y < dest->rows; y++)
    {
      for(int x = 0; x < dest->cols; x++)
        {
            Vec3f point = dest->at<Vec3f>(y, x);
	    // incorrect values (black on depth map)
            if(fabs(point[2]) > max) 
	      continue;
            fprintf(file,"%f-",point[2]);
        }
      fprintf(file,"endl\n");
    }
  fprintf(file,"\n");
}

void StereoMap(VideoCapture& vc_left,VideoCapture& vc_right,StereoCalibration* calib=NULL)
{
  char wait;
  // STEP 1 - INIT
  SafeQueue<Mat> q_in_left(DEFAULT_QUEUE_SIZE);
  SafeQueue<Mat> q_in_right(DEFAULT_QUEUE_SIZE);
  SafeQueue<Mat> q_undist_left(DEFAULT_QUEUE_SIZE);
  SafeQueue<Mat> q_undist_right(DEFAULT_QUEUE_SIZE);
  Timer time = Timer();
  StereoMatch bm = StereoMatch(STEREOBM);
  Mat depth;
  Mat left,right;
  for(int i=0;i<DEFAULT_QUEUE_SIZE;++i)
  {
    vc_left >> left;
    vc_right >> right;
    if(left.empty()||right.empty())
      exit(EXIT_FAILURE);
    else
    {
      q_in_left.enqueue(left);
      q_in_right.enqueue(right);
    }
  }
  FILE* file = fopen("output_depth.txt", "wt");
  // STEP 2 - THREAD 
  //c++11 thread safe bool
  atomic_bool end(false);
  
  thread t_left(GetUndistortFrames,
		&q_in_left,
		&q_undist_left,
		LEFT,
		&end,
		calib);

  thread t_right(GetUndistortFrames,
		 &q_in_right,
		 &q_undist_right,
		 RIGHT,
		 &end,
		 calib);
  
  Mat output,reprojectionMat;
  if(calib!=NULL)
    reprojectionMat=calib->getReprojectionMat();
  if(reprojectionMat.empty())
  {
    cerr << "No reprojection mat ... -> no depth map !\n";
  }
  //STEP 3 - TRAITMENT
  /* while there's some frame in both queue and threads aren't finished */  
  while(!end)
  {
    vc_left>>left;
    vc_right>>right;
    if(left.empty()||right.empty())
      exit(EXIT_FAILURE);
    else
    {
      q_in_left.enqueue(left);
      q_in_right.enqueue(right);
    }
    time.start();
    bm.compute(q_undist_left.dequeue(),q_undist_right.dequeue(),&output);
    time.end();
    //Faire un display mod pour ça a la limite
    putText(output, time.fastTxt(), Point(5, 25), FONT_HERSHEY_SIMPLEX, 1.0, Scalar::all(255));
    imshow("Disparity Map",output);
    wait = (char) waitKey(1);
    if(wait == ' ' && calib!=NULL)
    {
      // TODO -> Return by side effect on &depth for Poppy ?
      depthmap(output,&depth,reprojectionMat,file);
      cout << "wrote !\n";
    }
  } 
  t_left.join();
  t_right.join();
}

void usage()
{
  cout << "Usage : ./Depthmap -l <Left_input> -r <Right_input> -c <Stereo_calibration_file>\n"
       << "Output : output_depth.txt | Press space to save depthmap on output_depth.txt\n";
}

int main(int argc,char** argv)
{
  if(argc < 5)
  {
    //usage();
    return(EXIT_FAILURE);
  }
  string left,right,calibration;
  //Parsing
  for(int i = 1; i<argc; i++)
  {
    const char* s = argv[i];
    if(!strcmp(s,"-l"))
    {
      left=argv[++i];
      cout << "Left video file : " << left << endl;
    }
    else if(!strcmp(s,"-r"))
    {
      right=argv[++i];
      cout << "Right video file : " << right << endl;
    }
    else if(!strcmp(s,"-c"))
    {
      calibration=argv[++i];
      cout << "Calibration file : " << calibration << endl;
    }
  }
  if(left.empty()||right.empty())
  {
    cout << "Right or Left video file empty !\n";
    usage();
    exit(EXIT_FAILURE);
  }
  if(calibration.empty())
  {
    cout << "No input calibration file ! (no epipolar constraint) ...\n";
  }
  //open stream
  VideoCapture vc_left(left); 
  VideoCapture vc_right(right);
  if (!vc_left.isOpened() || !vc_right.isOpened()) 
    throw runtime_error("Can't open these files\n");
  StereoCalibration calib;
  if(!calibration.empty())
  {
    try
    {
      calib = StereoCalibration(calibration);
    }
    catch(exception e)
    {
      e.what();
    }
    cout << "Calibration file : " << calibration << " loaded !" << endl;
    StereoMap(vc_left,vc_right,&calib);
  }
  else
    StereoMap(vc_left,vc_right);
  exit(0);
}
  
  
