#include <opencv2/calib3d/calib3d.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <stdio.h>
#include <vector>
#include <string>
#include <iostream>
#include <algorithm>
#include <stdexcept>
#include <calib.hpp>

using namespace std;
using namespace cv;

/***************************************************************************************
 *                       StereoCalibration Methods                                     *
 ***************************************************************************************/

static bool displayImages(Mat&,Mat&,unsigned int); 

static inline FileStorage openFs(string& fileName,int flags)
{
  FileStorage fs;
  fs.open(fileName,flags);
  if (!fs.isOpened())
    throw runtime_error("file name for camera calibration incorrect\n");
  return fs;
}
  
StereoCalibration::StereoCalibration(string& StereoCalibration)
{
  FileStorage fs;
  try
  {
    fs = openFs(StereoCalibration,FileStorage::READ);
  }
  catch(exception e)
  {
    cerr << e.what() << endl;
  }
  cout << "Loading stereo camera file..." << endl;
  fs["image_width"] >> _imgSize.width;
  fs["image_height"] >> _imgSize.height;
  fs["square_size"] >> _squareSize;
  fs["camera_matrix_left"] >> _cameraMatrix[0];
  fs["camera_matrix_right"] >> _cameraMatrix[1];
  fs["distortion_coefficients_left"] >> _distCoeffs[0];
  fs["distortion_coefficients_right"] >> _distCoeffs[1];
  fs["R"] >> _R;
  fs["T"] >> _T;
  fs["E"] >> _E;
  fs["F"] >> _F;
  this->initialize();
}

//Serialize
void StereoCalibration::save(string fileName)
{
  cout << "Saving ..."<<endl;
  FileStorage fs;
  try
  {
    fs = openFs(fileName,FileStorage::WRITE);
  }
  catch(exception e)
  {
    cerr << e.what() << endl;
  }
  cout << "Saving stereo camera file..." << endl;
  fs << "image_width" << _imgSize.width;
  fs <<"image_height" << _imgSize.height;
  fs <<"square_size" << _squareSize;
  fs <<"camera_matrix_left" << _cameraMatrix[0];
  fs <<"camera_matrix_right" << _cameraMatrix[1];
  fs <<"distortion_coefficients_left" << _distCoeffs[0];
  fs <<"distortion_coefficients_right" << _distCoeffs[1];
  fs <<"R" << _R;
  fs <<"T" << _T;
  fs <<"E" << _E;
  fs <<"F" << _F;
}
  
StereoCalibration::StereoCalibration(string& calibrationFileL,string& calibrationFileR)
{
  //compare return 0 if equal 
  bool _sameCamera=!(calibrationFileL.compare(calibrationFileR));
  FileStorage fsRight,fsLeft;
  try
  {
    fsLeft = openFs(calibrationFileL,FileStorage::READ);
    if(!_sameCamera)
      fsRight = openFs(calibrationFileR,FileStorage::READ);
    else
      fsRight = openFs(calibrationFileR,FileStorage::READ);
  }
  catch(exception e)
  {
    cerr << e.what() << endl;
  }
  cout << "Loading single camera file - " << _sameCamera ? "2 same camera...":"2 different camera..."; 
  cout << endl;
  fsLeft["image_width"] >> _imgSize.width;
  fsLeft["image_height"] >> _imgSize.height;
  fsLeft["square_size"] >> _squareSize;
  fsLeft["camera_matrix"] >> _cameraMatrix[0];
  fsRight["camera_matrix"] >> _cameraMatrix[1];
  fsLeft["distortion_coefficients"] >> _distCoeffs[0];
  fsRight["distortion_coefficients"] >> _distCoeffs[1];
}

void StereoCalibration::undistortion(Mat & src,Mat* dest,unsigned short int flag)
{
  // If flag != LEFT or RIGHT
  if(flag != 1 && flag != 2)
    throw runtime_error("Invalid flag : LEFT or RIGHT\n");
  undistort(src,*dest,_cameraMatrix[flag-1], _distCoeffs[flag-1], _cameraMatrix[flag-1]);
}

inline bool displayImages(Mat& left,Mat& right,unsigned int delay=100000) 
{
    imshow("Left Image",left);
    imshow("Right Image",right);
    char key = (char) waitKey(delay);
    switch(key)
    {
    case ' ':
      return true;
    default:
      return false;
    }
}

/* seek chessboard & fill corners arrays */
bool StereoCalibration::findChessboardCornersAndDraw(Mat& mat_left, Mat& mat_right) 
{
  
  /* -> 8 bits for draw */
  cvtColor(mat_left, mat_left, CV_RGB2GRAY);
  cvtColor(mat_right, mat_right, CV_RGB2GRAY); 

  bool found_left = findChessboardCorners(mat_left, _boardSize, _corners[0], CALIB_CB_ADAPTIVE_THRESH + CALIB_CB_NORMALIZE_IMAGE + CV_CALIB_CB_FILTER_QUADS+CALIB_CB_FAST_CHECK);
  bool found_right = findChessboardCorners(mat_right,_boardSize, _corners[1],CALIB_CB_ADAPTIVE_THRESH + CALIB_CB_NORMALIZE_IMAGE + CV_CALIB_CB_FILTER_QUADS+CALIB_CB_FAST_CHECK);
  
  drawChessboardCorners(mat_left, _boardSize, _corners[0], found_left);
  drawChessboardCorners(mat_right, _boardSize, _corners[1], found_right);
  
  if (found_left && found_right) 
    return displayImages(mat_left,mat_right);
  else 
    return false;
}

void StereoCalibration::initialize()
{
  cout << "Initialize ...\n";
  stereoRectify(_cameraMatrix[0], _distCoeffs[0],_cameraMatrix[1], _distCoeffs[1],
		  _imgSize, _R, _T, 
		  _rotation[0], _rotation[1], _projection[0], _projection[1], _disparity,
		  CALIB_ZERO_DISPARITY, 1, _imgSize);


  initUndistortRectifyMap(_cameraMatrix[0],_distCoeffs[0], _rotation[0], _projection[0],_imgSize, CV_16SC2, _mapx[0],_mapy[0]);
  initUndistortRectifyMap(_cameraMatrix[1],_distCoeffs[1], _rotation[1], _projection[1],_imgSize, CV_16SC2, _mapx[1],_mapy[1]);
}
  

void StereoCalibration::calibrateFromVideo(string& vid_left, string& vid_right,unsigned int nbFrames) 
{
  VideoCapture vc_left(vid_left); 
  VideoCapture vc_right(vid_right);
  /* open test*/
  if (!vc_left.isOpened() || !vc_right.isOpened()) 
    throw runtime_error("Can't open these files\n");

  Mat input_left, input_right;
  bool found_corners,ok_l,ok_r;
  bool size=false;
  _nbFrames=nbFrames;
  cout << "Press spacebar to save a frame for calibration" << endl;
  while(nbFrames > 0)
  {
    ok_l=vc_left.read(input_left);
    ok_r=vc_right.read(input_right);
    if(!size)
    {
      _imgSize = input_left.size();
      size=true;
    }
    if(!ok_r||!ok_l||input_left.empty()||input_right.empty())
      throw runtime_error("Not enouth pattern found or saved\n");
    found_corners = findChessboardCornersAndDraw(input_left, input_right);
    if (found_corners) 
    {
      /* save detection */
      _imagesPoints[0].push_back(_corners[0]);
      _imagesPoints[1].push_back(_corners[1]);
      --nbFrames;
      cout << "Found corners - " << nbFrames <<endl;
    }
  }
  vc_left.release();
  vc_right.release();
  _patternFound=true;
  
}

void StereoCalibration::stereoCalib()
  {
    /* objectPoints should contain physical location of each corners but since we don’t know 
     that so we assign constant positions to all the corners and assume that camera is moving.*/
    if(!_patternFound)
    {
      cerr << "A call to calibrateFromVideo is necessary before this." <<endl;
      return;
    }
    cout << "Starting stereo calibration ..."<<endl;
    _objectPoints.resize(_nbFrames);
    for (int i=0; i<_nbFrames; i++) 
      for (int j=0; j<_boardSize.height; j++) 
	  for (int k=0; k<_boardSize.width; k++) 
	      _objectPoints[i].push_back(Point3f(k*_squareSize,j*_squareSize,0.0));
    /* Camera matrix - focal lengths & principal point*/
    if(_cameraMatrix[0].empty()||_cameraMatrix[1].empty())
    {
      _cameraMatrix[0] = Mat::eye(3, 3, CV_64F);
      _cameraMatrix[1] = Mat::eye(3, 3, CV_64F);
    }
    
    cout << "objectPoints.size() : " << _objectPoints.size() << endl;
    cout << "imagesPoints[0].size() : " << _imagesPoints[0].size() << endl;
    cout << "imagesPoints[1].size() : " << _imagesPoints[1].size() << endl;
    cout << "imgSize : " << _imgSize << endl;  

    double rms = stereoCalibrate(_objectPoints, 
				 _imagesPoints[0],
				 _imagesPoints[1],
                                 _cameraMatrix[0], 
				 _distCoeffs[0],
                                 _cameraMatrix[1], 
				 _distCoeffs[1],
                                 _imgSize, 
				 _R, _T, _E, _F,
				 CALIB_FIX_K3 + CALIB_RATIONAL_MODEL + _flags,
				 TermCriteria(TermCriteria::COUNT+TermCriteria::EPS, 100, 1e-5) );
    //CALIB_FIX_ASPECT_RATIO + CALIB_FIX_K3 + CALIB_FIX_K4 + CALIB_FIX_K5 + CALIB_RATIONAL_MODEL

    cout << rms << endl;
    cout << "R " << _R << endl 
	 << "T " << _T << endl 
	 << "E " << _E << endl 
	 << "F " << _F << endl
	 << "cam_mat_l " << _cameraMatrix[0] << endl 
	 << "cam_mat_r " << _cameraMatrix[1] << endl 
	 << "dist_l " << _distCoeffs[0] << endl 
	 << "dist_r " << _distCoeffs[1] << endl;
    
    if(rms>1)
    {
      cout << "-!- BE CAREFULL -> RMS is high (bad calibration) -!-" << endl;
      cout << "Done Calibration"<<endl;
    }
    this->save();
    this->initialize();
}

void StereoCalibration::setWidth(string width)
{
  if(sscanf(width.c_str(),"%u",&_boardSize.width ) != 1 || _boardSize.width <= 0)
  {
    cerr << "Invalid board width\n";
    exit(EXIT_FAILURE);
  }
  cout << "Board Width -> "<< _boardSize.width << endl;
}

void StereoCalibration::setHeight(string height)
{
  //return 1 if ok
  if(sscanf(height.c_str(),"%u",&_boardSize.height ) != 1 || _boardSize.height <= 0)
  {
    cerr << "Invalid board height\n";
    exit(EXIT_FAILURE);
  }
  cout << "Board Height -> "<< _boardSize.height << endl;
}
 
void StereoCalibration::setSquareSize(string squareSize)
{
  //return 1 if ok
  if(sscanf(squareSize.c_str(),"%u",&_squareSize ) != 1 || _squareSize <= 0)
  {
    cerr << "Invalid square size\n";
    exit(EXIT_FAILURE);
  }
}

void StereoCalibration::addFlag(int flag)
{
  _flags |= flag;
}
