#ifndef SAFE_QUEUE
#define SAFE_QUEUE

#include <queue>
#include <mutex>
#include <condition_variable>

using namespace std;

/* C++11 thread safe queue template*/
template <class T> class SafeQueue
{
public:
  // 0 : unlimited 
  SafeQueue(unsigned int max=0): m_queue(), 
				 m_mutex(),
				 m_cv(),
				 m_max(max)
				 
  {}

  bool enqueue(const T &t);
  T dequeue();
  bool empty()
  {
    lock_guard<mutex> lock(m_mutex);
    return m_queue.empty();
  }
  int size()
  {
    lock_guard<mutex> lock(m_mutex);
    return m_queue.size();
  }

private:
  queue<T> m_queue;
  mutex m_mutex;
  condition_variable m_cv;
  unsigned int m_max;
};

template <class T>
bool SafeQueue<T>::enqueue(const T &t)
{
  /* Lock thread*/
  lock_guard<mutex> lock(m_mutex);
  if (m_max != 0 && m_queue.size() >= m_max)
    /*queue is full*/
    return false;
  m_queue.emplace(t);
  /* ok - another thread can dequeue*/
  //cout << m_queue.size() << "/" << m_max << endl;
  m_cv.notify_one();
  return true;
}

template <class T>
T SafeQueue<T>::dequeue()
{
  /* Lock thread */
  unique_lock<mutex> lock(m_mutex);
  
  /* Our condition to dequeue -> m_queue not empty*/
  m_cv.wait(lock,[this](){return !m_queue.empty();});
  /*Get next elem*/
  T val = m_queue.front();
  m_queue.pop();
  return val;
}



#endif
