#define _POSIX_C_SOURCE 1
#include <unistd.h>
#include <termios.h>
#include <stdlib.h>
#include <stdio.h>

#include "terminal.h"

static struct termios terminal_sauvegarde;
static int terminal_descripteur = -1;

static void terminal_lire(struct termios *tampon);
static void terminal_ecrire(struct termios *tampon);

void
terminal_initialiser(int fd)
{
    if (!isatty(fd))
    {
	if (ttyname(fd) != NULL)
	    fprintf(stderr, "(%s) ", ttyname(fd));
	fprintf(stderr, "terminal_initialiser :"
		"le descripteur n'est pas un terminal\n");
	exit(EXIT_FAILURE);
    }
    terminal_descripteur = fd;
    terminal_lire(&terminal_sauvegarde);
    atexit(terminal_restaurer);
}

void
terminal_restaurer(void)
{
    if (terminal_descripteur != -1)
	terminal_ecrire(&terminal_sauvegarde);
}

void
terminal_lecture_par_caractere(int mode)
{
    struct termios tampon;
    static int eof = -1;
    static int eol = -1;

    terminal_lire(&tampon);
    if (mode == 1)
    {
	eof = tampon.c_cc[VEOF];
	eol = tampon.c_cc[VEOL];
	tampon.c_lflag &= ~ICANON;
	tampon.c_cc[VMIN] = 1;
	tampon.c_cc[VTIME] = 0;
    }
    else
    {
	if (eof != -1)
	{
	    tampon.c_cc[VEOF] = eof;
	    tampon.c_cc[VEOL] = eol;
	}
	tampon.c_lflag |= ICANON;
    }
    terminal_ecrire(&tampon);
}

#define TERMINAL_MODIFIER(CHAMP, DRAPEAU) \
struct termios tampon;        \
                              \
terminal_lire(&tampon);       \
if (mode == 1)                \
    tampon.CHAMP |= DRAPEAU;  \
else                          \
    tampon.CHAMP &= ~DRAPEAU; \
terminal_ecrire(&tampon);

void
terminal_echo(int mode)
{
    TERMINAL_MODIFIER(c_lflag, ECHO);
}

void
terminal_signaux(int mode)
{
    TERMINAL_MODIFIER(c_lflag, ISIG);
}

void 
terminal_controle_de_flux(int mode)
{
    TERMINAL_MODIFIER(c_iflag, (IXOFF|IXON));
}

void
terminal_post_traitement(int mode)
{
    TERMINAL_MODIFIER(c_oflag, OPOST);
}

int terminal_erase()
{
    struct termios tampon;

    terminal_lire(&tampon);
    return tampon.c_cc[VERASE];
}

int terminal_kill()
{
    struct termios tampon;

    terminal_lire(&tampon);
    return tampon.c_cc[VKILL];
}

static void
terminal_lire(struct termios *tampon)
{    
    if (tcdrain(terminal_descripteur) == -1 
	|| tcgetattr(terminal_descripteur, tampon) == -1)
    {
	perror(NULL);
	exit(EXIT_FAILURE);
    }
}

static void
terminal_ecrire(struct termios *tampon)
{
    if (tcsetattr(terminal_descripteur, TCSADRAIN, tampon) == -1)
    {
	perror(NULL);
	exit(EXIT_FAILURE);
    }
}
